﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/ens_ApiForAcbaa.h>
#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/core/ens_ServiceThread.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskActivateNotificationService.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskDeleteMessage.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskDeleteMyDesign.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskGetMyDesignBody.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskGetMyDesignHeaderList.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskPostMyDesign.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskReceiveMessageBody.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskReceiveMessageHeaderList.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskRegisterMyDesignAuthorProfile.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskSendMessage.h>
#include <nn/ens/detail/app/acbaa/task/ens_TaskSendMessageToNintendo.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_NpnsLibrary.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>
#include <nn/npns.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_Utf8StringUtil.h>

namespace nn { namespace ens {

void SendMessage(AsyncContext* pOutContext,
    const UserId& targetUserId,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_EQUAL(targetUserId, InvalidUserId);
    NN_SDK_REQUIRES(metadata.IsValid() && !metadata.IsNull());
    NN_SDK_REQUIRES(body.IsValid() && !body.IsNull());
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskSendMessage);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(targetUserId, metadata, body);

    detail::core::RegisterTask(pTask, pOutContext);
}

void SendMessageToNintendo(AsyncContext* pOutContext,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES(metadata.IsValid() && !metadata.IsNull());
    NN_SDK_REQUIRES(body.IsValid() && !body.IsNull());
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskSendMessageToNintendo);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(metadata, body);

    detail::core::RegisterTask(pTask, pOutContext);
}

void ReceiveMessageHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MessageHeader pOutHeaderList[],
    int count,
    int offset,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutTotalCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderList);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    for (int i = 0; i < count; i++)
    {
        // メタデータを使用しないのであれば、空でもよい。
        NN_SDK_REQUIRES(pOutHeaderList[i].metadata.IsValid());
    }

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskReceiveMessageHeaderList);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutCount, pOutTotalCount, pOutHeaderList, count, offset);

    detail::core::RegisterTask(pTask, pOutContext);
}

void ReceiveMessageBody(AsyncContext* pOutContext,
    ReceiveBuffer* pOutBody,
    const MessageId& messageId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutBody);
    NN_SDK_REQUIRES(pOutBody->IsValid() && !pOutBody->IsNull());
    NN_SDK_REQUIRES(messageId != InvalidMessageId);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskReceiveMessageBody);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutBody, messageId);

    detail::core::RegisterTask(pTask, pOutContext);
}

void DeleteMessage(AsyncContext* pOutContext,
    const MessageId pMessageIdList[],
    int count,
    const char* pReason,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pMessageIdList);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_NOT_NULL(pReason);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)

    size_t length = static_cast<size_t>(nn::util::Strnlen(pReason, INT_MAX));

    if (length > 0)
    {
        NN_SDK_REQUIRES(nn::util::VerifyUtf8String(pReason, length));
    }

#endif

    for (int i = 0; i < count; i++)
    {
        NN_SDK_REQUIRES_NOT_EQUAL(pMessageIdList[i], InvalidMessageId);
    }

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskDeleteMessage);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pMessageIdList, count, pReason);

    detail::core::RegisterTask(pTask, pOutContext);
}

void ActivateNotificationService(AsyncContext* pOutContext,
    const nn::account::Uid& user,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskActivateNotificationService);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(user);

    detail::core::RegisterTask(pTask, pOutContext);
}

nn::os::SystemEvent& GetNotificationEvent() NN_NOEXCEPT
{
    detail::util::NpnsLibrary::Initialize();

    return detail::util::NpnsLibrary::GetInstance().GetReceiveEvent();
}

bool PopNotificationData(NotificationData* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    detail::util::NpnsLibrary::Initialize();

    while (NN_STATIC_CONDITION(true))
    {
        nn::npns::NotificationData data;
        nn::Result result = nn::npns::Receive(&data);

        if (nn::npns::ResultNotReceived::Includes(result))
        {
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::Bit8 decodedPayload[(nn::npns::PayloadMaxLength / 4) * 3] = {};
        size_t decodedSize = 0;

        nn::util::Base64::Status status = nn::util::Base64::FromBase64String(&decodedSize, decodedPayload, sizeof (decodedPayload),
            data.GetPayload(), data.GetPayloadSize(), nn::util::Base64::Mode_NormalNoLinefeed);

        if (status != nn::util::Base64::Status_Success)
        {
            NN_DETAIL_ENS_INFO("[ens] The notification data was not base64 format. status = %d\n", status);
            continue;
        }

        NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, decodedPayload, decodedSize);

        detail::util::ResponseStructureReader<2> reader;

        reader.Add("$.type",
            pOut->type, sizeof (pOut->type));
        reader.Add("$.receiver_user_id",
            &pOut->receiver.value);

        if (!reader.Read<detail::util::MessagePackReader>(decodedPayload, decodedSize))
        {
            NN_DETAIL_ENS_INFO("[ens] The decoded notification data was not well-known format.\n");
            continue;
        }

        std::memcpy(pOut->payload, decodedPayload, decodedSize);
        pOut->payloadSize = decodedSize;

        break;
    }

    return true;

#else

    return false;

#endif
}

void RegisterMyDesignAuthorProfile(AsyncContext* pOutContext,
    MyDesignAuthorId* pOutId,
    const MyDesignAuthorName& name,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutId);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)

    size_t length = static_cast<size_t>(nn::util::Strnlen(name.value, sizeof (name.value)));

    NN_SDK_REQUIRES_LESS(length, sizeof (name.value));

    if (length > 0)
    {
        NN_SDK_REQUIRES(nn::util::VerifyUtf8String(name.value, length));
    }

#endif

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskRegisterMyDesignAuthorProfile);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutId, name);

    detail::core::RegisterTask(pTask, pOutContext);
}

void PostMyDesign(AsyncContext* pOutContext,
    MyDesignId* pOutId,
    const SendBuffer& metadata,
    const SendBuffer& body,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutId);
    NN_SDK_REQUIRES(metadata.IsValid() && !metadata.IsNull());
    NN_SDK_REQUIRES(body.IsValid() && !body.IsNull());
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskPostMyDesign);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutId, metadata, body);

    detail::core::RegisterTask(pTask, pOutContext);
}

void DeleteMyDesign(AsyncContext* pOutContext,
    const MyDesignId& myDesignId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_EQUAL(myDesignId, InvalidMyDesignId);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskDeleteMyDesign);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(myDesignId);

    detail::core::RegisterTask(pTask, pOutContext);
}

void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const MyDesignAuthorId& authorId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutTotalCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderList);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_EQUAL(authorId, InvalidMyDesignAuthorId);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    for (int i = 0; i < count; i++)
    {
        // メタデータを使用しないのであれば、空でもよい。
        NN_SDK_REQUIRES(pOutHeaderList[i].metadata.IsValid());
    }

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskGetMyDesignHeaderList);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutCount, pOutTotalCount, pOutHeaderList, count, offset, authorId);

    detail::core::RegisterTask(pTask, pOutContext);
}

void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const char* pSearchQuery,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutTotalCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderList);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(pSearchQuery);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    for (int i = 0; i < count; i++)
    {
        // メタデータを使用しないのであれば、空でもよい。
        NN_SDK_REQUIRES(pOutHeaderList[i].metadata.IsValid());
    }

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskGetMyDesignHeaderList);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutCount, pOutTotalCount, pOutHeaderList, count, offset, pSearchQuery);

    detail::core::RegisterTask(pTask, pOutContext);
}

void GetMyDesignHeaderList(AsyncContext* pOutContext,
    int* pOutCount,
    int* pOutTotalCount,
    MyDesignHeader pOutHeaderList[],
    int count,
    int offset,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutTotalCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderList);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    for (int i = 0; i < count; i++)
    {
        // メタデータを使用しないのであれば、空でもよい。
        NN_SDK_REQUIRES(pOutHeaderList[i].metadata.IsValid());
    }

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskGetMyDesignHeaderList);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutCount, pOutTotalCount, pOutHeaderList, count, offset);

    detail::core::RegisterTask(pTask, pOutContext);
}

void GetMyDesignBody(AsyncContext* pOutContext,
    ReceiveBuffer* pOutBody,
    const MyDesignId& myDesignId,
    const Credential& credential, const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutBody);
    NN_SDK_REQUIRES(pOutBody->IsValid() && !pOutBody->IsNull());
    NN_SDK_REQUIRES_NOT_EQUAL(myDesignId, InvalidMyDesignId);
    NN_SDK_REQUIRES(credential.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, detail::app_acbaa::task::TaskGetMyDesignBody);

    pTask->SetCredential(credential);
    pTask->SetNetworkServiceAccountIdToken(pToken);

    pTask->SetParameter(pOutBody, myDesignId);

    detail::core::RegisterTask(pTask, pOutContext);
}

}}
