﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace util {

ResponseStructureReaderImpl::ResponseStructureReaderImpl(Structure* pList, size_t num) NN_NOEXCEPT
    : m_pList(pList)
    , m_Num(num)
    , m_Count(0)
{
    NN_SDK_REQUIRES_NOT_NULL(pList);
    NN_SDK_REQUIRES_GREATER(num, 0u);
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, bool* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_Boolean, pJsonPath, nullptr, pOutValue, sizeof (bool));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, int8_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_S8, pJsonPath, nullptr, pOutValue, sizeof (int8_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, int16_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_S16, pJsonPath, nullptr, pOutValue, sizeof (int16_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, int32_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_S32, pJsonPath, nullptr, pOutValue, sizeof (int32_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, int64_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_S64, pJsonPath, nullptr, pOutValue, sizeof (int64_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, uint8_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_U8, pJsonPath, nullptr, pOutValue, sizeof (uint8_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, uint16_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_U16, pJsonPath, nullptr, pOutValue, sizeof (uint16_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, uint32_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_U32, pJsonPath, nullptr, pOutValue, sizeof (uint32_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, uint64_t* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_U64, pJsonPath, nullptr, pOutValue, sizeof (uint64_t));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, uint64_t* pOutValue, int8_t radix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(radix, 2, 36);

    AddImpl(ValueType_U64, pJsonPath, nullptr, pOutValue, sizeof (uint64_t), radix);
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, double* pOutValue) NN_NOEXCEPT
{
    AddImpl(ValueType_Double, pJsonPath, nullptr, pOutValue, sizeof (double));
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, char* pOutValue, size_t size) NN_NOEXCEPT
{
    AddImpl(ValueType_String, pJsonPath, nullptr, pOutValue, size);
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, size_t* pOutSize, void* pOutValue, size_t size) NN_NOEXCEPT
{
    AddImpl(ValueType_Binary, pJsonPath, pOutSize, pOutValue, size);
}

void ResponseStructureReaderImpl::Add(const char* pJsonPath, ReceiveBuffer* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    AddImpl(ValueType_Binary, pJsonPath, &pOutValue->receivedSize, pOutValue->pBuffer, pOutValue->bufferSize);
}

void ResponseStructureReaderImpl::AddImpl(ValueType valueType, const char* pJsonPath,
    size_t* pOutSize, void* pOutValue, size_t size, int8_t radix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pJsonPath);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    NN_SDK_ASSERT_LESS(static_cast<size_t>(m_Count), m_Num);
    NN_UNUSED(m_Num);

    m_pList[m_Count].pJsonPath = pJsonPath;
    m_pList[m_Count].valueType = valueType;
    m_pList[m_Count].pOutSize = pOutSize;
    m_pList[m_Count].pOutValue = pOutValue;
    m_pList[m_Count].size = size;
    m_pList[m_Count].radix = radix;
    m_pList[m_Count].isFound = false;

    m_Count++;
}

bool ResponseStructureReaderImpl::ResponseReadCallback(const JsonPath& jsonPath, const DataHolder& holder, void* pParam) NN_NOEXCEPT
{
    ResponseStructureReaderImpl* pThis = reinterpret_cast<ResponseStructureReaderImpl*>(pParam);

    for (int i = 0; i < pThis->m_Count; i++)
    {
        Structure& s = pThis->m_pList[i];

        if (s.isFound)
        {
            continue;
        }
        if (!jsonPath.Compare(s.pJsonPath))
        {
            continue;
        }

        switch (s.valueType)
        {
        case ValueType_Boolean:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<bool*>(s.pOutValue)));
            }
            break;
        case ValueType_S8:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<int8_t*>(s.pOutValue)));
            }
            break;
        case ValueType_S16:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<int16_t*>(s.pOutValue)));
            }
            break;
        case ValueType_S32:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<int32_t*>(s.pOutValue)));
            }
            break;
        case ValueType_S64:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<int64_t*>(s.pOutValue)));
            }
            break;
        case ValueType_U8:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<uint8_t*>(s.pOutValue)));
            }
            break;
        case ValueType_U16:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<uint16_t*>(s.pOutValue)));
            }
            break;
        case ValueType_U32:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<uint32_t*>(s.pOutValue)));
            }
            break;
        case ValueType_U64:
            {
                if (s.radix == 0)
                {
                    NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<uint64_t*>(s.pOutValue)));
                }
                else
                {
                    char v[65] = {};
                    NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(v, sizeof (v)));

                    errno = 0;

                    char* pEnd = nullptr;
                    uint64_t value = std::strtoull(v, &pEnd, s.radix);

                    if (pEnd != &v[holder.string.length])
                    {
                        return false;
                    }
                    if (errno == ERANGE)
                    {
                        return false;
                    }

                    *static_cast<uint64_t*>(s.pOutValue) = value;
                }
            }
            break;
        case ValueType_Double:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<double*>(s.pOutValue)));
            }
            break;
        case ValueType_String:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(static_cast<char*>(s.pOutValue), s.size));
            }
            break;
        case ValueType_Binary:
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(s.pOutSize, s.pOutValue, s.size));
            }
            break;
        default:
            return false;
        }

        s.isFound = true;
        break;
    }

    return true;
} // NOLINT(impl/function_size)

}}}}
