﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_MessagePackWriter.h>

namespace nn { namespace ens { namespace detail { namespace util {

namespace
{
    // 値を書き込む。
    inline void WriteByte(MessagePackWriter::Stream& stream, nn::Bit8 value) NN_NOEXCEPT
    {
        stream.pBuffer[0] = value;
        stream++;
    }

    // 数値を書き込む。
    template <typename T>
    void WriteNumeric(MessagePackWriter::Stream& stream, T value) NN_NOEXCEPT
    {
        T out;
        nn::util::StoreBigEndian(&out, value);

        std::memcpy(stream.pBuffer, &out, sizeof (T));

        stream += sizeof (T);
    }
}

MessagePackWriter::MessagePackWriter() NN_NOEXCEPT
    : m_pBuffer(nullptr)
    , m_Size(0)
{
    Clear();
}

void MessagePackWriter::SetBuffer(void* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_ASSERT_GREATER(size, 0u);

    m_pBuffer = reinterpret_cast<nn::Bit8*>(pBuffer);
    m_Size = size;

    Clear();
}

void MessagePackWriter::Clear() NN_NOEXCEPT
{
    m_Stream.pBuffer = m_pBuffer;
    m_Stream.size = m_Size;
    m_Stream.written = 0;
}

bool MessagePackWriter::WriteMap(size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    if (num <= 0x0F)
    {
        if (m_Stream.size < 1)
        {
            return false;
        }

        m_Stream.pBuffer[0] = 0x80 | static_cast<nn::Bit8>(num);
        m_Stream++;
    }
    else if (num <= 0xFFFF)
    {
        if (m_Stream.size < 3)
        {
            return false;
        }

        m_Stream.pBuffer[0] = 0xDE;
        m_Stream++;

        WriteNumeric(m_Stream, static_cast<uint16_t>(num));
    }
    else
    {
        if (m_Stream.size < 5)
        {
            return false;
        }

        m_Stream.pBuffer[0] = 0xDF;
        m_Stream++;

        WriteNumeric(m_Stream, static_cast<uint32_t>(num));
    }

    return true;
}

bool MessagePackWriter::WriteArray(size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    if (num <= 0x0F)
    {
        if (m_Stream.size < 1)
        {
            return false;
        }

        WriteByte(m_Stream, 0x90 | static_cast<nn::Bit8>(num));
    }
    else if (num <= 0xFFFF)
    {
        if (m_Stream.size < 3)
        {
            return false;
        }

        WriteByte(m_Stream, 0xDC);
        WriteNumeric(m_Stream, static_cast<uint16_t>(num));
    }
    else
    {
        if (m_Stream.size < 5)
        {
            return false;
        }

        WriteByte(m_Stream, 0xDD);
        WriteNumeric(m_Stream, static_cast<uint32_t>(num));
    }

    return true;
}

bool MessagePackWriter::WriteKey(const char* pKey) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t length = static_cast<size_t>(nn::util::Strnlen(pKey, INT_MAX));

    if (length == 0 || length >= m_Size)
    {
        return false;
    }

    return WriteValue(pKey, length);
}

bool MessagePackWriter::WriteValue(nullptr_t) NN_NOEXCEPT
{
    if (m_Stream.size < 1)
    {
        return false;
    }

    WriteByte(m_Stream, 0xC0);

    return true;
}

bool MessagePackWriter::WriteValue(bool value) NN_NOEXCEPT
{
    if (m_Stream.size < 1)
    {
        return false;
    }

    WriteByte(m_Stream, value ? 0xC3 : 0xC2);

    return true;
}

bool MessagePackWriter::WriteValue(int8_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(int16_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(int32_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(int64_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(uint8_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(uint16_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(uint32_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(static_cast<int64_t>(value));
}

bool MessagePackWriter::WriteValue(uint64_t value) NN_NOEXCEPT
{
    return WriteIntegerValue(value);
}

bool MessagePackWriter::WriteValue(float value) NN_NOEXCEPT
{
    if (m_Stream.size < 5)
    {
        return false;
    }

    WriteByte(m_Stream, 0xCA);
    WriteNumeric(m_Stream, value);

    return true;
}

bool MessagePackWriter::WriteValue(double value) NN_NOEXCEPT
{
    if (m_Stream.size < 9)
    {
        return false;
    }

    WriteByte(m_Stream, 0xCB);
    WriteNumeric(m_Stream, value);

    return true;
}

bool MessagePackWriter::WriteValue(const char* pValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    return WriteValue(pValue, static_cast<size_t>(nn::util::Strnlen(pValue, INT_MAX)));
}

bool MessagePackWriter::WriteValue(const char* pValue, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_LESS_EQUAL(length, UINT32_MAX);

    if (length > 0 && !nn::util::VerifyUtf8String(pValue, length))
    {
        return false;
    }

    if (length <= 0x1F)
    {
        if (m_Stream.size < 1 + length)
        {
            return false;
        }

        WriteByte(m_Stream, 0xA0 | static_cast<nn::Bit8>(length));
    }
    else if (length <= 0xFF)
    {
        if (m_Stream.size < 2 + length)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD9);
        WriteNumeric(m_Stream, static_cast<uint8_t>(length));
    }
    else if (length <= 0xFFFF)
    {
        if (m_Stream.size < 3 + length)
        {
            return false;
        }

        WriteByte(m_Stream, 0xDA);
        WriteNumeric(m_Stream, static_cast<uint16_t>(length));
    }
    else
    {
        if (m_Stream.size < 5 + length)
        {
            return false;
        }

        WriteByte(m_Stream, 0xDB);
        WriteNumeric(m_Stream, static_cast<uint32_t>(length));
    }

    if (length > 0)
    {
        std::memcpy(m_Stream.pBuffer, pValue, length);
        m_Stream += length;
    }

    return true;
}

bool MessagePackWriter::WriteValue(const void* pValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_LESS_EQUAL(size, UINT32_MAX);

    if (size <= 0xFF)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC4);
        WriteNumeric(m_Stream, static_cast<uint8_t>(size));
    }
    else if (size <= 0xFFFF)
    {
        if (m_Stream.size < 3 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC5);
        WriteNumeric(m_Stream, static_cast<uint16_t>(size));
    }
    else
    {
        if (m_Stream.size < 5 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC6);
        WriteNumeric(m_Stream, static_cast<uint32_t>(size));
    }

    if (size > 0)
    {
        std::memcpy(m_Stream.pBuffer, pValue, size);
        m_Stream += size;
    }

    return true;
}

bool MessagePackWriter::WriteValue(const SendBuffer& value) NN_NOEXCEPT
{
    return WriteValue(value.pBuffer, value.bufferSize);
}

bool MessagePackWriter::WriteValue(int8_t type, const void* pValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_LESS_EQUAL(size, UINT32_MAX);

    if (size == 1)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD4);
    }
    else if (size == 2)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD5);
    }
    else if (size == 4)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD6);
    }
    else if (size == 8)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD7);
    }
    else if (size == 16)
    {
        if (m_Stream.size < 2 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD8);
    }
    else if (size <= 0xFF)
    {
        if (m_Stream.size < 3 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC7);
        WriteNumeric(m_Stream, static_cast<uint8_t>(size));
    }
    else if (size <= 0xFFFF)
    {
        if (m_Stream.size < 4 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC8);
        WriteNumeric(m_Stream, static_cast<uint16_t>(size));
    }
    else
    {
        if (m_Stream.size < 6 + size)
        {
            return false;
        }

        WriteByte(m_Stream, 0xC9);
        WriteNumeric(m_Stream, static_cast<uint32_t>(size));
    }

    WriteNumeric(m_Stream, type);

    if (size > 0)
    {
        std::memcpy(m_Stream.pBuffer, pValue, size);
        m_Stream += size;
    }

    return true;
}

void* MessagePackWriter::GetBuffer() const NN_NOEXCEPT
{
    return m_pBuffer;
}

size_t MessagePackWriter::GetWrittenSize() const NN_NOEXCEPT
{
    return m_Stream.written;
}

bool MessagePackWriter::WriteIntegerValue(int64_t value) NN_NOEXCEPT
{
    if (value >= -32 && value <= 127)
    {
        if (m_Stream.size < 1)
        {
            return false;
        }

        WriteNumeric(m_Stream, static_cast<int8_t>(value));
    }
    else if (nn::util::IsIntValueRepresentable<int8_t>(value))
    {
        if (m_Stream.size < 2)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD0);
        WriteNumeric(m_Stream, static_cast<int8_t>(value));
    }
    else if (nn::util::IsIntValueRepresentable<int16_t>(value))
    {
        if (m_Stream.size < 3)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD1);
        WriteNumeric(m_Stream, static_cast<int16_t>(value));
    }
    else if (nn::util::IsIntValueRepresentable<int32_t>(value))
    {
        if (m_Stream.size < 5)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD2);
        WriteNumeric(m_Stream, static_cast<int32_t>(value));
    }
    else
    {
        if (m_Stream.size < 9)
        {
            return false;
        }

        WriteByte(m_Stream, 0xD3);
        WriteNumeric(m_Stream, value);
    }

    return true;
}

bool MessagePackWriter::WriteIntegerValue(uint64_t value) NN_NOEXCEPT
{
    if (nn::util::IsIntValueRepresentable<int64_t>(value))
    {
        return WriteIntegerValue(static_cast<int64_t>(value));
    }
    else
    {
        if (m_Stream.size < 9)
        {
            return false;
        }

        WriteByte(m_Stream, 0xCF);
        WriteNumeric(m_Stream, value);
    }

    return true;
}

}}}}
