﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackMemoryInputStream.h>

namespace nn { namespace ens { namespace detail { namespace util {

namespace
{
    enum FormatType
    {
        FormatType_Unknown,            //!< 不明なデータフォーマット
        FormatType_Nil,                //!< 空
        FormatType_False,              //!< 論理型（False）
        FormatType_True,               //!< 論理型（True）
        FormatType_PositiveFixInteger, //!< 整数型
        FormatType_NegativeFixInteger, //!< 整数型
        FormatType_Integer8,           //!< 整数型
        FormatType_Integer16,          //!< 整数型
        FormatType_Integer32,          //!< 整数型
        FormatType_Integer64,          //!< 整数型
        FormatType_UnsignedInteger8,   //!< 整数型
        FormatType_UnsignedInteger16,  //!< 整数型
        FormatType_UnsignedInteger32,  //!< 整数型
        FormatType_UnsignedInteger64,  //!< 整数型
        FormatType_Float32,            //!< 浮動小数点数型
        FormatType_Float64,            //!< 浮動小数点数型
        FormatType_FixString,          //!< 文字列型
        FormatType_String8,            //!< 文字列型
        FormatType_String16,           //!< 文字列型
        FormatType_String32,           //!< 文字列型
        FormatType_Binary8,            //!< バイナリデータ型
        FormatType_Binary16,           //!< バイナリデータ型
        FormatType_Binary32,           //!< バイナリデータ型
        FormatType_FixExtension1,      //!< 拡張データ型
        FormatType_FixExtension2,      //!< 拡張データ型
        FormatType_FixExtension4,      //!< 拡張データ型
        FormatType_FixExtension8,      //!< 拡張データ型
        FormatType_FixExtension16,     //!< 拡張データ型
        FormatType_Extension8,         //!< 拡張データ型
        FormatType_Extension16,        //!< 拡張データ型
        FormatType_Extension32,        //!< 拡張データ型
        FormatType_FixMap,             //!< 連想配列
        FormatType_Map16,              //!< 連想配列
        FormatType_Map32,              //!< 連想配列
        FormatType_FixArray,           //!< 配列
        FormatType_Array16,            //!< 配列
        FormatType_Array32,            //!< 配列
    };
}

namespace
{
    // バイナリデータからフォーマット種別への変換を行う。
    FormatType GetFormatType(nn::Bit8 data) NN_NOEXCEPT
    {
        if (data >= 0x00 && data <= 0x7F)
        {
            return FormatType_PositiveFixInteger;
        }
        if (data >= 0x80 && data <= 0x8F)
        {
            return FormatType_FixMap;
        }
        if (data >= 0x90 && data <= 0x9F)
        {
            return FormatType_FixArray;
        }
        if (data >= 0xA0 && data <= 0xBF)
        {
            return FormatType_FixString;
        }

        switch (data)
        {
        case 0xC0: return FormatType_Nil;
        case 0xC1: return FormatType_Unknown; // never used
        case 0xC2: return FormatType_False;
        case 0xC3: return FormatType_True;
        case 0xC4: return FormatType_Binary8;
        case 0xC5: return FormatType_Binary16;
        case 0xC6: return FormatType_Binary32;
        case 0xC7: return FormatType_Extension8;
        case 0xC8: return FormatType_Extension16;
        case 0xC9: return FormatType_Extension32;
        case 0xCA: return FormatType_Float32;
        case 0xCB: return FormatType_Float64;
        case 0xCC: return FormatType_UnsignedInteger8;
        case 0xCD: return FormatType_UnsignedInteger16;
        case 0xCE: return FormatType_UnsignedInteger32;
        case 0xCF: return FormatType_UnsignedInteger64;
        case 0xD0: return FormatType_Integer8;
        case 0xD1: return FormatType_Integer16;
        case 0xD2: return FormatType_Integer32;
        case 0xD3: return FormatType_Integer64;
        case 0xD4: return FormatType_FixExtension1;
        case 0xD5: return FormatType_FixExtension2;
        case 0xD6: return FormatType_FixExtension4;
        case 0xD7: return FormatType_FixExtension8;
        case 0xD8: return FormatType_FixExtension16;
        case 0xD9: return FormatType_String8;
        case 0xDA: return FormatType_String16;
        case 0xDB: return FormatType_String32;
        case 0xDC: return FormatType_Array16;
        case 0xDD: return FormatType_Array32;
        case 0xDE: return FormatType_Map16;
        case 0xDF: return FormatType_Map32;
        default:
            break;
        }

        if (data >= 0xE0 && data <= 0xFF)
        {
            return FormatType_NegativeFixInteger;
        }

        return FormatType_Unknown;
    }

    // 数値を読み込む。
    template <typename T>
    bool ReadNumeric(T* pOutValue, MessagePackInputStream* pStream) NN_NOEXCEPT
    {
        T value;
        NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(&value, sizeof (value)));

        *pOutValue = nn::util::LoadBigEndian(&value);

        return true;
    }
}

MessagePackReader::MessagePackReader() NN_NOEXCEPT
    : m_pDataBuffer(nullptr)
    , m_DataBufferSize(0)
    , m_pCallback(nullptr)
    , m_pCallbackParam(nullptr)
{
}

void MessagePackReader::SetCallback(ReadCallback pCallback, void* pParam) NN_NOEXCEPT
{
    m_pCallback = pCallback;
    m_pCallbackParam = pParam;
}

bool MessagePackReader::Read(const void* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    if (size == 0)
    {
        return false;
    }

    MessagePackMemoryInputStream stream;

    stream.Open(pData, size);

    return Read(&stream, nullptr, 0);
}

bool MessagePackReader::Read(MessagePackInputStream* pStream, void* pDataBuffer, size_t dataBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStream);
    NN_SDK_REQUIRES((pDataBuffer != nullptr && dataBufferSize > 0) || (pDataBuffer == nullptr && dataBufferSize == 0));

    m_pDataBuffer = reinterpret_cast<nn::Bit8*>(pDataBuffer);
    m_DataBufferSize = dataBufferSize;

    m_JsonPath.Clear();

    NN_DETAIL_ENS_RETURN_IF_FALSE(ReadRecursively(pStream));

    nn::Bit8 end;

    // 入力ストリーム内にデータがまだ残っていたら不正なデータ扱いとする。
    if (pStream->Read(&end, 1))
    {
        return false;
    }

    return true;
}

bool MessagePackReader::ReadRecursively(MessagePackInputStream* pStream) NN_NOEXCEPT
{
    nn::Bit8 firstByte = 0;
    NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(&firstByte, sizeof (firstByte)));

    switch (GetFormatType(firstByte))
    {
    case FormatType_Unknown:
        {
            return false;
        }
    case FormatType_Nil:
        {
            DataHolder holder = {DataHolderType_Nil};

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_False:
        {
            DataHolder holder = {DataHolderType_Boolean};
            holder.boolean = false;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_True:
        {
            DataHolder holder = {DataHolderType_Boolean};
            holder.boolean = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_PositiveFixInteger:
    case FormatType_NegativeFixInteger:
        {
            int value = static_cast<int8_t>(firstByte);

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Integer8:
        {
            int8_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Integer16:
        {
            int16_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Integer32:
        {
            int32_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Integer64:
        {
            int64_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_UnsignedInteger8:
        {
            uint8_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_UnsignedInteger16:
        {
            uint16_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_UnsignedInteger32:
        {
            uint32_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.s64 = value;
            holder.integer.isSigned = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_UnsignedInteger64:
        {
            uint64_t value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Integer};
            holder.integer.u64 = value;
            holder.integer.isSigned = false;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Float32:
        {
            float value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Float};
            holder.float64 = value;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Float64:
        {
            double value;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&value, pStream));

            DataHolder holder = {DataHolderType_Float};
            holder.float64 = value;

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixString:
        {
            size_t length = firstByte & 0x1F;

            DataHolder holder = {DataHolderType_String};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&holder.string, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_String8:
        {
            uint8_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_String};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&holder.string, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_String16:
        {
            uint16_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_String};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&holder.string, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_String32:
        {
            uint32_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_String};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&holder.string, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Binary8:
        {
            uint8_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Binary};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadBinary(&holder.binary, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Binary16:
        {
            uint16_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Binary};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadBinary(&holder.binary, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Binary32:
        {
            uint32_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Binary};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadBinary(&holder.binary, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixExtension1:
        {
            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, 1));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixExtension2:
        {
            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, 2));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixExtension4:
        {
            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, 4));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixExtension8:
        {
            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, 8));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixExtension16:
        {
            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, 16));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Extension8:
        {
            uint8_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Extension16:
        {
            uint16_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_Extension32:
        {
            uint32_t length;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

            DataHolder holder = {DataHolderType_Extension};
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadExtension(&holder.extension, pStream, length));

            NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));
        }
        break;
    case FormatType_FixMap:
        {
            size_t num = firstByte & 0x0F;

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadMap(pStream, num));
        }
        break;
    case FormatType_Map16:
        {
            uint16_t num;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&num, pStream));

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadMap(pStream, num));
        }
        break;
    case FormatType_Map32:
        {
            uint32_t num;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&num, pStream));

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadMap(pStream, num));
        }
        break;
    case FormatType_FixArray:
        {
            size_t num = firstByte & 0x0F;

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadArray(pStream, num));
        }
        break;
    case FormatType_Array16:
        {
            uint16_t num;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&num, pStream));

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadArray(pStream, num));
        }
        break;
    case FormatType_Array32:
        {
            uint32_t num;
            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&num, pStream));

            NN_DETAIL_ENS_RETURN_IF_FALSE(ReadArray(pStream, num));
        }
        break;
    default:
        {
            NN_SDK_ASSERT(0);
        }
        return false;
    }

    return true;
} // NOLINT(impl/function_size)

bool MessagePackReader::ReadString(String* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT
{
    if (length > 0)
    {
        if (!pStream->GetDataPointerFromMemoryCache(reinterpret_cast<const void**>(&outValue->pValue), length))
        {
            if (m_DataBufferSize < length)
            {
                return false;
            }

            NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(m_pDataBuffer, length));

            outValue->pValue = reinterpret_cast<const char*>(m_pDataBuffer);
        }

        NN_DETAIL_ENS_RETURN_IF_FALSE(nn::util::VerifyUtf8String(outValue->pValue, length));
    }

    outValue->length = length;

    return true;
}

bool MessagePackReader::ReadBinary(Binary* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT
{
    if (length > 0)
    {
        if (!pStream->GetDataPointerFromMemoryCache(&outValue->pValue, length))
        {
            if (m_DataBufferSize < length)
            {
                return false;
            }

            NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(m_pDataBuffer, length));

            outValue->pValue = m_pDataBuffer;
        }
    }

    outValue->length = length;

    return true;
}

bool MessagePackReader::ReadExtension(Extension* outValue, MessagePackInputStream* pStream, size_t length) NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&outValue->type, pStream));

    if (length > 0)
    {
        if (!pStream->GetDataPointerFromMemoryCache(&outValue->pValue, length))
        {
            if (m_DataBufferSize < length)
            {
                return false;
            }

            NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(m_pDataBuffer, length));

            outValue->pValue = m_pDataBuffer;
        }
    }

    outValue->length = length;

    return true;
}

bool MessagePackReader::ReadMap(MessagePackInputStream* pStream, size_t num) NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushObject());

    DataHolder holder = {DataHolderType_BeginMap};
    holder.integer.u64 = num;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    for (size_t i = 0; i < num; i++)
    {
        nn::Bit8 firstByte = 0;
        NN_DETAIL_ENS_RETURN_IF_FALSE(pStream->Read(&firstByte, sizeof (firstByte)));

        String key;

        // キーは文字列に限定する。（MessagePack の仕様上、文字列以外のキーは指定可能）
        switch (GetFormatType(firstByte))
        {
        case FormatType_FixString:
            {
                size_t length = firstByte & 0x1F;

                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&key, pStream, length));
            }
            break;
        case FormatType_String8:
            {
                uint8_t length;
                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&key, pStream, length));
            }
            break;
        case FormatType_String16:
            {
                uint16_t length;
                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&key, pStream, length));
            }
            break;
        case FormatType_String32:
            {
                uint32_t length;
                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadNumeric(&length, pStream));

                NN_DETAIL_ENS_RETURN_IF_FALSE(ReadString(&key, pStream, length));
            }
            break;
        default:
            return false;
        }

        NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushKey(key.pValue, key.length));

        NN_DETAIL_ENS_RETURN_IF_FALSE(ReadRecursively(pStream));
    }

    m_JsonPath.Pop();

    holder.type = DataHolderType_EndMap;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

bool MessagePackReader::ReadArray(MessagePackInputStream* pStream, size_t num) NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushArray());

    DataHolder holder = {DataHolderType_BeginArray};
    holder.integer.u64 = num;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    for (size_t i = 0; i < num; i++)
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(ReadRecursively(pStream));
    }

    m_JsonPath.Pop();

    holder.type = DataHolderType_EndArray;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

bool MessagePackReader::DoCallbackForNode(const DataHolder& holder) NN_NOEXCEPT
{
    if (m_pCallback)
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(m_pCallback(m_JsonPath, holder, m_pCallbackParam));
    }

    return true;
}

bool MessagePackReader::DoCallbackForLeaf(const DataHolder& holder) NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.IncrementArrayCountIfParentNodeIsArray());

    if (m_pCallback)
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(m_pCallback(m_JsonPath, holder, m_pCallbackParam));
    }

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

}}}}
