﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_Task.h>

namespace nn { namespace ens { namespace detail { namespace core {

namespace
{
    // アプリケーションの生存期間中、ユニークになる ID を発行する。
    TaskId GenerateId() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(std::atomic<uint64_t>, s_Id);

        TaskId taskId = {++s_Id};
        return taskId;
    }
}

Task::Task() NN_NOEXCEPT
    : m_CancelEvent(nn::os::EventClearMode_ManualClear)
    , m_Id(GenerateId())
    , m_pWorkBuffer(nullptr)
    , m_WorkBufferSize(0)
    , m_pContext(nullptr)
    , m_ErrorDetail(0)
{
}

Task::~Task() NN_NOEXCEPT
{
}

TaskId Task::GetId() const NN_NOEXCEPT
{
    return m_Id;
}

void Task::SetWorkBuffer(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    m_pWorkBuffer = pBuffer;
    m_WorkBufferSize = bufferSize;
}

void Task::RegisterAsyncContext(AsyncContext* pContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pContext);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_pContext = pContext;
}

void Task::UnregisterAsyncContext() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    m_pContext = nullptr;
}

void Task::Run() NN_NOEXCEPT
{
    nn::Result result = Main();

    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_pContext)
    {
        m_pContext->Complete(result, m_ErrorDetail);
        m_pContext = nullptr;
    }
}

void Task::Cancel() NN_NOEXCEPT
{
    m_CancelEvent.Signal();

    NN_UTIL_LOCK_GUARD(m_Mutex);

    if (m_pContext)
    {
        m_pContext->Complete(ResultCanceledByUser());
        m_pContext = nullptr;
    }
}

nn::os::SdkMutex& Task::GetParameterLock() NN_NOEXCEPT
{
    return m_Mutex;
}

void* Task::GetWorkBuffer() NN_NOEXCEPT
{
    return m_pWorkBuffer;
}

size_t Task::GetWorkBufferSize() const NN_NOEXCEPT
{
    return m_WorkBufferSize;
}

nn::os::Event& Task::GetCancelEvent() NN_NOEXCEPT
{
    return m_CancelEvent;
}

bool Task::IsAsyncContextRegistered() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Mutex.IsLockedByCurrentThread());

    return (m_pContext != nullptr);
}

void Task::SetErrorDetail(int32_t errorDetail) NN_NOEXCEPT
{
    m_ErrorDetail = errorDetail;
}

}}}}
