﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_HttpConnection.h>

namespace nn { namespace ens { namespace detail { namespace core {

HttpConnection::HttpConnection() NN_NOEXCEPT
    : m_pMultiHandle(nullptr)
    , m_pHandle(nullptr)
{
}

HttpConnection::~HttpConnection() NN_NOEXCEPT
{
    Cleanup();
}

nn::Result HttpConnection::Initialize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_pHandle, nullptr);

    m_pHandle = curl_easy_init();

    NN_RESULT_THROW_UNLESS(m_pHandle != nullptr, ResultHttpErrorOutOfMemory());

    m_pMultiHandle = curl_multi_init();

    if (!m_pMultiHandle)
    {
        curl_easy_cleanup(m_pHandle);
        m_pHandle = nullptr;

        NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
    }

    // 新しいコネクションを構築した時に古いコネクションが利用されないよう、同一ホストに対するコネクション数を 1 つに制限する。
    curl_multi_setopt(m_pMultiHandle, CURLMOPT_MAX_HOST_CONNECTIONS, 1);

    NN_RESULT_SUCCESS;
}

void HttpConnection::Cleanup() NN_NOEXCEPT
{
    if (m_pHandle)
    {
        curl_easy_cleanup(m_pHandle);
        m_pHandle = nullptr;

        curl_multi_cleanup(m_pMultiHandle);
        m_pMultiHandle = nullptr;
    }
}

void HttpConnection::Reset() NN_NOEXCEPT
{
    if (m_pHandle)
    {
        curl_easy_reset(m_pHandle);
    }
}

nn::Result HttpConnection::Perform(nn::os::Event* pCancelEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHandle);

    CURLMcode code = curl_multi_add_handle(m_pMultiHandle, m_pHandle);

    if (code != CURLM_OK)
    {
        if (code == CURLM_OUT_OF_MEMORY)
        {
            NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
        }
        else
        {
            NN_RESULT_THROW(ResultHttpErrorFailedInit());
        }
    }

    NN_UTIL_SCOPE_EXIT
    {
        curl_multi_remove_handle(m_pMultiHandle, m_pHandle);
    };

    int stillRunning = 0;

    curl_multi_perform(m_pMultiHandle, &stillRunning);

    do
    {
        int numFds = 0;
        code = curl_multi_wait(m_pMultiHandle, nullptr, 0, 10, &numFds);

        if (code != CURLM_OK)
        {
            if (code == CURLM_OUT_OF_MEMORY)
            {
                NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
            }
            else
            {
                NN_RESULT_THROW(ResultUnexpectedHttpError());
            }
        }
        if (numFds == 0)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }

        curl_multi_perform(m_pMultiHandle, &stillRunning);

        NN_RESULT_THROW_UNLESS(!(pCancelEvent && pCancelEvent->TryWait()), ResultCanceledByUser());
    }
    while (stillRunning > 0);

    CURLMsg* pMsg = nullptr;

    do
    {
        int num = 0;
        pMsg = curl_multi_info_read(m_pMultiHandle, &num);

        if (pMsg && (pMsg->msg == CURLMSG_DONE))
        {
            return detail::util::HandleHttpError(pMsg->data.result);
        }
    }
    while (pMsg);

    NN_RESULT_SUCCESS;
}

CURL* HttpConnection::GetHandle() NN_NOEXCEPT
{
    return m_pHandle;
}

}}}}
