﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_AuthenticationTask.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace core {

AuthenticationTask::AuthenticationTask() NN_NOEXCEPT
    : m_Credential(InvalidCredential)
    , m_AuthToken()
{
    m_TaskGetAuthToken.RegisterAsyncContext(&m_ContextForGetAuthToken);
}

AuthenticationTask::~AuthenticationTask() NN_NOEXCEPT
{
}

void AuthenticationTask::UnregisterAsyncContext() NN_NOEXCEPT
{
    m_TaskGetAuthToken.Cancel();

    HttpTask::UnregisterAsyncContext();
}

void AuthenticationTask::Cancel() NN_NOEXCEPT
{
    m_TaskGetAuthToken.Cancel();

    HttpTask::Cancel();
}

const Credential& AuthenticationTask::GetCredential() const NN_NOEXCEPT
{
    return m_Credential;
}

const AuthToken& AuthenticationTask::GetAuthToken() const NN_NOEXCEPT
{
    return m_AuthToken;
}

void AuthenticationTask::SetCredential(const Credential& credential) NN_NOEXCEPT
{
    m_Credential = credential;
}

void AuthenticationTask::SetAuthToken(const AuthToken& authToken) NN_NOEXCEPT
{
    m_AuthToken = authToken;
}

nn::Result AuthenticationTask::Main() NN_NOEXCEPT
{
    bool isTaskGetAuthTokenExecuted = false;

    if (m_AuthToken.value[0] == '\0')
    {
        NN_RESULT_DO(StepGetAuthToken());
        isTaskGetAuthTokenExecuted = true;
    }

    bool isRetryRequired = false;

    NN_RESULT_TRY(ExecuteSteps())
        NN_RESULT_CATCH(ResultAuthTokenExpired)
        {
            isRetryRequired = true;
        }
        NN_RESULT_CATCH(ResultAuthTokenVerificationFailed)
        {
            isRetryRequired = true;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    // 再試行は 1 回だけ。
    if (isRetryRequired && !isTaskGetAuthTokenExecuted)
    {
        NN_RESULT_DO(StepGetAuthToken());
        NN_RESULT_DO(ExecuteSteps());
    }

    NN_RESULT_SUCCESS;
}

nn::Result AuthenticationTask::StepGetAuthToken() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        m_TaskGetAuthToken.SetParameter(&m_AuthToken, m_Credential);
        m_TaskGetAuthToken.SetNetworkServiceAccountIdToken(GetNetworkServiceAccountIdToken());
    }

    m_TaskGetAuthToken.SetWorkBuffer(GetWorkBuffer(), GetWorkBufferSize());
    m_TaskGetAuthToken.SetHttpConnectionProvider(GetHttpConnectionProvider());

    m_TaskGetAuthToken.Run();

    SetErrorDetail(m_ContextForGetAuthToken.GetErrorDetail());
    NN_RESULT_DO(m_ContextForGetAuthToken.GetResult());

    NN_RESULT_SUCCESS;
}

}}}}
