﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/app/acbaa/task/ens_TaskGetMyDesignHeaderList.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace app_acbaa { namespace task {

TaskGetMyDesignHeaderList::TaskGetMyDesignHeaderList() NN_NOEXCEPT
    : m_pOutCount(nullptr)
    , m_pOutTotalCount(nullptr)
    , m_pOutHeaderList(nullptr)
    , m_Offset(0)
    , m_ListCount(0)
    , m_AuthorId(InvalidMyDesignAuthorId)
    , m_pSearchQuery(nullptr)
    , m_SetIndex(0)
    , m_IsHeaderListParsing(false)
{
}

TaskGetMyDesignHeaderList::~TaskGetMyDesignHeaderList() NN_NOEXCEPT
{
}

void TaskGetMyDesignHeaderList::SetParameter(int* pOutCount, int* pOutTotalCount, MyDesignHeader pOutHeaderList[],
    int count, int offset, const MyDesignAuthorId& authorId) NN_NOEXCEPT
{
    m_pOutCount = pOutCount;
    m_pOutTotalCount = pOutTotalCount;
    m_pOutHeaderList = pOutHeaderList;
    m_Offset = offset;
    m_ListCount = count;

    m_AuthorId = authorId;
    m_pSearchQuery = nullptr;
}

void TaskGetMyDesignHeaderList::SetParameter(int* pOutCount, int* pOutTotalCount, MyDesignHeader pOutHeaderList[],
    int count, int offset, const char* pSearchQuery) NN_NOEXCEPT
{
    m_pOutCount = pOutCount;
    m_pOutTotalCount = pOutTotalCount;
    m_pOutHeaderList = pOutHeaderList;
    m_Offset = offset;
    m_ListCount = count;

    m_AuthorId = InvalidMyDesignAuthorId;
    m_pSearchQuery = pSearchQuery;
}

void TaskGetMyDesignHeaderList::SetParameter(int* pOutCount, int* pOutTotalCount, MyDesignHeader pOutHeaderList[],
    int count, int offset) NN_NOEXCEPT
{
    m_pOutCount = pOutCount;
    m_pOutTotalCount = pOutTotalCount;
    m_pOutHeaderList = pOutHeaderList;
    m_Offset = offset;
    m_ListCount = count;

    m_AuthorId = InvalidMyDesignAuthorId;
    m_pSearchQuery = nullptr;
}

nn::Result TaskGetMyDesignHeaderList::ExecuteSteps() NN_NOEXCEPT
{
    NN_RESULT_DO(StepGet());

    NN_RESULT_SUCCESS;
}

nn::Result TaskGetMyDesignHeaderList::StepGet() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char query[1536] = {};
    int offset = 0;

    // 入力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), nn::ens::ResultInvalidState());

        if (m_AuthorId != InvalidMyDesignAuthorId)
        {
            offset += nn::util::SNPrintf(&query[offset], sizeof (query) - offset, "design_player_id=%llu&",
                m_AuthorId.value);
        }

        offset += nn::util::SNPrintf(&query[offset], sizeof (query) - offset, "offset=%d&limit=%d",
            m_Offset, m_ListCount);

        if (m_pSearchQuery)
        {
            int length = nn::util::Strnlen(m_pSearchQuery, static_cast<int>(SearchQueryLengthMax + 1));

            NN_RESULT_THROW_UNLESS(length > 0 && length <= SearchQueryLengthMax, ResultInvalidArgument());

            // SearchQueryBuilder はロジック簡略化のため、先頭に '&' を付けている。
            NN_RESULT_THROW_UNLESS(m_pSearchQuery[0] == '&', ResultInvalidArgument());

            NN_RESULT_THROW_UNLESS(detail::util::VerifyUriQueryString(m_pSearchQuery, length), ResultInvalidArgument());

            offset += nn::util::SNPrintf(&query[offset], sizeof (query) - offset, "%s", m_pSearchQuery);

            NN_SDK_ASSERT_LESS(static_cast<size_t>(offset), sizeof (query));
        }
    }

    char url[2048] = {};
    detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/designs?%s", query);

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "GET"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));

    detail::core::HttpRequestHeader header;

    NN_RESULT_DO(header.Add("Authorization", "Bearer %s", GetAuthToken().value));
    NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    // 出力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        detail::util::MessagePackReader reader;

        reader.SetCallback(ResponseReadCallback, this);

        std::memset(&m_ListParseBits, 0, sizeof (m_ListParseBits));

        NN_RESULT_THROW_UNLESS(reader.Read(GetResponse(), GetResponseSize()), ResultInvalidResponse());
    }

    NN_RESULT_SUCCESS;
}

bool TaskGetMyDesignHeaderList::ResponseReadCallbackImpl(const nn::ens::detail::util::JsonPath& jsonPath,
    const detail::util::DataHolder& holder) NN_NOEXCEPT
{
    if (!m_ListParseBits.total &&
        jsonPath.Compare("$.total"))
    {
        m_ListParseBits.total = true;

        NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(m_pOutTotalCount));
    }
    else if (!m_ListParseBits.count &&
        jsonPath.Compare("$.count"))
    {
        m_ListParseBits.count = true;

        NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(m_pOutCount));
    }
    else if ((holder.type == detail::util::DataHolderType_BeginArray || holder.type == detail::util::DataHolderType_EndArray) &&
        jsonPath.Match("$.headers[]"))
    {
        m_IsHeaderListParsing = (holder.type == detail::util::DataHolderType_BeginArray);
    }
    else if ((holder.type == detail::util::DataHolderType_BeginMap || holder.type == detail::util::DataHolderType_EndMap) &&
        jsonPath.Match("$.headers[*]"))
    {
        if (holder.type == detail::util::DataHolderType_BeginMap)
        {
            NN_DETAIL_ENS_RETURN_IF_FALSE(m_ListCount > m_SetIndex);

            MyDesignHeader& header = m_pOutHeaderList[m_SetIndex];

            header.id = InvalidMyDesignId;
            header.authorName.value[0] = '\0';
            header.updatedAt.value = 0;

            std::memset(&m_HeaderParseBits, 0, sizeof (m_HeaderParseBits));
        }
        else
        {
            const MyDesignHeader& header = m_pOutHeaderList[m_SetIndex];

            NN_DETAIL_ENS_RETURN_IF_FALSE(header.id != InvalidMyDesignId);
            NN_DETAIL_ENS_RETURN_IF_FALSE(header.updatedAt.value != 0);

            m_SetIndex++;
        }
    }
    else if (m_IsHeaderListParsing)
    {
        MyDesignHeader& header = m_pOutHeaderList[m_SetIndex];

        if (!m_HeaderParseBits.myDesignId &&
            jsonPath.Match("$.headers[*].id"))
        {
            m_HeaderParseBits.myDesignId = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.id.value));
        }
        else if (!m_HeaderParseBits.authorName &&
            jsonPath.Match("$.headers[*].design_player_name"))
        {
            m_HeaderParseBits.authorName = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(header.authorName.value, sizeof (header.authorName.value)));
        }
        else if (!m_HeaderParseBits.updatedAt &&
            jsonPath.Match("$.headers[*].updated_at"))
        {
            m_HeaderParseBits.updatedAt = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.updatedAt.value));
        }
        else if (!m_HeaderParseBits.digest &&
            jsonPath.Match("$.headers[*].digest"))
        {
            m_HeaderParseBits.digest = true;

            if (!(holder.type == detail::util::DataHolderType_String && holder.string.length == Digest::StringLength))
            {
                return false;
            }

            NN_DETAIL_ENS_RETURN_IF_FALSE(Digest::FromString(&header.digest, holder.string.pValue));
        }
        else if (!m_HeaderParseBits.metadata &&
            jsonPath.Match("$.headers[*].meta"))
        {
            m_HeaderParseBits.metadata = true;

            if (header.metadata.IsNull())
            {
                // メタデータバッファの空指定は許可するが、サイズ 0 のバイナリ以外だったらエラーとする。
                if (!(holder.type == detail::util::DataHolderType_Binary && holder.binary.length == 0))
                {
                    return false;
                }

                header.metadata.receivedSize = 0;
            }
            else
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.metadata));
            }
        }
    }

    return true;
} // NOLINT(impl/function_size)

bool TaskGetMyDesignHeaderList::ResponseReadCallback(const nn::ens::detail::util::JsonPath& jsonPath,
    const detail::util::DataHolder& holder, void* pParam) NN_NOEXCEPT
{
    TaskGetMyDesignHeaderList* pThis = reinterpret_cast<TaskGetMyDesignHeaderList*>(pParam);

    return pThis->ResponseReadCallbackImpl(jsonPath, holder);
}

}}}}}
