﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/app/acbaa/task/ens_TaskActivateNotificationService.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_NpnsLibrary.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>
#include <nn/npns.h>

namespace nn { namespace ens { namespace detail { namespace app_acbaa { namespace task {

TaskActivateNotificationService::TaskActivateNotificationService() NN_NOEXCEPT
    : m_User(nn::account::InvalidUid)
{
    m_NotificationToken.data[0] = '\0';
}

TaskActivateNotificationService::~TaskActivateNotificationService() NN_NOEXCEPT
{
}

void TaskActivateNotificationService::SetParameter(const nn::account::Uid& user) NN_NOEXCEPT
{
    m_User = user;
}

nn::Result TaskActivateNotificationService::ExecuteSteps() NN_NOEXCEPT
{
    NN_RESULT_DO(StepGetToken());
    NN_RESULT_DO(StepUploadToken());

    NN_RESULT_SUCCESS;
}

nn::Result TaskActivateNotificationService::StepGetToken() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    detail::util::NpnsLibrary::Initialize();

    // 通知トークン取得中はキャンセルできない。
    NN_RESULT_DO(nn::npns::CreateToken(&m_NotificationToken, m_User));

#else

    nn::util::Strlcpy(m_NotificationToken.data, "dummy notification token", sizeof (m_NotificationToken.data));

#endif

    // NN_DETAIL_ENS_INFO("[ens] NotificationToken = %s\n", m_NotificationToken.data);

    NN_RESULT_SUCCESS;
}

nn::Result TaskActivateNotificationService::StepUploadToken() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char url[256] = {};
    detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/notification_tokens");

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "POST"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));

    detail::core::HttpRequestHeader header;

    // 入力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        detail::util::MessagePackWriter writer;

        writer.SetBuffer(GetRequestBuffer(), GetRequestBufferSize());

        NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteMap(1));
        {
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("token"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_NotificationToken.data,
                static_cast<size_t>(nn::util::Strnlen(m_NotificationToken.data, sizeof (m_NotificationToken.data)))));
        }

        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, writer.GetBuffer()));
        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE, writer.GetWrittenSize()));
    }

    NN_RESULT_DO(header.Add("Authorization", "Bearer %s", GetAuthToken().value));
    NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    NN_RESULT_SUCCESS;
}

}}}}}
