﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eftvw2_PacketProcedure.h>
#include <nw/eft/eftvw2_ViewerSystem.h>
#include <nw/eft/eftcom_Message.h>
#include <nw/eft/eftcom_Protocol.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_Resource.h>
#include <nw/eft/eftvw2_Data.h>


#ifndef DLOG
#if 0
#define DLOG( x, ... ) printf( x, ##__VA_ARGS__ )
#else
#define DLOG( x, ... )
#endif
#endif//DLOG

namespace nw   {
namespace eftvw2 {

//------------------------------------------------------------------------------
//      パケットを処理
//------------------------------------------------------------------------------
void PacketProcedure::DoPacket( char* buffer, size_t szBuff)
{
    EFT_UNUSED_VARIABLE( buffer );
    EFT_UNUSED_VARIABLE( szBuff );

    nw::eftcom::Message* header = reinterpret_cast<nw::eftcom::Message*>( buffer );

    DLOG( "MessageType = %d\n", header->type );
    DLOG( "BufferSize  = %d\n", header->bufferSize );
    buffer += sizeof(nw::eftcom::Message);

    switch( header->type )
    {
    case nw::eftcom::MESSAGE_TYPE_CONTROL:
        {
            // 制御メッセージ.
            HandleMessageControl( buffer );
        }
        break;

    case nw::eftcom::MESSAGE_TYPE_SEND_BINARY:
        {
            // バイナリ転送.
            HandleMessageSendBinary( buffer, header->bufferSize );
        }
        break;

    case nw::eftcom::MESSAGE_TYPE_CREATE:
        {
            // インスタンス生成.
            HandleMessageCreate( buffer );
        }
        break;

    case nw::eftcom::MESSAGE_TYPE_MODIFY:
        {
            // リアルタイム編集.
            HandleMessageModify( buffer );
        }
        break;

    case nw::eftcom::MESSAGE_TYPE_DELETE:
        {
            // インスタンス削除.
            HandleMessageDelete( buffer );
        }
        break;

    case nw::eftcom::MESSAGE_TYPE_REMOVE_BINARY:
        {
            // バイナリ削除.
            HandleMessageRemoveBinary( buffer );
        }
        break;
    }
}


//------------------------------------------------------------------------------
//      制御メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageControl( char* pBuffer )
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Control Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nw::eftcom::MessageInfo info = (*reinterpret_cast<nw::eftcom::MessageInfo*>( pBuffer ));
    NW_UNUSED_VARIABLE( info );

    DLOG( "Total Packet Size = %d\n", info.packetSize );
    DLOG( "Number Of Packets = %u\n", info.numPacket );
    DLOG( "Process ID        = %u\n", info.processID );
}


//------------------------------------------------------------------------------
//      バイナリ転送メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageSendBinary( char* pBuffer, u32 bufferSize )
{
    NW_UNUSED_VARIABLE( bufferSize );

    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Send Binary Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nw::eftcom::SendBinaryMessage* binMsg = (nw::eftcom::SendBinaryMessage*)pBuffer;

    DLOG("Send binary message %d \n", binMsg->assetType);

    switch ( binMsg->assetType )
    {
    case eftcom::ASSET_TYPE_EMITTERSET:
        {
            void* binTop = pBuffer + sizeof( nw::eftcom::SendBinaryMessage );
            char* pPath = (char*)binTop;

            mViewerSystem->LoadEffectData( pPath, binMsg->guid );
            mViewerSystem->Play();
            //DLOG( "Resource Guid : %s\n", binMsg->guid.ToString() );
        }
        break;

    case eftcom::ASSET_TYPE_PREVIEW:
        {
            void* binTop = pBuffer + sizeof( nw::eftcom::SendBinaryMessage );
            char* pPath = (char*)binTop;

            void* pFileData = NULL;
            u32 fileSize = 0;
            if (FileSystem::Load( mHeap, pPath, &pFileData, &fileSize ) == false)
            {
                break;
            }

            if (pFileData == NULL)
            {
                break;
            }

            const u32 OFFSET = 80; // FileHeader + BinaryData = 16 + 32 + 32.
            void* fileBinTop = reinterpret_cast<char*>(pFileData) + OFFSET;

            nw::eft2::BinaryData* pBinary = static_cast<nw::eft2::BinaryData*>( fileBinTop );

            if ( pBinary->GetBinaryTag() != EFT_MAKE_TAG( 'P', 'R', 'E', 'V' ) )
            {
                mHeap->Free(pFileData);
                break;
            }

            ResEffectPreviewPack* pResEffectPreviewPack = pBinary->GetBinaryDataWithFlip<ResEffectPreviewPack>();

            mViewerSystem->SetResEffectPreview( binMsg->guid, pResEffectPreviewPack );

            mHeap->Free(pFileData);
            //DLOG( "Preview  Guid : %s\n", binMsg->guid.ToString() );
        }
        break;

    case eftcom::ASSET_TYPE_VIEWER_MODEL:
        {
            void* binTop = pBuffer + sizeof( nw::eftcom::SendBinaryMessage );
            char* pPath = (char*)binTop;

            void* pFileData = NULL;
            u32 fileSize = 0;
            if (FileSystem::Load( mHeap, pPath, &pFileData, &fileSize ) == false)
            {
                break;
            }

            if (pFileData == NULL)
            {
                break;
            }

            // 通信システム修正用
            const u32 OFFSET = 80; // FileHeader + BinaryData = 16 + 32 + 32.
            void* fileBinTop = reinterpret_cast<char*>(pFileData) + OFFSET;

            nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( fileBinTop );
            nw::eft2::BinaryData* pModelBinary = pNodeBinary->GetNextData();

            if ( pNodeBinary->GetBinaryTag() != EFT_MAKE_TAG( 'M', 'D', 'L', 'P' ) || pModelBinary == NULL)
            {
                mHeap->Free(pFileData);
                break;
            }

            // モデルプレビューを作成
            bool created = mViewerSystem->CreateModelPreview( binMsg->guid, pModelBinary->GetBinaryData(), pModelBinary->GetBinarySize() );

            // ResPreviewを設定
            if ( created )
            {
                ResModelPreviewPack* pResModelPreviewPack = pNodeBinary->GetBinaryDataWithFlip<ResModelPreviewPack>();
                mViewerSystem->SetResModelPreview( binMsg->guid, pResModelPreviewPack );
            }

            mHeap->Free(pFileData);
            //DLOG( "Preview  Guid : %s\n", binMsg->guid.ToString() );
        }
        break;

    case eftcom::ASSET_TYPE_VIEWER_DATA:
        {
            void* binTop = pBuffer + sizeof( nw::eftcom::SendBinaryMessage );
            char* pPath = (char*)binTop;

            void* pFileData = NULL;
            u32 fileSize = 0;
            if (FileSystem::Load( mHeap, pPath, &pFileData, &fileSize ) == false)
            {
                break;
            }

            if (pFileData == NULL)
            {
                break;
            }

            // 通信システム修正用
            const u32 OFFSET = 80; // FileHeader + BinaryData = 16 + 32 + 32.
            void* fileBinTop = reinterpret_cast<char*>( pFileData ) + OFFSET;
            u32 binarySize = fileSize - OFFSET;
            NW_UNUSED_VARIABLE( fileBinTop );

            // ビューアデータを保持するためのバッファを確保
            if ( mViewerTempDataSize < binarySize )
            {
                if ( mViewerTempData ) mHeap->Free( mViewerTempData );
                mViewerTempData = mHeap->Alloc( binarySize );
                mViewerTempDataSize = binarySize;
            }

            // リロードのたびにコピる
            memcpy( mViewerTempData, fileBinTop, binarySize );

            // ハンドリング先でフリップされる
            if ( mViewerDataCallback ) mViewerDataCallback( mHeap, mViewerDataCallbackObject, mViewerTempData );

            mHeap->Free(pFileData);
        }
        break;

    case eftcom::ASSET_TYPE_VIEWER_MESSAGE:
        {
            void* binTop = pBuffer + sizeof( nw::eftcom::SendBinaryMessage );
            char* pData = (char*)binTop;

            if (pData == NULL)
            {
                break;
            }

            // ビューアメッセージ用
            HandleMessageViewerControl(pData);
        }
        break;
    }
}


//------------------------------------------------------------------------------
//      インスタンス生成メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageCreate( char* pBuffer )
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect CREATE Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nw::eftcom::CreateMessage* binMsg = (nw::eftcom::CreateMessage*)pBuffer;

#if 0
    mViewerSystem->CreateEffectPreview( binMsg->resGuid, binMsg->previewGuid );
#else
    switch( binMsg->assetType )
    {
    case eftcom::ASSET_TYPE_EMITTERSET:
        {
            /* DO_NOTHING */
            /* このメッセージはエディタから送られません */
        }
        break;

    case eftcom::ASSET_TYPE_PREVIEW:
        {
            //DLOG( "Resource Guid : %s\n", binMsg->resGuid.ToString() );
            //DLOG( "Preview  Guid : %s\n", binMsg->previewGuid.ToString() );
            mViewerSystem->CreateEffectPreview( binMsg->resGuid, binMsg->previewGuid );
        }
        break;
    }
#endif
}


//------------------------------------------------------------------------------
//      リアルタイム編集メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageModify( char* pBuffer )
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Modifty Message.\n" );
    DLOG( "--------------------------------------------\n" );

    nw::eftcom::ModifyMessage* msg = (reinterpret_cast<nw::eftcom::ModifyMessage*>( pBuffer ));

    DLOG( "AssetID      = %u\n", msg->assetType );
//  DLOG( "GUID         = %s\n", msg.guid.ToString() );
    DLOG( "Offset       = %u\n", msg->offset );
    DLOG( "OverrideSize = %u\n", msg->overrideSize );
    DLOG( "WithReset    = %u\n", msg->isRequiredReset );
    DLOG( "--------------------------------------------\n" );

    switch( msg->assetType )
    {
    case nw::eftcom::ASSET_TYPE_EMITTER:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect EmitterData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            u8* pSrc = (u8 *)( (u32)pBuffer + sizeof(nw::eftcom::ModifyMessage) );

            // リソースのguidとエミッタ名で編集を行う。
            if ( msg->emitterName[0] != 0 )
            {
                const bool withReset = ( msg->isRequiredReset == 1 ) ? true : false;
                mViewerSystem->UpdateAssetEmitter(
                    msg->guid, msg->emitterName, msg->offset, pSrc, msg->overrideSize, withReset, static_cast<u8>(msg->endian) );
            }
        }
        break;

    case nw::eftcom::ASSET_TYPE_EMITTERSET:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect EmitterSetData!\n" );
            DLOG( "--------------------------------------------\n\n" );
        }
        break;

    case nw::eftcom::ASSET_TYPE_TEXTURE:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect TextureData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            /* 未実装 */
        }
        break;

    case nw::eftcom::ASSET_TYPE_PRIMITIVE:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect PrimitiveData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            /* 未実装 */
        }
        break;

    case nw::eftcom::ASSET_TYPE_SHADER:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect ShaderData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            /* 未実装 */
        }
        break;

    case nw::eftcom::ASSET_TYPE_PREVIEW:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect PreviewData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            // リソースのguidで編集を行う。
            u8* pSrc = (u8 *)( (u32)pBuffer + sizeof(nw::eftcom::ModifyMessage) );
            mViewerSystem->UpdateResEffectPreview( msg->guid, msg->offset, pSrc, msg->overrideSize, static_cast<u8>(msg->endian) );
        }
        break;

    case nw::eftcom::ASSET_TYPE_VIEWER_DATA:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect ViewerData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            // このクラスで保持しているテンポラリデータに対して上書きを行う
            u8* pSrc = (u8 *)( (u32)pBuffer + sizeof(nw::eftcom::ModifyMessage) );
            UpdateViewerData( msg->offset, pSrc, msg->overrideSize );
            if ( mViewerDataCallback ) mViewerDataCallback( mHeap, mViewerDataCallbackObject, mViewerTempData );
        }
        break;

    case nw::eftcom::ASSET_TYPE_VIEWER_MODEL:
        {
            DLOG( "--------------------------------------------\n" );
            DLOG( "Detect ViewerModelData!\n" );
            DLOG( "--------------------------------------------\n\n" );

            // リソースのguidで編集を行う。
            u8* pSrc = (u8 *)( (u32)pBuffer + sizeof(nw::eftcom::ModifyMessage) );
            mViewerSystem->UpdateResModelPreview( msg->guid, msg->offset, pSrc, msg->overrideSize, static_cast<u8>(msg->endian) );
        }
        break;
    }
}


//------------------------------------------------------------------------------
//      インスタンス削除メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageDelete( char* pBuffer )
{
    printf( "--------------------------------------------\n" );
    printf( "Detect Delete Message.\n" );
    printf( "--------------------------------------------\n" );
    nw::eftcom::DeleteMessage* msg = reinterpret_cast<nw::eftcom::DeleteMessage*>( pBuffer );

    switch( msg->assetType )
    {
    case nw::eftcom::ASSET_TYPE_EMITTERSET:
        {
            /* NOT_IMPLEMENT */
        }
        break;

    case nw::eftcom::ASSET_TYPE_PREVIEW:
        {
            mViewerSystem->UnLoadEffectData( msg->resGuid );
        }
        break;
    }
}


//------------------------------------------------------------------------------
//      バイナリ削除メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageRemoveBinary( char* pBuffer )
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Remove Binary Message.\n" );
    DLOG( "--------------------------------------------\n" );
    nw::eftcom::RemoveBinaryMessage* msg = reinterpret_cast<nw::eftcom::RemoveBinaryMessage*>( pBuffer );

    switch( msg->assetType )
    {
    case nw::eftcom::ASSET_TYPE_EMITTERSET:
        {
            /* NOT_IMPLEMENT */
        }
        break;

    case nw::eftcom::ASSET_TYPE_PREVIEW:
        {
            /* NOT_IMPLEMENT */
        }
        break;

    case nw::eftcom::ASSET_TYPE_VIEWER_MODEL:
        {
            mViewerSystem->DestroyModelPreview( msg->guid );
        }
        break;
    }
}


//------------------------------------------------------------------------------
//      ビューアの再生制御メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageViewerControl( char* pBuffer )
{
    void* binTop = pBuffer;
    u32* message = (u32*)binTop;
    u32 type = *message;

    switch ( type )
    {

    case eftcom::VIEWER_MESSAGE_TYPE_CONFIG:
        {
            message++;
            //　Config Dataを設定する
            if ( mViewerMessageCallback ) mViewerMessageCallback( mHeap, mViewerMessageCallbackObject, (void*)message );
        }
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_RESETFRAME:
        mViewerSystem->Play();
        mViewerSystem->ResetTime( false );
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_PLAY:
        mViewerSystem->Play();
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_PAUSE:
        if ( mViewerSystem->IsPause() )
        {
            mViewerSystem->Play();
        }
        else
        {
            mViewerSystem->Pause();
        }
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_STOP:
        mViewerSystem->Stop();
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_SETRANGE:
        {
            message++;
            s32 startFrame  = *message;
            message++;
            s32 endFrame    = *message;
            message++;
            s32 isLoop      = *message;

            mViewerSystem->SetFrameRange( (f32)startFrame, (f32)endFrame, isLoop > 0 ? 1 : 0 );
        }
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_STEPFRAME:
        {
            if ( !mViewerSystem->IsPause() )
            {
                mViewerSystem->Pause();
            }

            mViewerSystem->StepFrame();
        }
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_FORCEFADE:
        {
            mViewerSystem->ForceFade();
        }
        break;

    case eftcom::VIEWER_MESSAGE_TYPE_VISIBILITY:
        {
            message++;

            struct VisiblityData
            {
                eftcom::ASSET_TYPE assetType;
                eftcom::Guid       guid;
                char               targetName[64];
                bool               isVisible;
            };

            VisiblityData* pData = reinterpret_cast<VisiblityData*>( message );

            switch( pData->assetType )
            {

            // エミッタセットの表示/非表示
            case eftcom::ASSET_TYPE_EMITTERSET:
                {
                    mViewerSystem->SetEmitterSetVisibility( pData->guid, pData->isVisible );
                }
                break;

            // エミッタの表示/非表示
            case eftcom::ASSET_TYPE_EMITTER:
                {
                    mViewerSystem->SetEmitterVisibility( pData->guid, pData->targetName, pData->isVisible );
                }
                break;

            // プレビューの表示/非表示
            case eftcom::ASSET_TYPE_PREVIEW:
            case eftcom::ASSET_TYPE_VIEWER_MODEL:
                {
                    mViewerSystem->SetPreviewVisibility( pData->guid, pData->isVisible );
                }
                break;

            case eftcom::ASSET_TYPE_TEXTURE:
                break;

            case eftcom::ASSET_TYPE_PRIMITIVE:
                break;

            case eftcom::ASSET_TYPE_SHADER:
                break;

            case eftcom::ASSET_TYPE_VIEWER_DATA:
                break;

            case eftcom::ASSET_TYPE_VIEWER_MESSAGE:
                break;

            case eftcom::ASSET_TYPE_FORCE_4BYTE:
                break;

            }
        }
        break;

    // 描画順を取得
    case eftcom::VIEWER_MESSAGE_TYPE_DRAW_ORDER:
        {
            message++;

            struct OrderData
            {
                u8 groupId;
                u8 order;
            };

            OrderData* pData = reinterpret_cast<OrderData*>( message );

            // pDataが描画順データです。
            EFT_UNUSED_VARIABLE(pData);
#if 0
            for (int i = 0; i < 64; ++i)
            {
                printf("%2d: %d, %d\n", i, pData[i].groupId, pData[i].order);
            }
#endif
        }
        break;
    }
}


//---------------------------------------------------------------------------
//  ビューアデータを更新します。
//---------------------------------------------------------------------------
void PacketProcedure::UpdateViewerData( u32            offset,
                                        void*          overrideData,
                                        u32            overrideDataSize )
{
    if ( !mViewerTempData ) return;

    // テンポラリデータからResCameraを抽出
    nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( mViewerTempData );
    nw::eftvw2::ResCamera* camera     = pNodeBinary->GetBinaryDataWithFlip<nw::eftvw2::ResCamera>();

    // リアルタイム編集ですよフラグを立てる
    camera->isRealtimeEditing = 1;

#ifdef EFT_BI_ENDIAN
    // いったんBEに戻して上書き準備
    if ( pNodeBinary->endian != EFT_ENDIAN_TYPE )
    {
        camera->FlipEndian();
    }
#endif

    // 生のまま上書き
    u8* pDst = (u8*)( (u32)camera + offset );
    memcpy( pDst, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
    // 上書き後にLEに戻す
    if ( pNodeBinary->endian != EFT_ENDIAN_TYPE )
    {
        camera->FlipEndian();
    }
#endif

    return;
}


} // namespace eftvw2
} // namespace nw
