﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_Random.h>
#include <nw/eft/eft2_CurlNoiseData.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_AreaLoop.h>
#include <nw/eft/eft2_Callback.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_SuperStripe.h>
#include <nw/eft/eft2_StripeConnection.h>

namespace nw   {
namespace eft2 {

const float ViewParam::DefaultZOffset = 0.01f;  // Z-Fighting抑制のための深度オフセットの規定値

//　32bit float から 24bit float を生成します。
static u32 Float32ToBits24(f32 value);

bool System::g_Initialized = false;

//------------------------------------------------------------------------------
//  エフェクトシステムの生成を行います。
//------------------------------------------------------------------------------
System::System( const Config &config )
{
    if( g_Initialized )                   EFT_ERR( "[EFT] Error!! System is Initialized.\n" );
    if( !config.GetEffectHeap() )        EFT_ERR( "[EFT] Error!! EffectHeap is NULL.\n" );
    if( !config.GetEffectDynamicHeap() ) EFT_ERR( "[EFT] Error!! EffectDynamicHeap is NULL.\n" );

    m_DynamicUBOArray      = NULL;
    m_EmitterArray         = NULL;
    m_EmitterSetArray      = NULL;
    m_ResourceArray        = NULL;
    m_StreamOutEmitter     = NULL;
    m_StreamOutEmitterTail = NULL;

    // システム初期化
    Initialize( config.GetEffectHeap(), config.GetEffectDynamicHeap(), config );
}

//---------------------------------------------------------------------------
//  システムの初期化をします。
//---------------------------------------------------------------------------
void System::Initialize( Heap* staticHeap, Heap* dynamicHeap, const Config &config )
{
    // パラメータ初期化
    m_ResourceNum    = config.GetResourceNum() + Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM;
    m_EmitterSetNum  = config.GetEmitterSetNum();
    m_EmitterNum     = config.GetEmitterNum();
    m_StaticHeap     = staticHeap;

    // Misc.cpp の初期化
    SetStaticHeap(  staticHeap  );
    SetDynamicHeap( dynamicHeap );

    // Uboのキャッシュフラッシュ設定
    m_IsUboCpuChacheFlush = config.IsUniformBlockCpuChacheFlush();
    m_IsUboGpuChacheFlush = config.IsUniformBlockGpuChacheFlush();
    UniformBlockBuffer::SetCpuGpuChacheFlushSetting( m_IsUboCpuChacheFlush, m_IsUboGpuChacheFlush );

    m_BufferMode     = EFT_BUFFER_MODE_DOUBLE;

    {
        u32 temp = GetAllocedSizeFromStaticHeap();
        InitializeDelayFreeList( m_EmitterNum*3, GetBufferMode() ); // 1Gpuエミッタで最大3回動的確保が発生する
        m_DelayFreeWorkSize = GetAllocedSizeFromStaticHeap() - temp;
    }

    SetSuppressOutputLog( config.IsSuppressionLog() );

    m_IsEnableCalcProcess               = true;
    m_IsEnableDrawProcess               = true;
    m_IsEnableStreamOutProcess          = true;
    m_ZClipEnable                       = true;
    m_ColorBufferEnable[EFT_CPU_CORE_0] = true;
    m_ColorBufferEnable[EFT_CPU_CORE_1] = true;
    m_ColorBufferEnable[EFT_CPU_CORE_2] = true;
    m_IsBatchProcessStreamOutEmitter    = config.IsEnableStreamOutBatchProcess();
    m_EmitterSetIdx                     = 0;
    m_EmitterIdx                        = 0;
    m_EmitterSetCreateID                = 0;
    m_FreeEmitterNum                    = m_EmitterNum;
    m_DrawViewFlag[EFT_CPU_CORE_0]      = EFT_DRAW_VIEW_FLAG_ALL;
    m_DrawViewFlag[EFT_CPU_CORE_1]      = EFT_DRAW_VIEW_FLAG_ALL;
    m_DrawViewFlag[EFT_CPU_CORE_2]      = EFT_DRAW_VIEW_FLAG_ALL;
    m_GlobalCounter                     = 0;
    m_ProcessingInfo.Clear();

    memset( m_EmitterSetHead, NULL, sizeof( EmitterSet* ) * EFT_GROUP_MAX );
    memset( m_EmitterSetTail, NULL, sizeof( EmitterSet* ) * EFT_GROUP_MAX );

    memset( m_DelayCreateEmitterSetHead, NULL, sizeof( EmitterSet* ) * EFT_GROUP_MAX );
    memset( m_DelayCreateEmitterSetTail, NULL, sizeof( EmitterSet* ) * EFT_GROUP_MAX );

    m_DelayKillEmitterSetArray[EFT_CPU_CORE_0]      = NULL;
    m_DelayKillEmitterSetArray[EFT_CPU_CORE_1]      = NULL;
    m_DelayKillEmitterSetArray[EFT_CPU_CORE_2]      = NULL;
    m_DelayKillEmitterSetAddCount[EFT_CPU_CORE_0]   = 0;
    m_DelayKillEmitterSetAddCount[EFT_CPU_CORE_1]   = 0;
    m_DelayKillEmitterSetAddCount[EFT_CPU_CORE_2]   = 0;

    m_DrawPathFlag[EFT_CPU_CORE_0] = 0;
    m_DrawPathFlag[EFT_CPU_CORE_1] = 0;
    m_DrawPathFlag[EFT_CPU_CORE_2] = 0;

    m_CurrentCustomShader[EFT_CPU_CORE_0] = NULL;
    m_CurrentCustomShader[EFT_CPU_CORE_1] = NULL;
    m_CurrentCustomShader[EFT_CPU_CORE_2] = NULL;

    m_CurrentCustomShaderTextureSlot[EFT_CPU_CORE_0] = EFT_CUSTOM_TEXTURE_SLOT_0;
    m_CurrentCustomShaderTextureSlot[EFT_CPU_CORE_1] = EFT_CUSTOM_TEXTURE_SLOT_0;
    m_CurrentCustomShaderTextureSlot[EFT_CPU_CORE_2] = EFT_CUSTOM_TEXTURE_SLOT_0;

    m_ViewParamCopy[EFT_CPU_CORE_0] = NULL;
    m_ViewParamCopy[EFT_CPU_CORE_1] = NULL;
    m_ViewParamCopy[EFT_CPU_CORE_2] = NULL;

    m_CurrentShaderType[EFT_CPU_CORE_0] = EFT_SHADER_TYPE_NORMAL;
    m_CurrentShaderType[EFT_CPU_CORE_1] = EFT_SHADER_TYPE_NORMAL;
    m_CurrentShaderType[EFT_CPU_CORE_2] = EFT_SHADER_TYPE_NORMAL;

    m_EmitterSetInitializeCallback  = NULL;
    m_EmitterSetFinalizeCallback    = NULL;
    m_EmitterDrawProfiler           = NULL;
    m_EmitterCalcLodCallback        = NULL;
    m_EmitterDrawCullingCallback    = NULL;
    m_CustomFieldCallback           = NULL;
    m_TextureInitializer            = NULL;
    m_TextureFinalizer              = NULL;

    for ( u32 i = 0; i < EFT_CUSTOM_SHADER_UBO_MAX; i++ )
    {
        m_CommonCstmShaderUniformBlock[i]       = NULL;
        m_CommonCstmShaderUniformBlockSize[i]   = 0;
    }

    //---------------------------------------------------------
    // 各種ワーク生成
    //---------------------------------------------------------
    //----------------------------------
    // リソースワーク生成
    {
        m_ResourceWorkSize  = sizeof( Resource* ) * ( m_ResourceNum );
        m_ResourceArray = static_cast<Resource **>( AllocFromStaticHeap( m_ResourceWorkSize ) );
        memset( m_ResourceArray, 0, m_ResourceWorkSize );
    }

    //----------------------------------
    // エミッタセットワーク生成
    {
        m_EmitterSetWorkSize = sizeof( EmitterSet ) * m_EmitterSetNum;
        m_EmitterSetArray = static_cast<EmitterSet *>( AllocFromStaticHeap( m_EmitterSetWorkSize ) );
        memset( m_EmitterSetArray, 0, m_EmitterSetWorkSize );

        for( u32 i = 0; i < m_EmitterSetNum; i++ )
        {
            m_EmitterSetArray[i].m_System  = this;
            m_EmitterSetArray[i].m_IsUsage = false;
        }

        // 遅延エミッタセット削除配列を確保
        u32 delayWorkSize = sizeof( EmitterSet* ) * m_EmitterSetNum;
        m_DelayKillEmitterSetArray[EFT_CPU_CORE_0] = static_cast<EmitterSet **>( AllocFromStaticHeap( delayWorkSize ) );
        m_DelayKillEmitterSetArray[EFT_CPU_CORE_1] = static_cast<EmitterSet **>( AllocFromStaticHeap( delayWorkSize ) );
        m_DelayKillEmitterSetArray[EFT_CPU_CORE_2] = static_cast<EmitterSet **>( AllocFromStaticHeap( delayWorkSize ) );
        m_EmitterSetWorkSize += delayWorkSize*3;
    }

    //----------------------------------
    // ダイナミックUBOワーク生成
    u32 numBlock = ( GetBufferMode() == EFT_BUFFER_MODE_TRIPLE ) ? 3 : 2;
    u32 uboBlockSize = sizeof(EmitterDynamicUniformBlock);
    uboBlockSize = nw::eft2::RoundUp<u32>( uboBlockSize, 0x100 );
    {
        m_EmitterUboWorkSize = uboBlockSize * numBlock * m_EmitterNum;
        m_DynamicUBOArray = static_cast<EmitterDynamicUniformBlock *>( AllocFromStaticHeap( m_EmitterUboWorkSize, 0x100 ) );
        memset( m_DynamicUBOArray, 0, m_EmitterUboWorkSize );
    }

    //----------------------------------
    // エミッタワーク生成
    {
        m_EmitterWorkSize = sizeof( Emitter ) * m_EmitterNum;
        m_EmitterArray = static_cast<Emitter*>( AllocFromStaticHeap( m_EmitterWorkSize ) );
        memset( m_EmitterArray, 0, m_EmitterWorkSize );

        u8* ptr = reinterpret_cast<u8 *>( m_DynamicUBOArray );
        for( u32 i = 0; i < m_EmitterNum; i++ )
        {
            m_EmitterArray[i].emitterCalc = NULL;
            for ( u32 j = 0; j < numBlock; j++ )
            {
                m_EmitterArray[i].dynamicUBO[j].Set( reinterpret_cast<EmitterDynamicUniformBlock *>( ptr ),
                                                     sizeof(EmitterDynamicUniformBlock) );
                ptr += uboBlockSize;
            }
        }
    }

    //----------------------------------
    // エミッタ挙動計算クラス
    m_EmitterCalcWorkSize = sizeof(EmitterCalc);
    void* ptr = AllocFromStaticHeap( m_EmitterCalcWorkSize );
    m_EmitterCalc = (EmitterCalc *) new ( ptr ) EmitterCalc( this );

    //----------------------------------
    // 乱数初期化
    {
        u32 temp = GetAllocedSizeFromStaticHeap();
        Random::Initialize();
        m_RandomWorkSize = GetAllocedSizeFromStaticHeap()-temp;
    }

    //----------------------------------
    // カールノイズ初期化
    InitializeCurlNoise();

    //----------------------------------
    // テンポラリバッファ初期化
    bool isTriple = false;
    if( GetBufferMode() == EFT_BUFFER_MODE_TRIPLE )
    {
        isTriple = true;
    }

    if ( config.IsUseMultiCoreProcess() )
    {
        m_DrawTempBuffer[EFT_CPU_CORE_0].Initialize( config.GetTemporaryBufferSize(), isTriple );
        m_DrawTempBuffer[EFT_CPU_CORE_1].Initialize( config.GetTemporaryBufferSize(), isTriple );
        m_DrawTempBuffer[EFT_CPU_CORE_2].Initialize( config.GetTemporaryBufferSize(), isTriple );
    }
    else
    {
        m_DrawTempBuffer[EFT_CPU_CORE_1].Initialize( config.GetTemporaryBufferSize(), isTriple );
        m_DrawTempBuffer[EFT_CPU_CORE_0].Invalidate();
        m_DrawTempBuffer[EFT_CPU_CORE_2].Invalidate();
    }

    //----------------------------------
    // パーティクルソートバッファの初期化
    m_ParticleSortBufferCount = config.GetParticleSortBufferCount();
    for ( u32 core = 0; core < EFT_CPU_CORE_MAX; core++ )
    {
        m_ParticleSortBuffer[core] = reinterpret_cast<ParticleSortInfo*>( AllocFromStaticHeap( m_ParticleSortBufferCount * sizeof(ParticleSortInfo) ) );
    }

    //----------------------------------
    // エミッタセットソートバッファの初期化
    m_EmitterSetSortWorkSize = 0;
    for ( u32 core = 0; core < EFT_CPU_CORE_MAX; core++ )
    {
        m_SortEmittetSet[core]       = (sortEmitterSets *)AllocFromStaticHeap( sizeof( sortEmitterSets ) * m_EmitterSetNum );
        m_AddedSortBufferNum[core]   = 0;
        m_EmitterSetSortWorkSize     += sizeof( sortEmitterSets ) * m_EmitterSetNum;
    }

    //----------------------------------
    // コールバックの初期化
    for ( u32 i = 0; i < EFT_CALLBACK_ID_MAX; i++ )
    {
        m_IsEnableCallback[i]    = false;

        // 現状はエンディアン反転コールバックを強制的に付加。
        m_Callback[i].endianFlip = _EndianFlipCallback;

        // 描画設定後コールバックを強制的に付加
        m_Callback[i].renderStateSet = _BindReservedCustomShaderUniformBlock;
    }

#ifndef EFT_DEGRADATION_SPEC
    //----------------------------------
    // エミッタプラグインコールバックの初期化
    // ストライプ初期化
    StripeSystem::Initialize( m_StaticHeap, this, m_BufferMode, config.GetStripeNum() );
    ConnectionStripeSystem::Initialize( m_StaticHeap, this, m_BufferMode, config.GetStripeNum() );
    SuperStripeSystem::Initialize( m_StaticHeap, this, m_BufferMode, config.GetSuperStripeNum() );

    // 範囲内ループ初期化
    AreaLoopSystem::Initialize( this );
#endif

    //----------------------------------
    // 描画パスフラグの初期化
    MemUtil::FillZero( m_EnableDrawPath[EFT_CPU_CORE_0], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_EnableDrawPath[EFT_CPU_CORE_1], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_EnableDrawPath[EFT_CPU_CORE_2], EFT_GROUP_MAX * sizeof( u32 ) );

    MemUtil::FillZero( m_RequestFrameBufferTexturePath[EFT_CPU_CORE_0], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_RequestFrameBufferTexturePath[EFT_CPU_CORE_1], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_RequestFrameBufferTexturePath[EFT_CPU_CORE_2], EFT_GROUP_MAX * sizeof( u32 ) );

    MemUtil::FillZero( m_RequestDepthBufferTexturePath[EFT_CPU_CORE_0], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_RequestDepthBufferTexturePath[EFT_CPU_CORE_1], EFT_GROUP_MAX * sizeof( u32 ) );
    MemUtil::FillZero( m_RequestDepthBufferTexturePath[EFT_CPU_CORE_2], EFT_GROUP_MAX * sizeof( u32 ) );

    // 描画パスコールバック初期化
    for ( u32 i = 0; i < EFT_DRAW_PATH_CALLBACK_MAX; ++ i )
    {
        m_DrawPathCallbackFlag[i]                    = 0;
        m_DrawPathRenderStateSetCallback[i]          = NULL;
    }

    //----------------------------------
    // ビューユニフォームブロックの初期化
#ifdef EFT_USE_UNIFORM_BLOCK
    Shader::InitializeUniformBlock();
#endif

    // 同期オブジェクト初期化
    m_Mutex.Initialize();

    // メモリの使用状況をダンプします。
    OutputLog("System Static    WorkSize : %d \n", GetAllocedSizeFromStaticHeap() );
    OutputLog("  DelayFree      WorkSize : %d \n", m_DelayFreeWorkSize );
    OutputLog("  Random         WorkSize : %d \n", m_RandomWorkSize );
    OutputLog("  Resource       WorkSize : %d \n", m_ResourceWorkSize );
    OutputLog("  EmitterSet     WorkSize : %d ( EmitterSetNum:%d )\n", m_EmitterSetWorkSize, m_EmitterSetNum );
    OutputLog("  Emitter        WorkSize : %d ( EmitterNum:%d )\n", m_EmitterWorkSize, m_EmitterNum );
    OutputLog("  EmitterCalc    WorkSize : %d \n", m_EmitterCalcWorkSize );
    OutputLog("  EmitterUbo     WorkSize : %d \n", m_EmitterUboWorkSize );
    OutputLog("  EmitterSetSort WorkSize : %d \n", m_EmitterSetSortWorkSize );
#ifndef EFT_DEGRADATION_SPEC
    OutputLog("  NoiseTexture   WorkSize : %d \n", GetCurlNoiseTextureAllocedSize() );
#endif
    u32 tempBufferSize = m_DrawTempBuffer[EFT_CPU_CORE_0].GetSize() + m_DrawTempBuffer[EFT_CPU_CORE_1].GetSize() + m_DrawTempBuffer[EFT_CPU_CORE_2].GetSize();
    tempBufferSize *= 2;
    OutputLog("  TempBuffer     WorkSize : %d \n", tempBufferSize );
    OutputLog("-------------------------------\n" );
    //u32 total =  m_ResourceWorkSize+m_EmitterSetWorkSize+m_EmitterWorkSize+m_EmitterCalcWorkSize+
    //                m_EmitterUboWorkSize+m_EmitterSetSortWorkSize+GetCurlNoiseTextureAllocedSize()+tempBufferSize+m_DelayFreeWorkSize+m_RandomWorkSize;
    //OutputLog("Total WorkSize : %d \n", total );
    OutputLog("-------------------------------\n" );
    // ES2.0対応が済むまでは、エミッタプラグインを無効
#ifndef EFT_DEGRADATION_SPEC
    OutputLog("Stripe           WorkSize : %d \n", StripeSystem::GetWorkSize(), config.GetStripeNum() );
    OutputLog("SuperStripe      WorkSize : %d \n", SuperStripeSystem::GetWorkSize() );
    OutputLog("ConnectStripe    WorkSize : %d \n", ConnectionStripeSystem::GetWorkSize() );
    OutputLog("-------------------------------\n" );
#endif
}

//------------------------------------------------------------------------------
//  エフェクトシステムの解放を行います。
//------------------------------------------------------------------------------
System::~System()
{
    // 同期オブジェクト初期化
    m_Mutex.Finalize();

    //----------------------------------
    // ビューユニフォームブロックの初期化
#ifdef EFT_USE_UNIFORM_BLOCK
    Shader::FinalizeUniformBlock();
#endif

    for( u32 i = 0; i < m_EmitterSetNum; i++ )
    {
        m_EmitterSetArray[i].Finalize();
    }
    for( u32 i = 0; i < m_ResourceNum; i++ )
    {
        if ( m_ResourceArray[i] )
        {
            m_ResourceArray[i]->Finalize( NULL );
        }
    }

    if ( m_DynamicUBOArray ) FreeFromStaticHeap( m_DynamicUBOArray );
    if ( m_EmitterArray )    FreeFromStaticHeap( m_EmitterArray );
    if ( m_EmitterSetArray ) FreeFromStaticHeap( m_EmitterSetArray );

    if ( m_DelayKillEmitterSetArray[EFT_CPU_CORE_0] ) FreeFromStaticHeap( m_DelayKillEmitterSetArray[EFT_CPU_CORE_0] );
    if ( m_DelayKillEmitterSetArray[EFT_CPU_CORE_1] ) FreeFromStaticHeap( m_DelayKillEmitterSetArray[EFT_CPU_CORE_1] );
    if ( m_DelayKillEmitterSetArray[EFT_CPU_CORE_2] ) FreeFromStaticHeap( m_DelayKillEmitterSetArray[EFT_CPU_CORE_2] );

    if ( m_ResourceArray )        FreeFromStaticHeap( m_ResourceArray );
    if ( m_EmitterCalc )
    {
        m_EmitterCalc->~EmitterCalc();
        FreeFromStaticHeap( m_EmitterCalc );
    }

    m_DrawTempBuffer[EFT_CPU_CORE_0].Finalize();
    m_DrawTempBuffer[EFT_CPU_CORE_1].Finalize();
    m_DrawTempBuffer[EFT_CPU_CORE_2].Finalize();

    FreeFromStaticHeap( m_SortEmittetSet[EFT_CPU_CORE_0] );
    FreeFromStaticHeap( m_SortEmittetSet[EFT_CPU_CORE_1] );
    FreeFromStaticHeap( m_SortEmittetSet[EFT_CPU_CORE_2] );

    FreeFromStaticHeap( m_ParticleSortBuffer[EFT_CPU_CORE_0] );
    FreeFromStaticHeap( m_ParticleSortBuffer[EFT_CPU_CORE_1] );
    FreeFromStaticHeap( m_ParticleSortBuffer[EFT_CPU_CORE_2] );

    // カールノイズ終了処理
    FinalizeCurlNoise();

    // ランダム終了処理
    Random::Finalize();
    FinalizeDelayFreeList();

    SetStaticHeap( NULL );
    SetDynamicHeap( NULL );

    // ES2.0対応が済むまでは、エミッタプラグインを無効
#ifndef EFT_DEGRADATION_SPEC
    // ストライプ終了処理
    StripeSystem::Finalize( m_StaticHeap );
    SuperStripeSystem::Finalize( m_StaticHeap );
    ConnectionStripeSystem::Finalize( m_StaticHeap );
#endif
}

//------------------------------------------------------------------------------
//  リソースを登録する。
//------------------------------------------------------------------------------
bool System::EntryResource( Heap* heap, void* bin, u32 resId, bool delayCompile, Resource* residentResource )
{
    EFT_ASSERT_MSG( heap, "[EFT] heap is null." );
    EFT_ASSERT_MSG( bin,  "[EFT] bin is null." );
    EFT_ASSERT_MSG( static_cast<bool>( resId < m_ResourceNum ), "[EFT] resId is over." );

    if( m_ResourceArray[ resId ] != NULL )
    {
        ClearResource( heap, resId );
//      ClearResource( NULL, resId );
        OutputWarning( "resId %d is cleared.", resId );
    }

    void* ptr = heap->Alloc( sizeof(Resource) );
    if ( !ptr ) return false;
    m_ResourceArray[ resId ] = new (ptr) Resource( heap, bin, resId, this, delayCompile, residentResource );

    return true;
}

//---------------------------------------------------------------------------
//  指定IDのリソースをエフェクトシステムから破棄します。
//---------------------------------------------------------------------------
void System::ClearResource( Heap* heap, u32 resId )
{
    EFT_NULL_ASSERT( heap );
    EFT_ASSERT_MSG( ( resId < m_ResourceNum ), "[EFT] resId is over." );
    if ( !m_ResourceArray[ resId ] )
    {
        OutputWarning( "no resource to delete." );
        return;
    }

    if ( heap )
    {
        m_ResourceArray[ resId ]->Finalize( heap );
        heap->Free( m_ResourceArray[ resId ] );
    }

    m_ResourceArray[ resId ] = NULL;
}

#if EFT_GX2
//---------------------------------------------------------------------------
//  シェーダバイナリを配置前に戻す。
//---------------------------------------------------------------------------
void System::Unrelocate( void* pData )
{
    EFT_NULL_ASSERT( pData );

    nw::eft2::Resource::Unrelocate( pData );
}
#endif

//---------------------------------------------------------------------------
//  エミッタセットを確保します。
//---------------------------------------------------------------------------
EmitterSet* System::AllocEmitterSet()
{
    EmitterSet* emSet = NULL;
    u32         i     = 0;

    do
    {
        m_EmitterSetIdx++;
        if ( m_EmitterSetIdx >= m_EmitterSetNum ) m_EmitterSetIdx = 0;
        if( !m_EmitterSetArray[ m_EmitterSetIdx ].m_IsUsage )
        {
            emSet = &m_EmitterSetArray[ m_EmitterSetIdx ];
            break;
        }
    }
    while( ( ++i ) < m_EmitterSetNum );

    if( emSet == NULL ) { OutputWarning( "Emitter Set is Empty.\n" ); }

    return emSet;
}

//---------------------------------------------------------------------------
//  エミッタを確保します。
//---------------------------------------------------------------------------
Emitter* System::AllocEmitter()
{
    Emitter* emitter = NULL;
    u32 i = 0;

    do
    {
        m_EmitterIdx++;
        if ( m_EmitterIdx >= m_EmitterNum ) m_EmitterIdx = 0;
        if( !m_EmitterArray[ m_EmitterIdx ].IsAlive() )
        {
            emitter = &m_EmitterArray[ m_EmitterIdx ];
            break;
        }
    }
    while( (++i) < m_EmitterNum );

    if( emitter == NULL )
    {
        OutputWarning( "Emitter Alloced Failed.\n" );
        return NULL;
    }

    m_FreeEmitterNum--;

    return emitter;
}

//---------------------------------------------------------------------------
//  エミッタを初期化します。
//---------------------------------------------------------------------------
void System::InitializeEmitter( Emitter* emitter )
{
    EFT_NULL_ASSERT( emitter );
    emitter->emitterCalc = m_EmitterCalc;
}


//---------------------------------------------------------------------------
//! @brief      エミッタの終了処理をします。
//---------------------------------------------------------------------------
void System::FinalizeEmitter( Emitter* emitter )
{
    EFT_NULL_ASSERT( emitter );

    // dynamicUBOアドレスを一時的に退避する。
    DynamicUniformBlockBuffer tempDynamicUBO[EFT_BUFFER_ID_TRIPLE_MAX];
    tempDynamicUBO[EFT_BUFFER_ID_0].Set( emitter->dynamicUBO[EFT_BUFFER_ID_0].GetBuffer(),
                                         emitter->dynamicUBO[EFT_BUFFER_ID_0].GetBufferSize() );
    tempDynamicUBO[EFT_BUFFER_ID_1].Set( emitter->dynamicUBO[EFT_BUFFER_ID_1].GetBuffer(),
                                         emitter->dynamicUBO[EFT_BUFFER_ID_1].GetBufferSize() );
    tempDynamicUBO[EFT_BUFFER_ID_2].Set( emitter->dynamicUBO[EFT_BUFFER_ID_2].GetBuffer(),
                                         emitter->dynamicUBO[EFT_BUFFER_ID_2].GetBufferSize() );

    // emitterをクリア
    MemUtil::FillZero( emitter, sizeof( Emitter ) );

    // dynamicUBOアドレスを戻す
    emitter->dynamicUBO[EFT_BUFFER_ID_0].Set( tempDynamicUBO[EFT_BUFFER_ID_0].GetBuffer(),
                                              tempDynamicUBO[EFT_BUFFER_ID_0].GetBufferSize() );
    emitter->dynamicUBO[EFT_BUFFER_ID_1].Set( tempDynamicUBO[EFT_BUFFER_ID_1].GetBuffer(),
                                              tempDynamicUBO[EFT_BUFFER_ID_1].GetBufferSize() );
    emitter->dynamicUBO[EFT_BUFFER_ID_2].Set( tempDynamicUBO[EFT_BUFFER_ID_2].GetBuffer(),
                                              tempDynamicUBO[EFT_BUFFER_ID_2].GetBufferSize() );
    m_FreeEmitterNum++;
}


//---------------------------------------------------------------------------
//  指定IDのエミッタセットを放出します。
//---------------------------------------------------------------------------
bool System::CreateEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap, bool delay )
{
    EFT_NULL_ASSERT( handle );
    Resource* res = GetResource( resourceID );
    if ( !res ) return false;

    // 生成するエミッタ数
    u32 emitterNum = res->GetEmitterNum( emitterSetID );

    // エミッタ数チェック
    if( emitterNum > m_FreeEmitterNum )
    {
        handle->Invalidate();
        OutputWarning( "Emitter is Empty.\n" );
        return false;
    }

    // エミッタセット確保
    EmitterSet* set = AllocEmitterSet();
    if ( !set ) return false;

    handle->m_EmitterSet = set;
    handle->m_CreateID   = m_EmitterSetCreateID++;

    // エミッタセットの初期化
    if ( !set->Initialize( emitterSetID, handle->m_CreateID, resourceID, groupID, particleMax, heap ) )
    {
        set->Finalize();
        return false;
    }

    if ( !delay )
    {
        // 管理リストへ追加
        AddEmitterSetList( set, groupID );
    }
    else
    {
        // 遅延管理リストへ追加
        AddDelayCreateEmitterSetList( set, groupID );

        // 遅延生成フラグを立てる
        set->m_IsDelayCreate = true;
    }

    return true;
}

//---------------------------------------------------------------------------
//  指定IDのエミッタセットを生成します。
//---------------------------------------------------------------------------
bool System::CreateEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, Heap* heap, bool delay )
{
    return CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, 0, heap, delay );
}

//---------------------------------------------------------------------------
//  指定IDのエミッタセットをマニュアル放出状態で生成します。
//---------------------------------------------------------------------------
bool System::CreateManualEmitEmitterSetID( Handle* handle, s32 emitterSetID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap, int residentEmitterTime )
{
    bool ret = false;

    // チャイルド付きエミッタセットはサポートしない
    Resource* res = GetResource( resourceID );
    if ( !res ) return false;
    if ( res->IsExistChildEmitter( emitterSetID ) )
    {
        OutputWarning( "Manual EmitterSet isn't Allow Child Emitter.\n" );
        return false;
    }


    ret = CreateEmitterSetID( handle, emitterSetID, resourceID, groupID, particleMax, heap );

    // マニュアル放出モードへ移行
    if ( handle->IsValid() )
    {
        handle->GetEmitterSet()->SetManualParticleEmission();
        handle->GetEmitterSet()->m_ResidentEmitterTime = residentEmitterTime;
    }

    return ret;
}

//---------------------------------------------------------------------------
//  指定エミッタセットを再生成します。
//---------------------------------------------------------------------------
bool System::ReCreateEmitterSet( u32 resourceID, s32 emitterSetID )
{
    bool ret = false;

    // 指定エミッタセットを検索します。
    for( u32 groupID = 0; groupID < EFT_GROUP_MAX; groupID++ )
    {
        if ( !m_EmitterSetHead[groupID] ) continue;
        EmitterSet* emitterSet = m_EmitterSetHead[groupID];

        while( emitterSet )
        {
            EmitterSet* next = emitterSet->m_Next;

            // 同IDのエミッタセットがある場合はResetする。
            if ( emitterSet->GetResourceID()   == resourceID &&
                 emitterSet->GetEmitterSetID() == emitterSetID )
            {
                if ( emitterSet->IsFadeRequest() )
                {
                    emitterSet->Kill();
                }
                else
                {
                    emitterSet->Reset();
                    ret = true;
                }
            }

            emitterSet = next;
        }
    }

    return ret;
}

//---------------------------------------------------------------------------
//  引数名のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::RecreateEmitterSet2( const char* emitterSetName, u32 oldResId, u32 newResId )
{
    for( u32 i = 0; i < EFT_GROUP_MAX; i++ )
    {
        EmitterSet* emitterSet = m_EmitterSetHead[ i ];

        while( emitterSet )
        {
            EmitterSet* next = emitterSet->GetNext();

            if ( strcmp( emitterSet->GetName(), emitterSetName ) == 0 && emitterSet->m_ResourceID == oldResId )
            {
                emitterSet->m_ResourceID = newResId;
                emitterSet->Reset();
            }

            emitterSet = next;
        }
    }
}


//---------------------------------------------------------------------------
//  引数のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSet( EmitterSet* emitterSet, bool immediate )
{
    if ( !emitterSet->m_IsUsage )
    {
        OutputWarning( "EmitterSet is already removed.\n" );
        return;
    }

    m_Mutex.Lock();

    // 既に遅延削除リスト登録済みの場合は削除しない
    for( u32 core = 0; core < EFT_CPU_CORE_MAX; core++ )
    {
        if ( m_DelayKillEmitterSetAddCount[core] > 0 )
        {
            for ( u32 i = 0; i < m_DelayKillEmitterSetAddCount[core]; i++ )
            {
                if ( m_DelayKillEmitterSetArray[core][i] == emitterSet ) goto exit;
            }
        }
    }

    if ( immediate )
    {
        emitterSet->Finalize();
        RemoveEmitterSetList( emitterSet );
    }
    else
    {
        CpuCore core = GetCurrentCore();
        EFT_ASSERT( m_DelayKillEmitterSetAddCount[core] < m_EmitterSetNum );
        m_DelayKillEmitterSetArray[ core ][ m_DelayKillEmitterSetAddCount[core] ] = emitterSet;
        m_DelayKillEmitterSetAddCount[core]++;
    }

exit:
    m_Mutex.Unlock();
}


//---------------------------------------------------------------------------
//  引数名のエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSet( const char* emitterSetName, u32 resId )
{
    for( u32 i = 0; i < EFT_GROUP_MAX; i++ )
    {
        EmitterSet* emitterSet = m_EmitterSetHead[ i ];

        while( emitterSet )
        {
            EmitterSet* next = emitterSet->GetNext();

            if ( strcmp( emitterSet->GetName(), emitterSetName ) == 0 &&
                 emitterSet->GetResourceID() == resId )
            {
                KillEmitterSet( emitterSet );
            }

            emitterSet = next;
        }

        if( m_DelayCreateEmitterSetHead[ i ] )
        {
            emitterSet = m_DelayCreateEmitterSetHead[ i ];

            while( emitterSet )
            {
                EmitterSet* next = emitterSet->GetNext();
                if( strcmp( emitterSet->GetName(), emitterSetName ) == 0 && emitterSet->GetResourceID() == resId )
                {
                    RemoveDelayCreateEmitterSetList( emitterSet );
                    KillEmitterSet( emitterSet );
                }
                emitterSet = next;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  指定グループに所属するエミッタを削除します。
//---------------------------------------------------------------------------
void System::KillEmitterSetGroup( u8 groupID, bool immediate )
{
    if ( m_EmitterSetHead[groupID] )
    {
        EmitterSet* emitterSet = m_EmitterSetHead[groupID];
        EmitterSet* next       = NULL;

        while( emitterSet )
        {
            next = emitterSet->GetNext();
            KillEmitterSet( emitterSet, immediate );
            emitterSet = next;
        }
    }

    if ( !immediate ) return;

    if ( m_DelayCreateEmitterSetHead[groupID] )
    {
        EmitterSet* eset = m_DelayCreateEmitterSetHead[groupID];

        while( eset )
        {
            EmitterSet* next = eset->GetNext();
            KillEmitterSet( eset );
            eset->m_IsDelayCreate = false;
            eset = next;
        }
    }
}

//---------------------------------------------------------------------------
//  再生中の全てのエミッタセットを削除します。
//---------------------------------------------------------------------------
void System::KillAllEmitterSet( bool immediate )
{
    m_Mutex.Lock();

    for( u32 i = 0; i < EFT_GROUP_MAX; i++ )
    {
        KillEmitterSetGroup( static_cast<u8>( i ), immediate );
    }

    // ストリームアウトエミッタのリストを空に。
    m_StreamOutEmitter      = NULL;
    m_StreamOutEmitterTail  = NULL;

    m_Mutex.Unlock();
}

//---------------------------------------------------------------------------
//  生成されたエミッタセットを管理リストへ登録します。
//---------------------------------------------------------------------------
void System::AddEmitterSetList( EmitterSet* emitterSet, u8 groupID )
{
    if( m_EmitterSetHead[ groupID ] == NULL )
    {
        m_EmitterSetHead[ groupID ]          = emitterSet;
        emitterSet->m_Prev                   = NULL;
        emitterSet->m_Next                   = NULL;
    }
    else
    {
        m_EmitterSetTail[ groupID ]->m_Next   = emitterSet;
        emitterSet->m_Prev                   = m_EmitterSetTail[ groupID ];
        emitterSet->m_Next                   = NULL;
    }

    m_EmitterSetTail[ groupID ] = emitterSet;
}

//---------------------------------------------------------------------------
//  生成されたエミッタセットを遅延生成管理リストへ登録します。
//---------------------------------------------------------------------------
void System::AddDelayCreateEmitterSetList( EmitterSet* emitterSet, u8 groupID )
{
    if( m_DelayCreateEmitterSetHead[ groupID ] == NULL )
    {
        m_DelayCreateEmitterSetHead[ groupID ]              = emitterSet;
        emitterSet->m_Prev                                  = NULL;
        emitterSet->m_Next                                  = NULL;
    }
    else
    {
        m_DelayCreateEmitterSetTail[ groupID ]->m_Next      = emitterSet;
        emitterSet->m_Prev                                  = m_DelayCreateEmitterSetTail[ groupID ];
        emitterSet->m_Next                                  = NULL;
    }

    m_DelayCreateEmitterSetTail[ groupID ] = emitterSet;
}

//---------------------------------------------------------------------------
//  指定エミッタセットを管理リストから削除します。
//---------------------------------------------------------------------------
void System::RemoveEmitterSetList( EmitterSet* emitterSet )
{
    if ( emitterSet->m_Next == NULL && emitterSet->m_Prev == NULL && emitterSet->m_IsDelayCreate )
    {
        // 遅延生成リストから削除
        RemoveDelayCreateEmitterSetList( emitterSet );
        return;
    }

    if( emitterSet == m_EmitterSetHead[ emitterSet->m_GroupID ] )
    {
        m_EmitterSetHead[ emitterSet->m_GroupID ] = emitterSet->m_Next;

        if( m_EmitterSetHead[ emitterSet->m_GroupID ] != NULL )
        {
            m_EmitterSetHead[ emitterSet->m_GroupID ]->m_Prev = NULL;
        }

        if ( emitterSet == m_EmitterSetTail[ emitterSet->m_GroupID ] )
        {
            m_EmitterSetTail[ emitterSet->m_GroupID ] = NULL;
        }
    }
    else
    {
        if ( emitterSet == m_EmitterSetTail[ emitterSet->m_GroupID ] )
        {
            m_EmitterSetTail[ emitterSet->m_GroupID ] = emitterSet->m_Prev;
        }

        if( emitterSet->m_Next != NULL )
        {
            emitterSet->m_Next->m_Prev = emitterSet->m_Prev;
        }

        if( emitterSet->m_Prev != NULL )
        {
            emitterSet->m_Prev->m_Next    = emitterSet->m_Next;
        }
        else
        {
            OutputError( "EmitterSet Remove Failed.\n" );
        }
    }

    emitterSet->m_Next = NULL;
    emitterSet->m_Prev = NULL;
}

//---------------------------------------------------------------------------
//  指定エミッタセットを遅延生成管理リストから削除します。
//---------------------------------------------------------------------------
void System::RemoveDelayCreateEmitterSetList( EmitterSet* emitterSet )
{
    if( emitterSet == m_DelayCreateEmitterSetHead[ emitterSet->m_GroupID ] )
    {
        m_DelayCreateEmitterSetHead[ emitterSet->m_GroupID ] = emitterSet->m_Next;

        if( m_DelayCreateEmitterSetHead[ emitterSet->m_GroupID ] != NULL )
        {
            m_DelayCreateEmitterSetHead[ emitterSet->m_GroupID ]->m_Prev = NULL;
        }

        if ( emitterSet == m_DelayCreateEmitterSetTail[ emitterSet->m_GroupID ] )
        {
            m_DelayCreateEmitterSetTail[ emitterSet->m_GroupID ] = NULL;
        }
    }
    else
    {
        if ( emitterSet == m_DelayCreateEmitterSetTail[ emitterSet->m_GroupID ] )
        {
            m_DelayCreateEmitterSetTail[ emitterSet->m_GroupID ] = emitterSet->m_Prev;
        }

        if( emitterSet->m_Next != NULL )
        {
            emitterSet->m_Next->m_Prev = emitterSet->m_Prev;
        }

        if( emitterSet->m_Prev != NULL )
        {
            emitterSet->m_Prev->m_Next    = emitterSet->m_Next;
        }
        else
        {
            OutputError( "EmitterSet Remove Failed.\n" );
        }
    }

    emitterSet->m_IsDelayCreate = false;
    emitterSet->m_Next          = NULL;
    emitterSet->m_Prev          = NULL;
}


//---------------------------------------------------------------------------
//  内部で確保されたメモリの遅延解放やカウンタ等のリセットを行います。
//---------------------------------------------------------------------------
void System::BeginFrame()
{
    if ( !m_IsEnableCalcProcess ) return;

    m_GlobalCounter++;

    m_ProcessingInfo.Clear();
    m_ProcessingGroupID     = 0;
    MemUtil::FillZero( m_EnableDrawPath, EFT_GROUP_MAX * sizeof( u32 ) * EFT_CPU_CORE_MAX );
    MemUtil::FillZero( m_RequestFrameBufferTexturePath, EFT_GROUP_MAX * sizeof( u32 ) * EFT_CPU_CORE_MAX );
    MemUtil::FillZero( m_RequestDepthBufferTexturePath, EFT_GROUP_MAX * sizeof( u32 ) * EFT_CPU_CORE_MAX );

    m_StreamOutEmitter      = NULL;
    m_StreamOutEmitterTail  = NULL;

    // 遅延解放されるワークの解放を行います。
    FlushDelayFreeList();

    m_Mutex.Lock();

    // 遅延削除指定されたエミッタセットを削除
    for( u32 core = 0; core < EFT_CPU_CORE_MAX; core++ )
    {
        if ( m_DelayKillEmitterSetAddCount[core] > 0 )
        {
            for ( u32 i = 0; i < m_DelayKillEmitterSetAddCount[core]; i++ )
            {
                EFT_NULL_ASSERT( m_DelayKillEmitterSetArray[core][i] );
                m_DelayKillEmitterSetArray[core][i]->Finalize();
                RemoveEmitterSetList( m_DelayKillEmitterSetArray[core][i] );
                m_DelayKillEmitterSetArray[core][i] = NULL;
            }
        }
        m_DelayKillEmitterSetAddCount[ core ] = 0;
    }

    m_Mutex.Unlock();

    // 遅延生成されたエミッタセットをリストに追加
    bool doClear = false;
    for ( u8 group = 0; group < EFT_GROUP_MAX; group++ )
    {
        if ( m_DelayCreateEmitterSetHead[group] )
        {
            EmitterSet* eset = m_DelayCreateEmitterSetHead[group];

            while( eset )
            {
                EmitterSet* next = eset->GetNext();
                AddEmitterSetList( eset, group );
                eset->m_IsDelayCreate = false;
                eset = next;
            }

            doClear = true;
        }
    }

    if ( doClear )
    {
        MemUtil::FillZero( m_DelayCreateEmitterSetHead, sizeof( EmitterSet* ) * EFT_GROUP_MAX );
        MemUtil::FillZero( m_DelayCreateEmitterSetTail, sizeof( EmitterSet* ) * EFT_GROUP_MAX );
    }
}

//---------------------------------------------------------------------------
//  計算処理を行います。
//---------------------------------------------------------------------------
void System::Calc( u8 groupID, f32 frameRate, bool swapUbo )
{
    if ( !m_IsEnableCalcProcess ) return;

    if ( !m_EmitterSetHead[groupID] ) return;

    if ( frameRate != 0.0f )
    {
        m_ProcessingGroupID |= (u64)( (u64)0x1 << (u64)( groupID) );
    }

    EmitterSet* emitterSet = m_EmitterSetHead[groupID];
    while( emitterSet )
    {
        if ( emitterSet->IsCalc() )
        {
            emitterSet->Calc( frameRate, swapUbo, m_EmitterCalcLodCallback );
        }
        else
        {
            emitterSet->Calc( 0.0f, swapUbo, m_EmitterCalcLodCallback );
        }

        if ( emitterSet->IsAlive() )
        {
            m_ProcessingInfo.emitterSetNum++;
            m_ProcessingInfo.emitterNum             += emitterSet->GetProcessingEmitterCount();
            m_ProcessingInfo.calcSkipEmitterNum     += emitterSet->GetCalcSkipEmitterCount();
            m_ProcessingInfo.cpuParticleNum         += emitterSet->GetProcessingCpuParticleCount();
            m_ProcessingInfo.gpuParticleNum         += emitterSet->GetProcessingGpuParticleCount();
            m_ProcessingInfo.gpusoParticleNum       += emitterSet->GetProcessingGpuSoParticleCount();
            m_ProcessingInfo.emitterAnimNum         += emitterSet->GetProcessingEmitterAnimNum();
            m_ProcessingInfo.cpuEmitterNum          += emitterSet->GetProcessingCpuEmitterCount();
            m_ProcessingInfo.gpuEmitterNum          += emitterSet->GetProcessingGpuEmitterCount();
            m_ProcessingInfo.gpusoEmitterNum        += emitterSet->GetProcessingGpuSoEmitterCount();
            m_ProcessingInfo.stripeNum              += emitterSet->GetProcessingStripeCount();
            m_ProcessingInfo.superStripeNum         += emitterSet->GetProcessingSuperStripeCount();
            m_ProcessingInfo.allocedDynamicHeapSize += emitterSet->GetAllocedFromDynamicHeapSize();

            m_EnableDrawPath[GetCurrentCore()][groupID] |= emitterSet->GetDrawPath();
            m_RequestFrameBufferTexturePath[GetCurrentCore()][groupID] |= emitterSet->GetRequestFrameBufferTextureDrawPath();
            m_RequestDepthBufferTexturePath[GetCurrentCore()][groupID] |= emitterSet->GetRequestDepthBufferTextureDrawPath();

            emitterSet = emitterSet->GetNext();
        }
        else
        {
            EmitterSet* next = emitterSet->GetNext();
            if( emitterSet->m_IsUsage )
            {
                KillEmitterSet( emitterSet );
            }
            else
            {
                // MEMO: 中途半端に消されてしまったエミッタ。ハンドルを確実に無効にするために暫定で CreateId を無効化する。
                OutputWarning("Invalid EmitterSet has been detected. Invalidates handle.");
                emitterSet->m_EmitterSetCreateID = 0xFFFFFFFF;
            }
            emitterSet = next;
        }
    }
}

//---------------------------------------------------------------------------
//  GPUキャッシュのフラッシュを行います。
//---------------------------------------------------------------------------
void System::FlushGpuCache()
{
#if EFT_IS_CAFE
    GX2Invalidate(static_cast<GX2InvalidateType>(
        GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE |
        GX2_INVALIDATE_UNIFORM_BLOCK | GX2_INVALIDATE_SHADER ),
        NULL, 0xFFFFFFFF);
#endif
}


//---------------------------------------------------------------------------
//  ビュー情報の設定を行います。
//---------------------------------------------------------------------------
void System::SetViewParam( ViewParam* viewParam )
{
    CpuCore core = GetCurrentCore();
    m_ViewParam[core] = *viewParam;

#ifdef EFT_USE_UNIFORM_BLOCK
    TemporaryUniformBlockBuffer viewTempUbo;
    m_ViewParamCopy[core] = reinterpret_cast<ViewParam*>( viewTempUbo.Alloc( this, sizeof( ViewParam ) ) );
    if ( !m_ViewParamCopy[core] ) return;
    MemUtil::Copy( m_ViewParamCopy[core], viewParam, sizeof( ViewParam ) );
    viewTempUbo.Validate();

    // ビューのUBOを設定する
    Shader::BindViewUniformBlock( &viewTempUbo );

    for ( u32 i = 0; i < EFT_CUSTOM_SHADER_UBO_MAX; i++ )
    {
        if ( m_CommonCstmShaderUniformBlock[i] )
        {
            void* tempBuffer = this->AllocFromTempBuffer( m_CommonCstmShaderUniformBlockSize[i] );
            MemUtil::Copy( tempBuffer, m_CommonCstmShaderUniformBlock[i], m_CommonCstmShaderUniformBlockSize[i] );
            Shader::BindCommonCustomShaderUniformBlock(
                static_cast<CustomShaderUboID>( i ), tempBuffer, m_CommonCstmShaderUniformBlockSize[i], true );
        }
    }
#endif

#ifdef EFT_USE_UNIFORM_REGISTER
    m_ViewParamCopy[core] = reinterpret_cast<ViewParam*>( AllocFromTempBuffer( sizeof( ViewParam ) ) );
    if ( !m_ViewParamCopy[core] ) return;
    MemUtil::Copy( m_ViewParamCopy[core], viewParam, sizeof( ViewParam ) );
#endif

}


//---------------------------------------------------------------------------
//  描画処理を行います。
//---------------------------------------------------------------------------
void System::Draw( u8 groupID, u32 drawPathFlag, bool sort, bool calcStreamOut, void* userParam )
{
    if ( !m_EmitterSetHead[groupID] ) return;

    EmitterSet* emitterSet = m_EmitterSetHead[groupID];
    CpuCore core = GetCurrentCore();

    if ( !sort )
    {
        while( emitterSet )
        {
            // 描画ビューフラグ判定
            if ( emitterSet->GetDrawViewFlag() & m_DrawViewFlag[core] && emitterSet->GetDrawPath() & drawPathFlag )
            {
                if ( emitterSet->IsVisible() )
                {
                    emitterSet->Draw( m_CurrentShaderType[GetCurrentCore()], drawPathFlag, m_DrawViewFlag[core], calcStreamOut, userParam,
                                      m_EmitterDrawCullingCallback, m_EmitterDrawProfiler );
                }
            }
            emitterSet = emitterSet->GetNext();
        }
    }
    else
    {
        AddSortBuffer( groupID, drawPathFlag );
        DrawSortBuffer( calcStreamOut, userParam );
    }
}


//------------------------------------------------------------------------------
//  描画を行うIDグループをソートバッファに追加します。
//------------------------------------------------------------------------------
void System::AddSortBuffer( u8 groupID, u32 drawPathFlag )
{
    EmitterSet* emitterSet = m_EmitterSetHead[groupID];
    if ( !emitterSet ) return;

    CpuCore core = GetCurrentCore();

    nw::math::MTX44* viewMat = &m_ViewParam[core].viewMat;

    while( emitterSet )
    {
        // 描画ビューフラグ判定
        if ( emitterSet->GetDrawViewFlag() & m_DrawViewFlag[core] && emitterSet->GetDrawPath() & drawPathFlag )
        {
            if  ( emitterSet->IsVisible() )
            {
                nw::math::VEC3 epos;
                emitterSet->GetPos( epos );

                m_SortEmittetSet[core][m_AddedSortBufferNum[core]].emitterSet = emitterSet;
                f32 viewZ = viewMat->f._20 * epos.x + viewMat->f._21 * epos.y + viewMat->f._22 * epos.z + viewMat->f._23 * 1.0f;
                m_SortEmittetSet[core][m_AddedSortBufferNum[core]].z =  ( (( emitterSet->GetDrawPriority())<<24) | Float32ToBits24( viewZ ) );
                m_AddedSortBufferNum[core]++;
            }
        }

        emitterSet = emitterSet->GetNext();
    }

    m_DrawPathFlag[ core ] |= drawPathFlag;
}

//------------------------------------------------------------------------------
//  ソートバッファの描画処理を行います。
//------------------------------------------------------------------------------
void System::DrawSortBuffer( bool calcStreamOut, void* userParam )
{
    CpuCore core = GetCurrentCore();

    if ( m_AddedSortBufferNum[core] == 0 )
    {
        m_DrawPathFlag[core]  = 0;
        return;
    }

    // ソート
    qsort( m_SortEmittetSet[core], m_AddedSortBufferNum[core], sizeof(sortEmitterSets), CompareEmitterSetViewZ );

    // ソートした結果を描画
    for ( u32 i = 0; i < m_AddedSortBufferNum[core]; i++ )
    {
        m_SortEmittetSet[core][i].emitterSet->Draw( m_CurrentShaderType[core],
                                                   m_DrawPathFlag[core],
                                                   m_DrawViewFlag[core],
                                                   calcStreamOut,
                                                   userParam,
                                                   m_EmitterDrawCullingCallback,
                                                   m_EmitterDrawProfiler );
    }

    // ソートバッファをリセット
    m_AddedSortBufferNum[ core ]   = 0;
    m_DrawPathFlag[ core ]         = 0;
}

//------------------------------------------------------------------------------
//  ソートバッファの描画処理を行います。
//------------------------------------------------------------------------------
//void System::DrawSortBuffer( void* userParam, DrawEmitterProfilerCallback profiler )
//{
//    DrawSortBuffer( true, userParam, profiler );
//}

//---------------------------------------------------------------------------
//  描画処理を行います。
//---------------------------------------------------------------------------
//void System::Draw( u8 groupID, u32 drawPathFlag, bool sort, void* userParam, DrawEmitterProfilerCallback profiler )
//{
//    // StreamOutは計算させる（デフォルト）処理に回す
//    Draw( groupID, drawPathFlag, sort, true, userParam, profiler );
//}




//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファをスワップします。
//---------------------------------------------------------------------------
void System::SwapBuffer()
{
    m_DrawTempBuffer[EFT_CPU_CORE_0].Swap();
    m_DrawTempBuffer[EFT_CPU_CORE_1].Swap();
    m_DrawTempBuffer[EFT_CPU_CORE_2].Swap();
}

//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファからメモリを確保します。
//---------------------------------------------------------------------------
void* System::AllocFromTempBuffer( u32 size )
{
    return m_DrawTempBuffer[GetCurrentCore()].Alloc( size );
}

//---------------------------------------------------------------------------
//  描画処理用テンポラリバッファのキャッシュフラッシュを行います。
//---------------------------------------------------------------------------
void System::FlushTempBuffer()
{
    m_DrawTempBuffer[EFT_CPU_CORE_0].FlushCache();
    m_DrawTempBuffer[EFT_CPU_CORE_1].FlushCache();
    m_DrawTempBuffer[EFT_CPU_CORE_2].FlushCache();
}

//---------------------------------------------------------------------------
//  リソース更新に伴うアップデートを行います。
//---------------------------------------------------------------------------
void System::UpdateFromResource( EmitterResource* emitterResSet, const bool withReset )
{
    for( u32 i = 0; i < EFT_GROUP_MAX; i++ )
    {
        if ( m_EmitterSetHead[i] )
        {
            EmitterSet* emitterSet = m_EmitterSetHead[i];

            while( emitterSet )
            {
                if( withReset )
                {
                    // リセット指定の場合、リセットしてメモリサイズなどの再計算。
                    emitterSet->Reset();
                }
                else
                {
                    // そうでない場合、リソースのアップデート。
                    bool result = emitterSet->UpdateFromResource( emitterResSet );
                    if( !result )
                    {
                        // リソース更新に失敗した場合、エミッタセットを消しておく（即時）
                        EmitterSet* failedEmitterSet = emitterSet;
                        emitterSet = emitterSet->m_Next;
                        KillEmitterSet( failedEmitterSet, true );
                        continue;
                    }
                }

                emitterSet = emitterSet->m_Next;
            }
        }
    }
}

//---------------------------------------------------------------------------
//! @brief      エミッタ情報をダンプする
//---------------------------------------------------------------------------
void DumpEmitterInformation( Emitter* emitter, const int debugLevel )
{
    // デバッグレベルに応じて、出す情報量を調整する。
    switch( debugLevel )
    {
    case 0:
        OutputWarning( "  %s - %s\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name );
        break;
    case 1:
    default:
        OutputWarning( "  %s-%s-GroupId:%d\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name, emitter->groupID );
        break;
    }
}

//------------------------------------------------------------------------------
//  32bit float から 24bit float を生成します。
//------------------------------------------------------------------------------
static u32 Float32ToBits24(f32 value)
{
    enum
    {
        SIGN32 = 0x80000000,
        SIGN24 = 0x00800000,

        EXP_BIAS32 = 127,
        EXP_BIAS24 = 63,
        EXP_MASK32 = 0x7F800000,
        EXP_MASK24 = 0x007F0000,

        FRACTION_WIDTH32 = 23,
        FRACTION_MASK32  = 0x007FFFFF,
        FRACTION_WIDTH24 = 16,
        FRACTION_MASK24  = 0x0000FFFF
    };

    u32 bits32 = *reinterpret_cast<u32*>(&value);

    u32 sign = bits32 & SIGN32;
    int exp = (int)((bits32 & EXP_MASK32) >> FRACTION_WIDTH32);
    u32 fraction = bits32 & FRACTION_MASK32;

    u32 bits24 = 0;
    bits24 |= (sign != 0) ? SIGN24 : 0;

    if ((bits32 & ~SIGN32) == 0)
    {
        exp = 0;
    }
    else
    {
        exp = exp - EXP_BIAS32 + EXP_BIAS24;
    }

    fraction = fraction >> (FRACTION_WIDTH32 - FRACTION_WIDTH24);

    if (exp < 0)
    {
        // +0 もしくは -0 なのでそのまま。
    }
    else if (exp > 127)
    {
        // 無限大の処理
        // TODO: IEEE float の無限大の表現がGPU上で有効なのかどうかを要確認。
        bits24 = (u32)0x7F << FRACTION_WIDTH24;
    }
    else
    {
        bits24 |= fraction & FRACTION_MASK24;
        bits24 |= ((u32)exp & 0x7F) << FRACTION_WIDTH24;
    }

    return bits24;
}

//------------------------------------------------------------------------------
//  ソート比較用関数
//------------------------------------------------------------------------------
int System::CompareEmitterSetViewZ( const void *a, const void *b )
{
    sortEmitterSets *src = (sortEmitterSets *)a;
    sortEmitterSets *dst = (sortEmitterSets *)b;

    if ((src->z < 0.0f) && (dst->z < 0.0f))
    {
        if (src->z < dst->z)
            return -1;
        else
            return 1;
    } else {
        if (src->z > dst->z)
            return -1;
        else
            return 1;
    }
}


//---------------------------------------------------------------------------
//  エミッタセットIDを検索します。
//---------------------------------------------------------------------------
s32 System::SearchEmitterSetID( const char* emitterSetName, s32 resId ) const
{
    const Resource* res = GetResource( resId );
    if ( res )
    {
        return res->SearchEmitterSetID( emitterSetName );
    }

    return EFT_INVALID_EMITTER_SET_ID;
}

//---------------------------------------------------------------------------
//  エミッタセット名を検索します。
//---------------------------------------------------------------------------
const char* System::SearchEmitterSetName( s32 emitterSetId , s32 resId )
{
    Resource* res = GetResource( resId );
    if ( res )
    {
        return res->GetEmitterSetName( emitterSetId );
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを設定します。
//---------------------------------------------------------------------------
void System::SetDrawPathRenderStateSetCallback( DrawPathCallBackID id, DrawPathFlag flag,  DrawPathRenderStateSetCallback callback )
{
    if ( id < EFT_DRAW_PATH_CALLBACK_MAX )
    {
        m_DrawPathCallbackFlag[id]               = flag;
        m_DrawPathRenderStateSetCallback[id] = callback;
    }
}

//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを設定します。
//---------------------------------------------------------------------------
void System::SetDrawPathRenderStateSetCallback( DrawPathFlag flag,  DrawPathRenderStateSetCallback callback )
{
    m_DrawPathCallbackFlag[EFT_DRAW_PATH_CALLBACK_0]           = flag;
    m_DrawPathRenderStateSetCallback[EFT_DRAW_PATH_CALLBACK_0] = callback;
}

//---------------------------------------------------------------------------
//  描画パス 描画設定コールバックを取得します。
//---------------------------------------------------------------------------
DrawPathRenderStateSetCallback System::GetDrawPathRenderStateSetCallback( DrawPathFlag flag )
{
    if ( flag == 0 ) return NULL;

    for( s32 i = 0; i < EFT_DRAW_PATH_CALLBACK_MAX; ++i )
    {
        if ( flag & m_DrawPathCallbackFlag[i] ) return m_DrawPathRenderStateSetCallback[i];
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  ストリームアウトエミッタを一時リストに追加します。
//---------------------------------------------------------------------------
void System::AddStreamOutEmitterList( Emitter* streaOutEmitter )
{
    if ( !m_StreamOutEmitter )
    {
        m_StreamOutEmitter                      = streaOutEmitter;
        m_StreamOutEmitterTail                  = m_StreamOutEmitter;
        m_StreamOutEmitterTail->nextSoEmitter   = NULL;
    }
    else
    {
        m_StreamOutEmitterTail->nextSoEmitter   = streaOutEmitter;
        m_StreamOutEmitterTail                  = streaOutEmitter;
        m_StreamOutEmitterTail->nextSoEmitter   = NULL;
    }
}

//---------------------------------------------------------------------------
//  ストリームアウトエミッタリストを処理します。
//---------------------------------------------------------------------------
void System::BatchCalculationStreamOutEmitter( void* userParam, u32 processEmitterFlag )
{
    if ( !m_StreamOutEmitter ) return;

    u32 globalCounter       = GetGlobalCounter();
    Shader* streamOutShader = NULL;
    Emitter* soEmitter      = m_StreamOutEmitter;

    while( soEmitter )
    {
        EmitterSet* pEmitterSet = soEmitter->GetEmitterSet();

        if ( pEmitterSet )
        {
            if ( !streamOutShader )
            {
                nw::eft2::Resource* resource = GetResource( pEmitterSet->m_ResourceID );
                if ( resource )
                {
                    streamOutShader = resource->GetStreamOutShader();
                }
            }

            if ( streamOutShader && ( soEmitter->GetProcessEmitterFlag() & processEmitterFlag ) != 0 )
            {
                if ( ( soEmitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO ) && m_IsEnableStreamOutProcess )
                {
                    soEmitter->emitterCalc->CalcStreamOut( soEmitter, streamOutShader, globalCounter, true, userParam );
                }
            }
        }

        soEmitter = soEmitter->nextSoEmitter;
    }
}

//---------------------------------------------------------------------------
//  パーティクル Zソート
//---------------------------------------------------------------------------
// ソート関数
int _CompareParticleZ( const void* a, const void* b )
{
    const System::ParticleSortInfo *valA = reinterpret_cast<const System::ParticleSortInfo*>( a );
    const System::ParticleSortInfo *valB = reinterpret_cast<const System::ParticleSortInfo*>( b );

    if ( ( valA->value < 0 ) && ( valB->value < 0 ) )
    {
        if ( valA->value < valB->value ) { return -1; }
        else                           { return  1; }
    }
    else
    {
        if ( valA->value > valB->value ) { return -1; }
        else                           { return  1; }
    }
}

//---------------------------------------------------------------------------
//  パーティクルソート（昇順）
//---------------------------------------------------------------------------
int _CompareGreater( const void* a, const void* b )
{
    // aが先に来るなら負数を、そうでないなら正数を返す。
    const System::ParticleSortInfo* valA = reinterpret_cast<const System::ParticleSortInfo*>( a );
    const System::ParticleSortInfo* valB = reinterpret_cast<const System::ParticleSortInfo*>( b );

    if     ( valA->value < valB->value ) { return -1; }
    else if( valA->value > valB->value ) { return  1; }
    else
    {
        // valueが等しい場合はindex順（配列に積まれた順）に従う（※これは昇順／降順に依らない）
        if ( valA->index < valB->index ) { return -1; }
        else                             { return  1; }
    }
}

//---------------------------------------------------------------------------
//  パーティクルソート（降順）
//---------------------------------------------------------------------------
int _CompareLess( const void* a, const void* b )
{
    // aが先に来るなら負数を、そうでないなら正数を返す。
    const System::ParticleSortInfo* valA = reinterpret_cast<const System::ParticleSortInfo*>( a );
    const System::ParticleSortInfo* valB = reinterpret_cast<const System::ParticleSortInfo*>( b );

    if     ( valA->value > valB->value ) { return -1; }
    else if( valA->value < valB->value ) { return  1; }
    else
    {
        // valueが等しい場合はindex順（配列に積まれた順）に従う（※これは昇順／降順に依らない）
        if ( valA->index < valB->index ) { return -1; }
        else                             { return  1; }
    }
}

//---------------------------------------------------------------------------
//  パーティクルのソートを行い、その結果の配列の先頭を返します。
//---------------------------------------------------------------------------
bool System::GetSortedPtclList( ParticleSortInfo** result, u32* arraySize, Emitter* emitter, const ParticleSortType sortType, const CpuCore core )
{
    // 放出中のパーティクル数がバッファサイズ（コンフィグ指定）を超えた場合
    if( emitter->ptclNum > m_ParticleSortBufferCount )
    {
        // 警告を出して失敗させる
        OutputWarning( "Particle sort has failed. More buffer size is needed. \n" );
        *result = NULL;
        *arraySize = 0;
        return false;
    }

    *result = m_ParticleSortBuffer[ core ];  // 先頭アドレスを取得。前回結果が入っている可能性がある。
    *arraySize = 0;

    // ソートタイプで分岐。現状は昇順／降順／Zソートに対応
    switch( sortType )
    {
    case EFT_SORT_TYPE_ORDER:  // 昇順（年寄り順）
        {
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                if( emitter->particlePosAttr[i].IsAlive( emitter->time ) )
                {
                    (*result)[ *arraySize ].value = emitter->particlePosAttr[i].localVec.w; // 生成時刻
                    (*result)[ *arraySize ].index = i;
                    (*arraySize)++;
                }
            }
            qsort( *result, *arraySize, sizeof( ParticleSortInfo ), _CompareGreater );
        }
        break;
    case EFT_SORT_TYPE_REV_ORDER: // 降順（若い順）
        {
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                if( emitter->particlePosAttr[i].IsAlive( emitter->time ) )
                {
                    (*result)[ *arraySize ].value = emitter->particlePosAttr[i].localVec.w; // 生成時刻
                    (*result)[ *arraySize ].index = i;
                    (*arraySize)++;
                }
            }
            qsort( *result, *arraySize, sizeof( ParticleSortInfo ), _CompareLess );
        }
        break;
    case EFT_SORT_TYPE_Z_SORT:          // Zソート
        {
            const ViewParam* view = this->GetViewParam();
            for( u32 i = 0; i < emitter->ptclNum; i++ )
            {
                if( emitter->particlePosAttr[i].IsAlive( emitter->time ) )
                {
                    (*result)[ *arraySize ].value =
                        view->viewMat.f._20 * emitter->particlePosAttr[i].localPos.x +
                        view->viewMat.f._21 * emitter->particlePosAttr[i].localPos.y +
                        view->viewMat.f._22 * emitter->particlePosAttr[i].localPos.z +
                        view->viewMat.f._23;
                    (*result)[ *arraySize ].index = i;
                    (*arraySize)++;
                }
            }
            qsort( *result, *arraySize, sizeof( ParticleSortInfo ), _CompareParticleZ );
        }
        break;
    default:
        EFT_ASSERT( 0 );
        break;
    }

    return true;
}

} // namespace eft2
} // namespace nw

