﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_Resource.h>
#include <nw/eft/eft2_Macro.h>
#include <nw/eft/eft2_Callback.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_RenderContext.h>
#include <nw/eft/eft2_ShaderFlag.h>
#include <nw/eft/eft2_AreaLoop.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_StripeConnection.h>
#include <nw/eft/eft2_SuperStripe.h>

namespace /* anonymous */ {

//--------------------------------------------------------------------------
//! @brief      バイナリからエミッタ時間アニメを取り出します.
//--------------------------------------------------------------------------
nw::eft2::ResAnimEmitterKeyParamSet* GetEmitterAnimationTable(nw::eft2::BinaryData *bin)
{
#ifdef  EFT_BI_ENDIAN
    u8 beforeFlipped = bin->flipped;
#endif

    nw::eft2::ResAnimEmitterKeyParamSet* animBin = bin->GetBinaryDataWithFlip<nw::eft2::ResAnimEmitterKeyParamSet>();

#ifdef  EFT_BI_ENDIAN
    if ( bin->endian != EFT_ENDIAN_TYPE && animBin->keyNum >= 2 && beforeFlipped != bin->flipped )
    {
        // キーが2個以上ある時は2個目以降をここでフリップ(本当は出力コードをいじりたいが……)
        nw::eft2::EndianUtil::FlipArray( animBin->keyNum - 1, animBin->keyValue + 1 );
    }
#endif

    return animBin;
}

} // namespace /* anonymous */


//------------------------------------------------------------------------------
namespace nw   {
namespace eft2 {


//------------------------------------------------------------------------------
//  コンストラクタ
//------------------------------------------------------------------------------
Resource::Resource( Heap *heap, void *bin, u32 resourceID, System* system, bool delayCompile, Resource* residentResource )
{
#ifdef EFT_OGL
    memset( reinterpret_cast< void* >( &m_VshArg ), 0, sizeof( nw::eft2::ShaderCodeArg ) );
    memset( reinterpret_cast< void* >( &m_FshArg ), 0, sizeof( nw::eft2::ShaderCodeArg ) );
#endif

    m_System            = system;
    m_BinaryTop         = bin;
    m_pResidentResource = residentResource;
    ResHeader* header   = reinterpret_cast<ResHeader*>( bin );
    u32 tag             = EFT_MAKE_TAG( 'E', 'F', 'T', 'B' );
#ifdef  EFT_BI_ENDIAN
    if ( header->cpuEndian != EFT_ENDIAN_TYPE )
    {
        // ResHeaderのみタグをu32で受けるので、比較用のタグもひっくり返しておく。
        header->FlipEndian();
        tag = EFT_MAKE_TAG( 'B', 'T', 'F', 'E' );
    }
#endif//EFT_BI_ENDIAN

    if ( tag != header->tag )
    {
        OutputError( "Binary Tag Error.\n" );
    }

    if( EFT_BINARY_VERSION != header->version )
    {
        OutputLog( "Data    Version:%d\n", header->version );
        OutputLog( "Runtime Version:%d\n", EFT_BINARY_VERSION );
        OutputError( "Binary Version Error.\n" );
    }

#if EFT_IS_CAFE
    EFT_ASSERT_MSG( !((u32)bin % EFT_PTCL_BINARY_ALIGNMENT), "[EFT] Binary Alignment Error!  EFT_PTCL_BINARY_ALIGNMENT!!" );
#endif

    void* treeTop = ( reinterpret_cast<u8*>(bin) + sizeof( ResHeader ) );
    Initialize( heap, treeTop, resourceID, delayCompile );
}

//------------------------------------------------------------------------------
//  初期化
//------------------------------------------------------------------------------
void Resource::Initialize( Heap* heap, void* bin, u32 resourceID, bool delayCompile )
{
    m_ResHeap.SetHeap( heap );

    m_ResTexAllocedSize              = 0;
    m_ResPrimAllocedSize             = 0;
    m_ResShaderBinAllocedSize        = 0;
    m_ResShaderCodeAllocedSize       = 0;
    m_ResEmitterSetAllocedSize       = 0;
    m_UboAllocedSize                 = 0;
    m_ResourceID                     = resourceID;
    m_EmitterSetCount                = 0;
    m_EmitterCount                   = 0;
    m_EmitterSetResArray             = NULL;
    m_BindEmitterSetResArray         = NULL;
    m_PrimitiveCount                 = 0;
    m_PrimitiveArray                 = NULL;
    m_TextureCount                   = 0;
    m_TextureArray                   = NULL;
    m_TextureArrayPtr                = NULL;
    m_DelayCompile                   = delayCompile;
    m_ShaderCount                    = 0;
    m_TotalResrouceSize              = 0;
    m_EmitterResourceSize            = 0;
    m_TextureResourceSize            = 0;
    m_PrimitiveResourceSize          = 0;
    m_EmitterCount                   = 0;
    m_ShaderCodeResourceSize         = 0;
        m_IsAlive                    = true;
    // バイナリ内をトレースする
    BinaryData* top = reinterpret_cast<BinaryData*>( bin );
    Trace( top );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄を行います。
//---------------------------------------------------------------------------
Resource::~Resource()
{
    Finalize( &m_ResHeap );
}

//------------------------------------------------------------------------------
//  バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::Trace( BinaryData* topData )
{
    BinaryData* temp = topData;

    m_TotalResrouceSize     = 0;      // メモリ上のバイナリサイズ
    u32 binarySizeNoOffset  = 0;      // パディングを抜いた実バイナリサイズ

    while( temp )
    {
        u32 binarySize = temp->GetBinarySize();
        u32 binaryOffset = temp->GetNextBinaryOffset();

        // TextureArray
        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'T', 'E', 'X', 'A' ) )
        {
            m_TotalResrouceSize += binarySize;
            binarySizeNoOffset  += binaryOffset;
            u32 tempSize = m_ResHeap.GetAllocatedSize();
            TraceTextureArray( temp );
            m_ResTexAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // PrimitiveArray
        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'P', 'R', 'M', 'A' ) )
        {
            m_TotalResrouceSize += binarySize;
            binarySizeNoOffset  += binaryOffset;
            u32 tempSize = m_ResHeap.GetAllocatedSize();
            TracePrimitiveArray( temp );
            m_ResPrimAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // ShaderArray
        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'S', 'H', 'D', 'A' ) )
        {
            m_TotalResrouceSize += binarySize;
            binarySizeNoOffset  += binaryOffset;
            u32 tempSize = m_ResHeap.GetAllocatedSize();

            // userDataにオフラインで計算されたシェーダ数
            TraceShaderArray( temp, temp->GetChildNum() );
            m_ResShaderBinAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }

        // ShaderCodeArray
#ifdef EFT_ONLINE_SHADER_COMPILE
        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'S', 'H', 'C', 'A' ) )
        {
            m_TotalResrouceSize += binarySize;
            binarySizeNoOffset  += binaryOffset;
            u32 tempSize = m_ResHeap.GetAllocatedSize();
            TraceShaderCodeArray( temp );
            m_ResShaderCodeAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
        }
#endif
        temp = temp->GetNextData();
    }

    // EmitterSetArray をトレース
    temp = topData;
    while( temp )
    {
        u32 binarySize = temp->GetBinarySize();
        u32 binaryOffset = temp->GetNextBinaryOffset();

        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'S', 'T', 'A' ) )
        {
            m_TotalResrouceSize += binarySize;
            binarySizeNoOffset  += binaryOffset;
#ifdef EFT_ONLINE_SHADER_COMPILE
            {
                u32 tempSize = m_ResHeap.GetAllocatedSize();
                #ifndef EFT_DEGRADATION_SPEC
                m_ShaderManager.Initialize( &m_ResHeap, temp->GetChildNum() * 32 );
                #else
                m_ShaderManager.Initialize( &m_ResHeap, temp->GetChildNum() * 8 );
                #endif
                m_ShaderManagerAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
            }
#endif
            {
                u32 tempSize = m_ResHeap.GetAllocatedSize();
                TraceEmitterSetArray( temp );
                m_ResEmitterSetAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
            }
        }
        temp = temp->GetNextData();
    }
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Resource::Finalize( Heap *heap )
{
    if ( heap ) m_ResHeap.SetHeap( heap );

    m_StreamOutShader.Finalize( &m_ResHeap );
    m_ShaderManager.Finalize( &m_ResHeap );

    for( u32 i = 0; i < m_PrimitiveCount; i++ )
    {
        m_PrimitiveArray[i].Finalize( &m_ResHeap );
    }
    if ( m_PrimitiveArray ) m_ResHeap.Free( m_PrimitiveArray );

    for( u32 i = 0; i < m_TextureCount; i++ )
    {
        m_TextureArray[i].Finalize();
    }
    if ( m_TextureArrayPtr ) m_ResHeap.Free( m_TextureArrayPtr );

    for( u32 i = 0; i < m_EmitterSetCount; i++ )
    {
        for( u32 j = 0; j < m_EmitterSetResArray[i].emitterAllNum; j++ )
        {
            FinalizeEmitterResource( &m_EmitterSetResArray[i].emitterResSet[j] );
            if ( m_EmitterSetResArray[i].emitterResSet[j].emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
            {
                m_ResHeap.Free( m_EmitterSetResArray[i].emitterResSet[j].resEmitterStaticUniformBlock );
            }
        }
        m_ResHeap.Free( m_EmitterSetResArray[i].emitterResSet );
    }
    if ( m_EmitterSetResArray ) m_ResHeap.Free( m_EmitterSetResArray );
    if ( m_BindEmitterSetResArray ) m_ResHeap.Free( m_BindEmitterSetResArray );

    // コンバイナシェーダのリソースを解放する
#ifdef EFT_OGL
    if ( m_FshArg.combinerCodeSize ) m_ResHeap.Free( m_FshArg.combinerCodeSize );
    if ( m_FshArg.combinerCode ) m_ResHeap.Free( m_FshArg.combinerCode );
    m_FshArg.combinerCode = NULL;
    m_FshArg.combinerCodeSize = NULL;
#endif
    m_PrimitiveArray         = NULL;
    m_TextureArrayPtr        = NULL;
    m_EmitterSetResArray     = NULL;
    m_BindEmitterSetResArray = NULL;

    EFT_ASSERT( m_ResHeap.GetAllocatedCount() == 0 );
}

//------------------------------------------------------------------------------
//  TextureArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceTextureArray( BinaryData* data )
{
    m_TextureResourceSize = 0;
    m_TextureCount        = data->GetDirectChildNum( EFT_MAKE_TAG( 'T', 'E', 'X', 'R' ) );
    if ( m_TextureCount == 0 ) { return; }

    u32 textureArraySize = sizeof(TextureResource) * m_TextureCount + 32;
    m_TextureArrayPtr    = reinterpret_cast<TextureResource*>( m_ResHeap.Alloc( textureArraySize ) );
    m_TextureArray       = (TextureResource *) new ( m_TextureArrayPtr ) TextureResource[ m_TextureCount ];

    BinaryData* textureData = data->GetDirectChildData( EFT_MAKE_TAG( 'T', 'E', 'X', 'R' ) );
    EFT_NULL_ASSERT( textureData );
    u32 i = 0;

    while( textureData )
    {
        ResTexture* texRes = textureData->GetBinaryDataWithFlip<ResTexture>();

        // GX2 Texture
        BinaryData* texGX2Bin = textureData->GetDirectChildData( EFT_MAKE_TAG( 'G', 'X', '2', 'B' ) );
        if ( texGX2Bin )
        {
            m_TextureArray[i].InitializeGx2bImage( texRes->guid, texRes, texGX2Bin->GetBinaryData(), texGX2Bin->GetBinarySize() );
            m_TextureResourceSize += texGX2Bin->GetBinarySize();
        }

        // Original(32 or 16bit) Texture
        BinaryData* texOrgbBin = textureData->GetDirectChildData( EFT_MAKE_TAG( 'O', 'R', 'G', 'B' ) );
        if ( texOrgbBin )
        {
            m_TextureArray[i].InitializeOrgbImage( texRes->guid, texRes, texOrgbBin->GetBinaryData(), texOrgbBin->GetBinarySize() );
            m_TextureResourceSize += texOrgbBin->GetBinarySize();
        }

        // PVR Texture
        TextureResourceInitializer initializer = m_System->GetTextureInitializer();
        TextureResourceFinalizer   finalzier   = m_System->GetTextureFinalizer();
        BinaryData* texPvrbBin = textureData->GetDirectChildData( EFT_MAKE_TAG( 'P', 'V', 'R', 'B' ) );
        if ( texPvrbBin && initializer && finalzier )
        {
            if ( initializer( &m_TextureArray[i], texRes->guid, texRes, texPvrbBin->GetBinaryData(), texPvrbBin->GetBinarySize(), &m_ResHeap ) )
            {
                m_TextureResourceSize += texPvrbBin->GetBinarySize();
            }
        }


        textureData = textureData->GetNextData();
        ++i;
    }

    return;
}


//------------------------------------------------------------------------------
//  ResPrimitive
//------------------------------------------------------------------------------
struct ResVertexElementInfo
{
    u32     Count;          //!< 配列サイズ.
    u32     Element;        //!< 配列構成要素数.

    void FlipEndian()
    {
        EndianUtil::Flip( &Count );
        EndianUtil::Flip( &Element );
    }
};

struct ResPrimitive
{
    u64                  UniqueID;          //!< 識別用ID.
    ResVertexElementInfo Position;          //!< 位置座標.
    ResVertexElementInfo Normal;            //!< 法線ベクトル.
    ResVertexElementInfo Tangent;           //!< 接線ベクトル.
    ResVertexElementInfo Color;             //!< 頂点カラー.
    ResVertexElementInfo TexCoord0;         //!< テクスチャ座標0
    ResVertexElementInfo TexCoord1;         //!< テクスチャ座標1
    u32                  IndexCount;        //!< 頂点インデックス数.
    u32                  OffsetPosition;    //!< 座標配列オフセット
    u32                  OffsetNormal;      //!< 法線配列オフセット
    u32                  OffsetTangent;     //!< 接線配列オフセット
    u32                  OffsetColor;       //!< カラー配列オフセット
    u32                  OffsetTexCoord;    //!< UV配列オフセット
    u32                  OffsetIndex;       //!< インデックス配列オフセット

    u8* GetPositionArray()
    {
        return CalcOffset(OffsetPosition);
    }

    u8* GetNormalArray()
    {
        return CalcOffset(OffsetNormal);
    }

    u8* GetTangentArray()
    {
        return CalcOffset(OffsetTangent);
    }

    u8* GetColorArray()
    {
        return CalcOffset(OffsetColor);
    }

    u8* GetTexCoordArray()
    {
        return CalcOffset(OffsetTexCoord);
    }

    u8* GetIndexArray()
    {
        return CalcOffset(OffsetIndex);
    }

    void FlipEndian()
    {
#ifdef  EFT_ENDIAN_LITTLE
        Position.FlipEndian();
        Normal.FlipEndian();
        Tangent.FlipEndian();
        Color.FlipEndian();
        TexCoord0.FlipEndian();
        TexCoord1.FlipEndian();
        EndianUtil::Flip( &UniqueID );
        EndianUtil::Flip( &IndexCount );
        EndianUtil::Flip( &OffsetPosition );
        EndianUtil::Flip( &OffsetNormal );
        EndianUtil::Flip( &OffsetTangent );
        EndianUtil::Flip( &OffsetColor );
        EndianUtil::Flip( &OffsetTexCoord );
        EndianUtil::Flip( &OffsetIndex );

        FlipVertexArray( GetPositionArray() );
        FlipVertexArray( GetNormalArray() );
        FlipVertexArray( GetTangentArray() );
        FlipVertexArray( GetColorArray() );
        FlipVertexArray( GetTexCoordArray() );
        FlipIndexArray();
#endif
    }

private:
    u8* CalcOffset(u32 offset)
    {
        return offset == 0 ? NULL : reinterpret_cast<u8*>(this) + offset;
    }

#ifdef  EFT_ENDIAN_LITTLE
    void FlipVertexArray(u8* vertexArray)
    {
        if (vertexArray == NULL)
        {
            return;
        }

        nw::math::VEC4* vec4Array = reinterpret_cast<nw::math::VEC4*>(vertexArray);
        EndianUtil::FlipArray(Position.Count, vec4Array);
    }

    void FlipIndexArray()
    {
        u32* indexArray = reinterpret_cast<u32*>(GetIndexArray());
        if (indexArray == NULL)
        {
            return;
        }

        EndianUtil::FlipArray<u32>(IndexCount, indexArray);
    }
#endif
};


//------------------------------------------------------------------------------
//  PrimitiveArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TracePrimitiveArray( BinaryData* data )
{
    m_PrimitiveResourceSize = 0;
    m_PrimitiveCount       = data->GetDirectChildNum( EFT_MAKE_TAG( 'P', 'R', 'I', 'M' ) );
    if ( m_PrimitiveCount == 0 ){ return; }

    m_PrimitiveArray     = reinterpret_cast<Primitive*>
        ( m_ResHeap.Alloc( sizeof(Primitive) * m_PrimitiveCount ) );

    BinaryData* primData = data->GetDirectChildData( EFT_MAKE_TAG( 'P', 'R', 'I', 'M' ) );
    EFT_NULL_ASSERT( primData );

    u32 i = 0;

    while( primData )
    {
        ResPrimitive* pResPrimitive = primData->GetBinaryDataWithFlip<ResPrimitive>();

        void* tanBin = pResPrimitive->GetTangentArray();
        void* colBin = pResPrimitive->GetColorArray();

        bool result = m_PrimitiveArray[i].Initialize(
            &m_ResHeap, pResPrimitive->UniqueID, pResPrimitive->IndexCount, pResPrimitive->Position.Count,
            pResPrimitive->GetPositionArray(),
            pResPrimitive->GetNormalArray(),
            tanBin,
            colBin,
            pResPrimitive->GetTexCoordArray(),
            pResPrimitive->GetIndexArray() );

        if( !result )
        {
            // プリミティブ初期化に失敗
            OutputWarning( "Faild to initialize primitive: Index %d, UniqueID %d \n", i, pResPrimitive->UniqueID );
            i++;
            primData = primData->GetNextData();
            continue;
        }

        // 初期化に成功
        // 偽装した時しか追加でアロケートしない
        u32 bufferSize = pResPrimitive->Position.Count * sizeof( nw::math::VEC4 );
        m_PrimitiveResourceSize += ( tanBin == NULL ? bufferSize : 0 ) + ( colBin == NULL ? bufferSize : 0 );
        i++;
        primData = primData->GetNextData();
    }

    EFT_ASSERT( m_PrimitiveCount == i );
}

//------------------------------------------------------------------------------
//  ShaderArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceShaderArray( BinaryData* data, u32 shaderNum )
{
    EFT_UNUSED_VARIABLE( data );
    m_ShaderCount        = shaderNum;
    if( m_ShaderCount == 0 ) return;

#ifdef EFT_OFFLINE_SHADER_COMPILE
    BinaryData* shaderData = data->GetDirectChildData( EFT_MAKE_TAG( 'S', 'H', 'D', 'B' ) );

    u32 tempSize = m_ResHeap.GetAllocatedSize();

    // シェーダマネージャを初期化
    m_ShaderManager.Initialize( &m_ResHeap, shaderNum );

    // 登録する。
    m_ShaderManager.SetShaderBinary( &m_ResHeap, shaderData->GetBinaryData() );

    // ストリームアウト用のシェーダリソースをセットアップする
    m_StreamOutShader.Invalidate();
    if ( m_pResidentResource != NULL )
    {
        m_StreamOutShader.InitializeShader(
                              &m_ResHeap,
                              m_pResidentResource->GetShaderManager()->GetShaderBinary(),
                              m_pResidentResource->GetShaderManager()->GetVertexShaderNum()-1,
                              m_pResidentResource->GetShaderManager()->GetFragmentShaderNum()-1 );
    }
    else
    {
        m_StreamOutShader.InitializeShader(
                              &m_ResHeap,
                              m_ShaderManager.GetShaderBinary(),
                              m_ShaderManager.GetVertexShaderNum()-1,
                              m_ShaderManager.GetFragmentShaderNum()-1 );
    }
    m_StreamOutShader.InitializeLocation();
    m_StreamOutShader.BindingUniformBlock();
    m_StreamOutShader.SetupShader( &m_ResHeap );
    EFT_ASSERT( m_StreamOutShader.GetShaderResource()->GetVertexShader()->hasStreamOut == GX2_TRUE );

    m_ResShaderBinAllocedSize = m_ResHeap.GetAllocatedSize() - tempSize;
#endif

}

//------------------------------------------------------------------------------
//  Shader Source Code Array をトレースする
//------------------------------------------------------------------------------
#ifdef EFT_ONLINE_SHADER_COMPILE
void Resource::TraceShaderCodeArray( BinaryData* data )
{
    memset( (void*)&m_VshArg, 0, sizeof(nw::eft2::ShaderCodeArg) );
    memset( (void*)&m_FshArg, 0, sizeof(nw::eft2::ShaderCodeArg) );

    m_VshArg.streamout      = NULL;
    m_VshArg.streamoutSize  = 0;
    m_FshArg.streamout      = NULL;
    m_FshArg.streamoutSize  = 0;

    // コンバイナシェーダの個数を数える
    m_FshArg.combinerCodeCount = 0;
    BinaryData* next = data->GetChildData();
    while( next )
    {
        if ( next->GetBinaryTag() == EFT_MAKE_TAG( 'S', 'H', 'E', 'C' ) )
        {
            m_FshArg.combinerCodeCount++;
        }
        next = next->GetNextData();
    }

    // コンバイナシェーダのリソースを確保する.
    if ( m_FshArg.combinerCodeCount != 0 )
    {
        m_FshArg.combinerCodeSize = reinterpret_cast<GLint*>( m_ResHeap.Alloc( sizeof(GLint) * m_FshArg.combinerCodeCount ) );
        memset( m_FshArg.combinerCodeSize, 0, sizeof(GLint) * m_FshArg.combinerCodeCount );
        m_FshArg.combinerCode     = reinterpret_cast<GLchar**>( m_ResHeap.Alloc( sizeof(GLchar*) * m_FshArg.combinerCodeCount ) );
        memset( m_FshArg.combinerCode, NULL, sizeof(GLchar*) * m_FshArg.combinerCodeCount );
    }

    u32 iCombinerShader = 0;

    m_ShaderCodeResourceSize = 0;
    next = data->GetChildData();
    while( next )
    {
        switch ( next->GetBinaryTag() )
        {
            // Genericシェーダ宣言
        case EFT_MAKE_TAG( 'S', 'H', 'D', 'G' ):
            m_VshArg.specDecGenericCode     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.specDecGenericCodeSize = next->GetBinarySize();
            m_FshArg.specDecGenericCode     = m_VshArg.specDecGenericCode;
            m_FshArg.specDecGenericCodeSize = m_VshArg.specDecGenericCodeSize;
            break;

            // 実機用シェーダ宣言
        case EFT_MAKE_TAG( 'S', 'H', 'D', 'T' ):
            m_VshArg.specDecTargetCode     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.specDecTargetCodeSize = next->GetBinarySize();
            m_FshArg.specDecTargetCode     = m_VshArg.specDecTargetCode;
            m_FshArg.specDecTargetCodeSize = m_VshArg.specDecTargetCodeSize;
            break;

            // eft_ParticlDeclaration.glsl
        case EFT_MAKE_TAG( 'S', 'H', 'C', 'D' ):
            m_VshArg.glsl         = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.glslSize     = next->GetBinarySize();
            m_FshArg.glsl         = m_VshArg.glsl;
            m_FshArg.glslSize     = m_VshArg.glslSize;
            break;

            // eft_ParticleDeclaration.vsh
        case EFT_MAKE_TAG( 'S', 'H', 'D', 'V' ):
            m_VshArg.dec          = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.decSize      = next->GetBinarySize();
            break;

            // eft_ParticleDeclaration.fsh
        case EFT_MAKE_TAG( 'S', 'H', 'D', 'F' ):
            m_FshArg.dec          = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.decSize      = next->GetBinarySize();
            break;

            // eft_StreamOutDeclaration.vsh
        case EFT_MAKE_TAG( 'S', 'H', 'D', 'C' ):
            m_VshArg.soDec     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.soDecSize = next->GetBinarySize();
            break;

            // eft_Particle.vsh
        case EFT_MAKE_TAG( 'S', 'H', 'P', 'V' ):
            m_VshArg.program      = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.programSize  = next->GetBinarySize();
            break;

            // eft_Particle.fsh
        case EFT_MAKE_TAG( 'S', 'H', 'P', 'F' ):
            m_FshArg.program      = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.programSize  = next->GetBinarySize();
            break;

            // eft_StreamOut.vsh
        case EFT_MAKE_TAG( 'S', 'H', 'S', 'T' ):
            m_VshArg.streamout       = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.streamoutSize   = next->GetBinarySize();
            break;

            // CustomSahderGeneral.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', 'G' ):
            m_VshArg.customGeneralCode     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customGeneralCodeSize = next->GetBinarySize();
            break;
            // CustomSahderGeneral.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', 'G' ):
            m_FshArg.customGeneralCode     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customGeneralCodeSize = next->GetBinarySize();
            break;
            // CustomShaderGeneral.csh
        case EFT_MAKE_TAG( 'C', 'S', 'C', 'G'):
            m_VshArg.soCustom = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.soCustomSize = next->GetBinarySize();
            break;
            // CustomSahder0.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '0' ):
            m_VshArg.customCode[0]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[0]    = next->GetBinarySize();
            break;
            // CustomSahder0.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '0' ):
            m_FshArg.customCode[0]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[0]    = next->GetBinarySize();
            break;
            // CustomSahder1.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '1' ):
            m_VshArg.customCode[1]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[1]    = next->GetBinarySize();
            break;
            // CustomSahder1.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '1' ):
            m_FshArg.customCode[1]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[1]    = next->GetBinarySize();
            break;
            // CustomSahder2.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '2' ):
            m_VshArg.customCode[2]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[2]    = next->GetBinarySize();
            break;
            // CustomSahder2.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '2' ):
            m_FshArg.customCode[2]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[2]    = next->GetBinarySize();
            break;
            // CustomSahder3.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '3' ):
            m_VshArg.customCode[3]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[3]    = next->GetBinarySize();
            break;
            // CustomSahder3.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '3' ):
            m_FshArg.customCode[3]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[3]    = next->GetBinarySize();
            break;
            // CustomSahder4.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '4' ):
            m_VshArg.customCode[4]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[4]    = next->GetBinarySize();
            break;
            // CustomSahder4.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '4' ):
            m_FshArg.customCode[4]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[4]    = next->GetBinarySize();
            break;
            // CustomSahder5.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '5' ):
            m_VshArg.customCode[5]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[5]    = next->GetBinarySize();
            break;
            // CustomSahder5.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '5' ):
            m_FshArg.customCode[5]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[5]    = next->GetBinarySize();
            break;
            // CustomSahder6.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '6' ):
            m_VshArg.customCode[6]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[6]    = next->GetBinarySize();
            break;
            // CustomSahder6.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '6' ):
            m_FshArg.customCode[6]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[6]    = next->GetBinarySize();
            break;
            // CustomSahder7.vsh
        case EFT_MAKE_TAG( 'C', 'S', 'V', '7' ):
            m_VshArg.customCode[7]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.customCodeSize[7]    = next->GetBinarySize();
            break;
            // CustomSahder7.fsh
        case EFT_MAKE_TAG( 'C', 'S', 'F', '7' ):
            m_FshArg.customCode[7]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.customCodeSize[7]    = next->GetBinarySize();
            break;

            // EmitterPlugin1.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '0' ):
            m_VshArg.emitterPluginCode[0]       = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[0]   = next->GetBinarySize();
            break;

            // EmitterPlugin2.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '1' ):
            m_VshArg.emitterPluginCode[1]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[1]    = next->GetBinarySize();
            break;

            // EmitterPlugin3.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '2' ):
            m_VshArg.emitterPluginCode[2]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[2]    = next->GetBinarySize();
            break;

            // EmitterPlugin4.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '3' ):
            m_VshArg.emitterPluginCode[3]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[3]    = next->GetBinarySize();
            break;

            // EmitterPlugin5.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '4' ):
            m_VshArg.emitterPluginCode[4]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[4]    = next->GetBinarySize();
            break;

            // EmitterPlugin6.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '5' ):
            m_VshArg.emitterPluginCode[5]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[5]    = next->GetBinarySize();
            break;

            // EmitterPlugin7.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '6' ):
            m_VshArg.emitterPluginCode[6]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[6]    = next->GetBinarySize();
            break;

            // EmitterPlugin8.vsh
        case EFT_MAKE_TAG( 'E', 'P', 'V', '7' ):
            m_VshArg.emitterPluginCode[7]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_VshArg.emitterPluginCodeSize[7]    = next->GetBinarySize();
            break;

            // EmitterPlugin1.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '0' ):
            m_FshArg.emitterPluginCode[0]       = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[0]   = next->GetBinarySize();
            break;

            // EmitterPlugin2.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '1' ):
            m_FshArg.emitterPluginCode[1]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[1]    = next->GetBinarySize();
            break;

            // EmitterPlugin3.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '2' ):
            m_FshArg.emitterPluginCode[2]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[2]    = next->GetBinarySize();
            break;

            // EmitterPlugin4.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '3' ):
            m_FshArg.emitterPluginCode[3]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[3]    = next->GetBinarySize();
            break;

            // EmitterPlugin5.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '4' ):
            m_FshArg.emitterPluginCode[4]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[4]    = next->GetBinarySize();
            break;

            // EmitterPlugin6.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '5' ):
            m_FshArg.emitterPluginCode[5]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[5]    = next->GetBinarySize();
            break;

            // EmitterPlugin7.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '6' ):
            m_FshArg.emitterPluginCode[6]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[6]    = next->GetBinarySize();
            break;

            // EmitterPlugin8.fsh
        case EFT_MAKE_TAG( 'E', 'P', 'F', '7' ):
            m_FshArg.emitterPluginCode[7]        = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.emitterPluginCodeSize[7]    = next->GetBinarySize();
            break;

            // Effect Combiner
        case EFT_MAKE_TAG( 'S', 'H', 'E', 'C' ):
            m_FshArg.combinerCode[iCombinerShader]     = reinterpret_cast<GLchar*>(next->GetBinaryData() );
            m_FshArg.combinerCodeSize[iCombinerShader] = next->GetBinarySize();
            iCombinerShader++;
            break;
        }

        m_ShaderCodeResourceSize += next->GetBinarySize();
        next = next->GetNextData();
    }

    // シェーダコードを設定
    ShaderResource::SetShaderCode( m_VshArg, m_FshArg );

    // ストリームアウト用のシェーダリソースを生成する
#ifndef EFT_DEGRADATION_SPEC
    m_StreamOutShader.Invalidate();
    m_StreamOutShader.InitializeStreamOutShader();
    if ( m_StreamOutShader.GetShaderResource()->IsError() )
    {
        OutputError( "[EFT] StreamOut Shader Compile Failed.\n" );
    }
    m_StreamOutShader.InitializeLocation();
    m_StreamOutShader.BindingUniformBlock();
#endif
}
#endif

//------------------------------------------------------------------------------
//  EmitterSetArray バイナリをトレースする
//------------------------------------------------------------------------------
void Resource::TraceEmitterSetArray( BinaryData* data )
{
    m_EmitterSetCount = data->GetDirectChildNum( EFT_MAKE_TAG( 'E', 'S', 'E', 'T' ) );
    EFT_ASSERT( m_EmitterSetCount != 0 );
    m_EmitterSetResArray = reinterpret_cast<EmitterSetResource*>
        ( m_ResHeap.Alloc( sizeof(EmitterSetResource) * m_EmitterSetCount ) );
    memset( m_EmitterSetResArray, NULL, sizeof(EmitterSetResource) * m_EmitterSetCount );

    // バインド用
    m_BindEmitterSetResArray = reinterpret_cast<EmitterSetResource**>
        ( m_ResHeap.Alloc( sizeof(EmitterSetResource*) * m_EmitterSetCount ) );
    memset( m_BindEmitterSetResArray, NULL, sizeof(EmitterSetResource*) * m_EmitterSetCount );

    BinaryData* emSetData = data->GetDirectChildData( EFT_MAKE_TAG( 'E', 'S', 'E', 'T' ) );
    EFT_NULL_ASSERT( emSetData );
    u32 i = 0;

    while( emSetData )
    {
        m_EmitterSetResArray[i].emSetData = emSetData;
        InitializeEmitterSetResource( &m_EmitterSetResArray[i], m_DelayCompile );
        emSetData = emSetData->GetNextData();
        i++;
    }
}

//------------------------------------------------------------------------------
//  EmitterSetResource を初期化する
//------------------------------------------------------------------------------
void Resource::InitializeEmitterSetResource( EmitterSetResource* emSetRes, bool delayCompile )
{
    emSetRes->visibility = true;

    // エンディアンフリップしつつ構造体取得
    emSetRes->emitterSetRes = emSetRes->emSetData->GetBinaryDataWithFlip<ResEmitterSet>();

    u32 tag = EFT_MAKE_TAG( 'E', 'M', 'T', 'R' );
    // 子を含まない総エミッタ数
    emSetRes->emitterNum    =  emSetRes->emSetData->GetDirectChildNum( tag );

    // 子を含む総エミッタ数
    emSetRes->emitterAllNum = emSetRes->emitterSetRes->emitterAllNum;
    m_EmitterCount += emSetRes->emitterAllNum;

    // TODO:子エミッタへの対応。
    emSetRes->emitterResSet = reinterpret_cast<EmitterResource*>
        ( m_ResHeap.Alloc( sizeof(EmitterResource) * ( emSetRes->emitterAllNum ) ) );

    BinaryData* emData = emSetRes->emSetData->GetDirectChildData( tag );
    EFT_NULL_ASSERT( emData );

    // シェーダマネージャーを設定
    emSetRes->shaderManager = &m_ShaderManager;

    // テクスチャリソースの設定
    emSetRes->textureNum   = m_TextureCount;
    emSetRes->textureArray = m_TextureArray;

    // プリミティブリソースの設定
    emSetRes->primitiveNum   = m_PrimitiveCount;
    emSetRes->primitiveArray = m_PrimitiveArray;

    EmitterResource* cur    = NULL;
    EmitterResource* prev   = NULL;
    u32 i = 0;
    u32 nchild = 0;
    u32 parentIndex = 0;

    emSetRes->isLifeInfinity = false;
    emSetRes->isLoopEffect   = false;

    while( emData )
    {
        cur = &emSetRes->emitterResSet[i];
        EFT_ASSERT( i < emSetRes->emitterAllNum );

        cur->next = NULL;
        if ( prev ) prev->next = cur;

        // エミッタリソースの初期化
        InitializeEmitterResource( cur, emData, parentIndex );
        i++;
        parentIndex++;

        // ループ・無限寿命判定
        if ( cur->emitterData->ptcl.isLifeInfinity )
        {
            emSetRes->isLifeInfinity = true;
        }
        if ( !cur->emitterData->emission.isOneTime )
        {
            emSetRes->isLoopEffect = true;
        }

        // シェーダマネージャに登録
        eft2::ResShader* resShader = &cur->emitterData->shader;
#ifdef EFT_OGL
        cur->shaderIndex            = m_ShaderManager.AddEmitterResSet( cur, NULL, delayCompile );
        if ( strlen( cur->emitterData->shader.userShaderDefine1 ) > 0 )
        {
            cur->drawPathShader1    = m_ShaderManager.AddEmitterResSet( cur, resShader->userShaderDefine1, delayCompile );
        }
        else
        {
            cur->drawPathShader1    = 0xFFFFFFFF;
        }
        if ( strlen( cur->emitterData->shader.userShaderDefine2 ) > 0 )
        {
            cur->drawPathShader2    = m_ShaderManager.AddEmitterResSet( cur, resShader->userShaderDefine2, delayCompile );
        }
        else
        {
            cur->drawPathShader2    = 0xFFFFFFFF;
        }
#endif
#if EFT_GX2
        cur->shaderIndex    = m_ShaderManager.AddShaderSet( &m_ResHeap, resShader->vertexShaderIndex, resShader->computeShaderIndex );  // computeShaderIndexにpixelShaderIndexが入ってくる
        if ( strlen( cur->emitterData->shader.userShaderDefine1 ) > 0 )
        {
            cur->drawPathShader1    = m_ShaderManager.AddShaderSet( &m_ResHeap, resShader->userVertexShaderIndex1, resShader->userPixelShaderIndex1 );
        }
        else
        {
            cur->drawPathShader1    = 0xFFFFFFFF;
        }
        if ( strlen( cur->emitterData->shader.userShaderDefine2 ) > 0 )
        {
            cur->drawPathShader2    = m_ShaderManager.AddShaderSet( &m_ResHeap, resShader->userVertexShaderIndex2, resShader->userPixelShaderIndex2 );
        }
        else
        {
            cur->drawPathShader2    = 0xFFFFFFFF;
        }
#endif
        // 子エミッタへの対応。
        memset( cur->childEmitterResSet, 0, sizeof(EmitterResource*) * EFT_EMITTER_INSET_NUM );
        cur->childEmitterResCount = 0;

        s32 childIndex = 0;

        if ( emData->GetChildData() )
        {
            EmitterResource* child = NULL;
            //EFT_ASSERT( i < emSetRes->emitterNum );
            // 各エミッタリソースごとの初期化
            BinaryData* emChildData = emData->GetChildData();

            while( emChildData )
            {
                child = &emSetRes->emitterResSet[i];
                i++;

                // エミッタリソースの初期化
                InitializeEmitterResource( child, emChildData, childIndex );
                child->childEmitterResCount = 0;
                childIndex++;

                // ループ・無限寿命判定
                if ( child->emitterData->ptcl.isLifeInfinity )
                {
                    emSetRes->isLifeInfinity = true;
                }

                // チャイルドは親の寿命で放出を終了するためワンタイム扱い。
                eft2::ResShader* resChildShader = &child->emitterData->shader;
#ifdef EFT_OGL
                child->shaderIndex          = m_ShaderManager.AddEmitterResSet( child, NULL, delayCompile );
                if ( strlen( child->emitterData->shader.userShaderDefine1 ) > 0 )
                {
                    child->drawPathShader1  = m_ShaderManager.AddEmitterResSet( child, resChildShader->userShaderDefine1, delayCompile );
                }
                else
                {
                    child->drawPathShader1  = 0xFFFFFFFF;
                }
                if ( strlen( child->emitterData->shader.userShaderDefine2 ) > 0 )
                {
                    child->drawPathShader2  = m_ShaderManager.AddEmitterResSet( child, resChildShader->userShaderDefine2, delayCompile );
                }
                else
                {
                    child->drawPathShader2  = 0xFFFFFFFF;
                }
#endif
#if EFT_GX2
                child->shaderIndex          = m_ShaderManager.AddShaderSet( &m_ResHeap, resChildShader->vertexShaderIndex, resChildShader->computeShaderIndex );  // computeShaderIndexにpixelShaderIndexが入ってくる
                if ( strlen( child->emitterData->shader.userShaderDefine1 ) > 0 )
                {
                    child->drawPathShader1  = m_ShaderManager.AddShaderSet( &m_ResHeap, resChildShader->userVertexShaderIndex1, resChildShader->userPixelShaderIndex1 );
                }
                else
                {
                    child->drawPathShader1  = 0xFFFFFFFF;
                }
                if ( strlen( child->emitterData->shader.userShaderDefine2 ) > 0 )
                {
                     child->drawPathShader2 = m_ShaderManager.AddShaderSet( &m_ResHeap, resChildShader->userVertexShaderIndex2, resChildShader->userPixelShaderIndex2 );
                }
                else
                {
                    child->drawPathShader2  = 0xFFFFFFFF;
                }
#endif
                cur->childEmitterResSet[nchild]       = child;
                cur->childEmitterResSet[nchild]->next = NULL;

                memset( cur->childEmitterResSet[nchild]->childEmitterResSet, 0, sizeof(EmitterResource*) * EFT_EMITTER_INSET_NUM );

                nchild++;
                emChildData = emChildData->GetNextData();
                cur->childEmitterResCount = nchild;
            }
        }

        m_EmitterResourceSize += emData->GetBinarySize();
        emData = emData->GetNextData( tag );
        nchild = 0;
        prev   = cur;
    }
}


//---------------------------------------------------------------------------
//  emitterDataを元に他メンバーを更新します。
//---------------------------------------------------------------------------
void EmitterResource::Update()
{
    // 暫定処置
    // uboにFIXED Type の場合は、カラー/アルファ値を書き込む
    if( emitterData->ptclColor.color0Type == EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        emitterData->staticUbo.color0Anim.value[ 0 ].x = emitterData->ptclColor.color0.x;
        emitterData->staticUbo.color0Anim.value[ 0 ].y = emitterData->ptclColor.color0.y;
        emitterData->staticUbo.color0Anim.value[ 0 ].z = emitterData->ptclColor.color0.z;
    }
    if( emitterData->ptclColor.alpha0Type == EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        emitterData->staticUbo.alpha0Anim.value[ 0 ].x = emitterData->ptclColor.alpha0;
    }
    if( emitterData->ptclColor.color1Type == EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        emitterData->staticUbo.color1Anim.value[ 0 ].x = emitterData->ptclColor.color1.x;
        emitterData->staticUbo.color1Anim.value[ 0 ].y = emitterData->ptclColor.color1.y;
        emitterData->staticUbo.color1Anim.value[ 0 ].z = emitterData->ptclColor.color1.z;
    }
    if( emitterData->ptclColor.alpha1Type == EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        emitterData->staticUbo.alpha1Anim.value[ 0 ].x = emitterData->ptclColor.alpha1;
    }

    // 暫定処置
    // 繰り返し数からテクスチャスケールを設定する
    switch( emitterData->textureAnim0.repeat )
    {
    case 0:
        emitterData->staticUbo.texShiftAnim0.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim0.uvScale.y = 1.0f;
        break;
    case 1:
        emitterData->staticUbo.texShiftAnim0.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim0.uvScale.y = 1.0f;
        break;
    case 2:
        emitterData->staticUbo.texShiftAnim0.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim0.uvScale.y = 2.0f;
        break;
    case 3:
        emitterData->staticUbo.texShiftAnim0.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim0.uvScale.y = 2.0f;
        break;
    }
    switch( emitterData->textureAnim1.repeat )
    {
    case 0:
        emitterData->staticUbo.texShiftAnim1.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim1.uvScale.y = 1.0f;
        break;
    case 1:
        emitterData->staticUbo.texShiftAnim1.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim1.uvScale.y = 1.0f;
        break;
    case 2:
        emitterData->staticUbo.texShiftAnim1.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim1.uvScale.y = 2.0f;
        break;
    case 3:
        emitterData->staticUbo.texShiftAnim1.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim1.uvScale.y = 2.0f;
        break;
    }
    switch( emitterData->textureAnim2.repeat )
    {
    case 0:
        emitterData->staticUbo.texShiftAnim2.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim2.uvScale.y = 1.0f;
        break;
    case 1:
        emitterData->staticUbo.texShiftAnim2.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim2.uvScale.y = 1.0f;
        break;
    case 2:
        emitterData->staticUbo.texShiftAnim2.uvScale.x = 1.0f;
        emitterData->staticUbo.texShiftAnim2.uvScale.y = 2.0f;
        break;
    case 3:
        emitterData->staticUbo.texShiftAnim2.uvScale.x = 2.0f;
        emitterData->staticUbo.texShiftAnim2.uvScale.y = 2.0f;
        break;
    }

    // UVスクロール値のキャンセル
    if( !emitterData->textureAnim0.isScroll )
    {
        emitterData->staticUbo.texShiftAnim0.scrollInit.x = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scrollInit.y = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scrollAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scrollAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scrollInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scrollInitRand.y = 0.0f;
    }
    if( !emitterData->textureAnim1.isScroll )
    {
        emitterData->staticUbo.texShiftAnim1.scrollInit.x = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scrollInit.y = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scrollAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scrollAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scrollInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scrollInitRand.y = 0.0f;
    }
    if( !emitterData->textureAnim2.isScroll )
    {
        emitterData->staticUbo.texShiftAnim2.scrollInit.x = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scrollInit.y = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scrollAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scrollAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scrollInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scrollInitRand.y = 0.0f;
    }

    if( !emitterData->textureAnim0.isRotate )
    {
        emitterData->staticUbo.texShiftAnim0.rotAdd = 0.0f;
        emitterData->staticUbo.texShiftAnim0.rotInit = 0.0f;
        emitterData->staticUbo.texShiftAnim0.rotInitRand = 0.0f;
    }
    if( !emitterData->textureAnim1.isRotate )
    {
        emitterData->staticUbo.texShiftAnim1.rotAdd = 0.0f;
        emitterData->staticUbo.texShiftAnim1.rotInit = 0.0f;
        emitterData->staticUbo.texShiftAnim1.rotInitRand = 0.0f;
    }
    if( !emitterData->textureAnim2.isRotate )
    {
        emitterData->staticUbo.texShiftAnim2.rotAdd = 0.0f;
        emitterData->staticUbo.texShiftAnim2.rotInit = 0.0f;
        emitterData->staticUbo.texShiftAnim2.rotInitRand = 0.0f;
    }

    if( !emitterData->textureAnim0.isScale )
    {
        emitterData->staticUbo.texShiftAnim0.scaleInit.x = 1.0f;
        emitterData->staticUbo.texShiftAnim0.scaleInit.y = 1.0f;
        emitterData->staticUbo.texShiftAnim0.scaleAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scaleAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scaleInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim0.scaleInitRand.y = 0.0f;
    }
    if( !emitterData->textureAnim1.isScale )
    {
        emitterData->staticUbo.texShiftAnim1.scaleInit.x = 1.0f;
        emitterData->staticUbo.texShiftAnim1.scaleInit.y = 1.0f;
        emitterData->staticUbo.texShiftAnim1.scaleAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scaleAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scaleInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim1.scaleInitRand.y = 0.0f;
    }
    if( !emitterData->textureAnim2.isScale )
    {
        emitterData->staticUbo.texShiftAnim2.scaleInit.x = 1.0f;
        emitterData->staticUbo.texShiftAnim2.scaleInit.y = 1.0f;
        emitterData->staticUbo.texShiftAnim2.scaleAdd.x = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scaleAdd.y = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scaleInitRand.x = 0.0f;
        emitterData->staticUbo.texShiftAnim2.scaleInitRand.y = 0.0f;
    }

    if( !emitterData->ptcl.isRotateX )
    {
        emitterData->staticUbo.rotateInit.x = 0.0f;
        emitterData->staticUbo.rotateInitRand.x = 0.0f;
        emitterData->staticUbo.rotateAdd.x = 0.0f;
        emitterData->staticUbo.rotateAddRand.x = 0.0f;
    }
    if( !emitterData->ptcl.isRotateY )
    {
        emitterData->staticUbo.rotateInit.y = 0.0f;
        emitterData->staticUbo.rotateInitRand.y = 0.0f;
        emitterData->staticUbo.rotateAdd.y = 0.0f;
        emitterData->staticUbo.rotateAddRand.y = 0.0f;
    }
    if( !emitterData->ptcl.isRotateZ )
    {
        emitterData->staticUbo.rotateInit.z = 0.0f;
        emitterData->staticUbo.rotateInitRand.z = 0.0f;
        emitterData->staticUbo.rotateAdd.z = 0.0f;
        emitterData->staticUbo.rotateAddRand.z = 0.0f;
    }

    // 8キーアニメ、未使用キーを最終キーの値でクランプ
    if( emitterData->staticUbo.color0AnimKeynum > 0 )
    {
        for( int i = emitterData->staticUbo.color0AnimKeynum; i < 8; i++ )
        {
            emitterData->staticUbo.color0Anim.value[ i ] = emitterData->staticUbo.color0Anim.value[ emitterData->staticUbo.color0AnimKeynum - 1 ];
        }
    }
    if( emitterData->staticUbo.color1AnimKeynum > 0 )
    {
        for( int i = emitterData->staticUbo.color1AnimKeynum; i < 8; i++ )
        {
            emitterData->staticUbo.color1Anim.value[ i ] = emitterData->staticUbo.color1Anim.value[ emitterData->staticUbo.color1AnimKeynum - 1 ];
        }
    }
    if( emitterData->staticUbo.alpha0AnimKeyNum > 0 )
    {
        for( int i = emitterData->staticUbo.alpha0AnimKeyNum; i < 8; i++ )
        {
            emitterData->staticUbo.alpha0Anim.value[ i ] = emitterData->staticUbo.alpha0Anim.value[ emitterData->staticUbo.alpha0AnimKeyNum - 1 ];
        }
    }
    if( emitterData->staticUbo.alpha1AnimKeyNum > 0 )
    {
        for( int i = emitterData->staticUbo.alpha1AnimKeyNum; i < 8; i++ )
        {
            emitterData->staticUbo.alpha1Anim.value[ i ] = emitterData->staticUbo.alpha1Anim.value[ emitterData->staticUbo.alpha1AnimKeyNum - 1 ];
        }
    }
    if( emitterData->staticUbo.scaleAnimKeyNum > 0 )
    {
        for( int i = emitterData->staticUbo.scaleAnimKeyNum; i < 8; i++ )
        {
            emitterData->staticUbo.scale8keyAnim.value[ i ] = emitterData->staticUbo.scale8keyAnim.value[ emitterData->staticUbo.scaleAnimKeyNum - 1 ];
        }
    }
    if( emitterData->staticUbo.shaderAnimKeyNum > 0 )
    {
        for( int i = emitterData->staticUbo.shaderAnimKeyNum; i < 8; i++ )
        {
            emitterData->staticUbo.shaderAnim.value[ i ] = emitterData->staticUbo.shaderAnim.value[ emitterData->staticUbo.shaderAnimKeyNum - 1 ];
        }
    }

    // 8キーアニメのループ対応
    if( emitterData->ptcl.color0AnimIsLoop )
    {
        emitterData->staticUbo.color0AnimLoopRate = static_cast< f32 >( emitterData->ptcl.color0AnimLoopRate );
    }
    else
    {
        emitterData->staticUbo.color0AnimLoopRate = 0.0f;
    }
    emitterData->staticUbo.color0AnimIsLoopInitRandom = ( emitterData->ptcl.color0AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if( emitterData->ptcl.alpha0AnimIsLoop )
    {
        emitterData->staticUbo.alpha0AnimLoopRate = static_cast< f32 >( emitterData->ptcl.alpha0AnimLoopRate );
    }
    else
    {
        emitterData->staticUbo.alpha0AnimLoopRate = 0.0f;
    }
    emitterData->staticUbo.alpha0AnimIsLoopInitRandom = ( emitterData->ptcl.alpha0AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if( emitterData->ptcl.color1AnimIsLoop )
    {
        emitterData->staticUbo.color1AnimLoopRate = static_cast< f32 >( emitterData->ptcl.color1AnimLoopRate );
    }
    else
    {
        emitterData->staticUbo.color1AnimLoopRate = 0.0f;
    }
    emitterData->staticUbo.color1AnimIsLoopInitRandom = ( emitterData->ptcl.color1AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if( emitterData->ptcl.alpha1AnimIsLoop )
    {
        emitterData->staticUbo.alpha1AnimLoopRate = static_cast< f32 >( emitterData->ptcl.alpha1AnimLoopRate );
    }
    else
    {
        emitterData->staticUbo.alpha1AnimLoopRate = 0.0f;
    }
    emitterData->staticUbo.alpha1AnimIsLoopInitRandom = ( emitterData->ptcl.alpha1AnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    //
    if( emitterData->ptcl.scaleAnimIsLoop )
    {
        emitterData->staticUbo.scaleAnimLoopRate = static_cast< f32 >( emitterData->ptcl.scaleAnimLoopRate );
    }
    else
    {
        emitterData->staticUbo.scaleAnimLoopRate = 0.0f;
    }
    emitterData->staticUbo.scaleAnimIsLoopInitRandom = ( emitterData->ptcl.scaleAnimIsLoopInitRandom ) ? 1.0f : 0.0f;

    // 暫定処置:
    // プリミティブ型のエミッタ形状以外の場合は、放出タイプを無効化する
    if( !( emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE ||
        emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE ||
        emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE ||
        emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE64 ||
        emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_PRIMITIVE ) )
    {
        emitterData->volume.primEmitType = static_cast< u32 >( EFT_EMIT_FROM_PRIMITIVE_INVALID );
    }

    //「球(分割)」、「球(分割64)」対応
    if( emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE )
    {
        if( emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
        {
            // UI上のIndex値から放出レートにマッピング。
            const f32 pattern[] =
            {
                2, 3, 4, 6, 8, 12, 20, 32
            };
            emitterData->emission.rate = pattern[ emitterData->volume.volumeTblIndex ];
        }
        else
        {
            // それ以外の場合はMakerからの値をそのまま採用する
        }
    }
    else if( emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE64 )
    {
        if( emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
        {
            // UI上のIndex値を加算して放出レートに合わせる。
            emitterData->emission.rate = emitterData->volume.volumeTblIndex64;
        }
        else
        {
            // それ以外の場合はMakerからの値をそのまま採用する
        }
    }
    else if( emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_PRIMITIVE )
    {
        // 入力されたプリミティブの頂点数に合わせる。
        if( volumePrimitive )
        {
            // 放出タイプで放出数も切り替える。
            switch( emitterData->volume.primEmitType )
            {
            case EFT_EMIT_FROM_PRIMITIVE_UNISON:
                emitterData->emission.rate = static_cast< f32 >( volumePrimitive->GetVertexNum() );
                break;
            }
        }
        else
        {
            emitterData->emission.rate = 1.0;
        }
    }
    else if( emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE ||
             emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE )
    {
        if( emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
        {
            // それ以外：円等分割 or 線等分割 で、かつ一斉放出の場合、
            // 放出レートを1で上書き（EffectMaker上での固定がやや高コストなので、こちらでクッションしておく）
            emitterData->emission.rate = 1.0f;
        }
    }

    // シェーダフラグ
    ShaderFlag shaderFlag;
    shaderFlag.Invalidate();
    shaderFlag.Initialize( this );

    // フラグを書き込む
    emitterData->staticUbo.flag[ 0 ] = shaderFlag.GetFlag0();
    emitterData->staticUbo.flag[ 1 ] = shaderFlag.GetFlag1();

    // カスタムフィールドフラグ
    if ( fieldCustomData )
    {
        emitterData->staticUbo.flag[ 3 ] = fieldCustomData->flags;
    }

    // リソース内エミッタマトリクス
    nw::math::MTX34MakeSRT( &resMatrixSRT,
        &emitterData->emitter.scale,
        &emitterData->emitter.rotate,
        &emitterData->emitter.trans );

    nw::math::VEC3 scale;
    scale.Set( 1.0f, 1.0f, 1.0f );
    nw::math::MTX34MakeSRT( &resMatrixRT,
        &scale,
        &emitterData->emitter.rotate,
        &emitterData->emitter.trans );

    // テクスチャサンプラ―を初期化
    textureSampler[ EFT_TEXTURE_SLOT_0 ].Initialize(
        &emitterData->textureSampler0 );
    textureSampler[ EFT_TEXTURE_SLOT_1 ].Initialize(
        &emitterData->textureSampler1 );
    textureSampler[ EFT_TEXTURE_SLOT_2 ].Initialize(
        &emitterData->textureSampler2 );

#if 0   // フィールドのアニメを８キー固定にしたので、いったんOFF
    if ( fieldPosAddData )
    {
        ResAnim8KeyParamSetAdapter adapter( &fieldPosAddData->animParam );
        adapter.FlipEndian();
        fieldPosAddData->animParam.keyValue = adapter.GetKeyValue();
    }

    if ( fieldRandomData )
    {
        ResAnim8KeyParamSetAdapter adapter( &fieldRandomData->animParam );
        adapter.FlipEndian();
        fieldRandomData->animParam.keyValue = adapter.GetKeyValue();
    }

    if ( fieldMagnetData )
    {
        ResAnim8KeyParamSetAdapter adapter( &fieldMagnetData->animParam );
        adapter.FlipEndian();
        fieldMagnetData->animParam.keyValue = adapter.GetKeyValue();
    }

    if ( fieldConvergenceData )
    {
        ResAnim8KeyParamSetAdapter adapter( &fieldConvergenceData->animParam );
        adapter.FlipEndian();
        fieldConvergenceData->animParam.keyValue = adapter.GetKeyValue();
    }

    if ( fieldSpinData )
    {
        ResAnim8KeyParamSetAdapter adapter( &fieldSpinData->outerAnimParam );
        adapter.FlipEndian();
        fieldSpinData->outerAnimParam.keyValue = adapter.GetKeyValue();
        adapter = ResAnim8KeyParamSetAdapter( &fieldSpinData->rotateAnimParam );
        adapter.FlipEndian();
        fieldSpinData->rotateAnimParam.keyValue = adapter.GetKeyValue();
    }
#endif
    // フィールドのUBOを作成する
    if( fieldUniformBlock )
    {
        if( fieldRandomData )
        {
            fieldUniformBlock->randomParam.x = fieldRandomData->randomVel.x;
            fieldUniformBlock->randomParam.y = fieldRandomData->randomVel.y;
            fieldUniformBlock->randomParam.z = fieldRandomData->randomVel.z;
            fieldUniformBlock->randomParam.w = static_cast< f32 >( fieldRandomData->blank );    // interval

            fieldUniformBlock->randomParam1.x = fieldRandomData->waveParamHzRate0;
            fieldUniformBlock->randomParam1.y = fieldRandomData->waveParamHzRate1;
            fieldUniformBlock->randomParam1.z = fieldRandomData->waveParamHzRate2;
            fieldUniformBlock->randomParam1.w = fieldRandomData->waveParamHzRate3;

            fieldUniformBlock->randomParam2.x = fieldRandomData->waveParam0;
            fieldUniformBlock->randomParam2.y = fieldRandomData->waveParam1;
            fieldUniformBlock->randomParam2.z = fieldRandomData->waveParam2;
            fieldUniformBlock->randomParam2.w = fieldRandomData->waveParam3;

            fieldUniformBlock->randomParam3.x = static_cast< f32 >( fieldRandomData->enableUnifiedPhase );
            fieldUniformBlock->randomParam3.y = static_cast< f32 >( fieldRandomData->enableDetailedOption );
            fieldUniformBlock->randomParam3.w = static_cast< f32 >( fieldRandomData->enableAirRegist );

            fieldUniformBlock->randomParam4.x = fieldRandomData->unifiedPhaseSpeed;
            fieldUniformBlock->randomParam4.y = fieldRandomData->unifiedPhaseDistribution;
        }
        if( fieldRandomDataFe1 )
        {
            fieldUniformBlock->randomFe1Param.x = fieldRandomDataFe1->randomVel.x;
            fieldUniformBlock->randomFe1Param.y = fieldRandomDataFe1->randomVel.y;
            fieldUniformBlock->randomFe1Param.z = fieldRandomDataFe1->randomVel.z;
            fieldUniformBlock->randomFe1Param.w = static_cast< f32 >( fieldRandomDataFe1->blank );
        }
        if( fieldPosAddData )
        {
            fieldUniformBlock->posAddParam.x = fieldPosAddData->fieldPosAdd.x;
            fieldUniformBlock->posAddParam.y = fieldPosAddData->fieldPosAdd.y;
            fieldUniformBlock->posAddParam.z = fieldPosAddData->fieldPosAdd.z;
            fieldUniformBlock->posAddParam.w = static_cast< f32 >( fieldPosAddData->isFieldPosAddGlobal );
        }
        if( fieldMagnetData )
        {
            fieldUniformBlock->magnetParam.x = fieldMagnetData->fieldMagnetPos.x;
            fieldUniformBlock->magnetParam.y = fieldMagnetData->fieldMagnetPos.y;
            fieldUniformBlock->magnetParam.z = fieldMagnetData->fieldMagnetPos.z;
            fieldUniformBlock->magnetParam.w = fieldMagnetData->fieldMagnetPower;

            fieldUniformBlock->magnetParam1.x = static_cast< f32 >( fieldMagnetData->isFollowEmitter );
        }
        if( fieldConvergenceData )
        {
            fieldUniformBlock->convergenceParam.x = fieldConvergenceData->fieldConvergencePos.x;
            fieldUniformBlock->convergenceParam.y = fieldConvergenceData->fieldConvergencePos.y;
            fieldUniformBlock->convergenceParam.z = fieldConvergenceData->fieldConvergencePos.z;
            fieldUniformBlock->convergenceParam.w = fieldConvergenceData->fieldConvergenceRatio;

            fieldUniformBlock->convergenceParam1.x = static_cast< f32 >( fieldConvergenceData->fieldConvergenceType );
        }
        if( fieldSpinData )
        {
            fieldUniformBlock->spinParam.x = fieldSpinData->fieldSpinRotate;
            fieldUniformBlock->spinParam.y = fieldSpinData->fieldSpinOuter;
            fieldUniformBlock->spinParam.z = static_cast< f32 >( fieldSpinData->fieldSpinAxis );
        }
        if( fieldCollisionData )
        {
            fieldUniformBlock->collisionParam.x = ( fieldCollisionData->fieldCollisionType * 2 ) + static_cast< f32 >( fieldCollisionData->fieldCollisionIsWorld );
            fieldUniformBlock->collisionParam.y = fieldCollisionData->fieldCollisionCoef;
            fieldUniformBlock->collisionParam.z = fieldCollisionData->fieldCollisionRegist;
            fieldUniformBlock->collisionParam.w = fieldCollisionData->fieldCollisionCoord;
        }
        if( fieldCurlNoiseData )
        {
            fieldUniformBlock->curlNoiseParam.x = fieldCurlNoiseData->fieldCurlNoiseInfluence.x;
            fieldUniformBlock->curlNoiseParam.y = fieldCurlNoiseData->fieldCurlNoiseInfluence.y;
            fieldUniformBlock->curlNoiseParam.z = fieldCurlNoiseData->fieldCurlNoiseInfluence.z;
            fieldUniformBlock->curlNoiseParam.w = fieldCurlNoiseData->fieldCurlNoiseScale;

            fieldUniformBlock->curlNoiseParam1.x = fieldCurlNoiseData->fieldCurlNoiseSpeed.x;
            fieldUniformBlock->curlNoiseParam1.y = fieldCurlNoiseData->fieldCurlNoiseSpeed.y;
            fieldUniformBlock->curlNoiseParam1.z = fieldCurlNoiseData->fieldCurlNoiseSpeed.z;
            fieldUniformBlock->curlNoiseParam1.w = fieldCurlNoiseData->fieldCurlNoiseBase;

            fieldUniformBlock->curlNoiseParam2.x = fieldCurlNoiseData->isFieldCurlNoiseBaseRandom;
            fieldUniformBlock->curlNoiseParam2.y = fieldCurlNoiseData->isWorldCoordinate;
        }
        if( fieldCustomData )
        {
            fieldUniformBlock->customFieldParam0.x = fieldCustomData->value0;
            fieldUniformBlock->customFieldParam0.y = fieldCustomData->value1;
            fieldUniformBlock->customFieldParam0.z = fieldCustomData->value2;
            fieldUniformBlock->customFieldParam0.w = fieldCustomData->value3;

            fieldUniformBlock->customFieldParam1.x = fieldCustomData->value4;
            fieldUniformBlock->customFieldParam1.y = fieldCustomData->value5;
            fieldUniformBlock->customFieldParam1.z = fieldCustomData->value6;
            fieldUniformBlock->customFieldParam1.w = fieldCustomData->value7;
        }
    }

    // CPU参照用にコピー
    if( emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
    {
        memcpy( resEmitterStaticUniformBlock, &emitterData->staticUbo, sizeof( EmitterStaticUniformBlock ) );
    }

    rotateInit.x = emitterData->staticUbo.rotateInit.x;
    rotateInit.y = emitterData->staticUbo.rotateInit.y;
    rotateInit.z = emitterData->staticUbo.rotateInit.z;
}

//------------------------------------------------------------------------------
//  EmitterResource を初期化する
//------------------------------------------------------------------------------
void Resource::InitializeEmitterResource( EmitterResource* emitterRes, BinaryData* resEmitter, u32 idx )
{
    emitterRes->emitterIndex = idx;

    // 表示/非表示
    emitterRes->visibility = true;

    // ResEmitter アドレス解決
    emitterRes->emitterData = resEmitter->GetBinaryDataWithFlip<ResEmitter>();

    if ( emitterRes->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
    {
        // CPUエミッタでは参照用にStaticUboをコピーする
        u32 allocSize = sizeof(EmitterStaticUniformBlock);
        allocSize = nw::eft2::RoundUp( allocSize, 32 );
        emitterRes->resEmitterStaticUniformBlock   = (EmitterStaticUniformBlock*)m_ResHeap.Alloc( allocSize );
    }
    else
    {
        emitterRes->resEmitterStaticUniformBlock = &emitterRes->emitterData->staticUbo;
    }

    // 挙動タイプ切り替え
    //emitterRes->emitterData->emitter.calcType = EFT_PARTICLE_CALC_TYPE_CPU;
    emitterRes->emitterData->emitter.calcType = emitterRes->emitterData->volume.isGpuEmitter;

    // テクスチャリソース初期化
    emitterRes->texture[EFT_TEXTURE_SLOT_0] = NULL;
    emitterRes->texture[EFT_TEXTURE_SLOT_1] = NULL;
    emitterRes->texture[EFT_TEXTURE_SLOT_2] = NULL;
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_0].Invalidate();
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_1].Invalidate();
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_2].Invalidate();

    // テクスチャリソースをguidで検索して設定
    emitterRes->texture[EFT_TEXTURE_SLOT_0] = GetTexture( emitterRes->emitterData->textureSampler0.guid );
    emitterRes->texture[EFT_TEXTURE_SLOT_1] = GetTexture( emitterRes->emitterData->textureSampler1.guid );
    emitterRes->texture[EFT_TEXTURE_SLOT_2] = GetTexture( emitterRes->emitterData->textureSampler2.guid );

    // プリミティブリソースをindexで検索して設定
    emitterRes->primitive = GetPrimitive( emitterRes->emitterData->ptcl.primitiveIndex );

    emitterRes->volumePrimitive       = NULL;
    emitterRes->volumePrimitive       = GetPrimitive( emitterRes->emitterData->volume.primitiveIndex );

    // サブデータのアドレス解決
    emitterRes->fieldRandomData       = NULL;
    emitterRes->fieldRandomDataFe1    = NULL;
    emitterRes->fieldMagnetData       = NULL;
    emitterRes->fieldSpinData         = NULL;
    emitterRes->fieldCollisionData    = NULL;
    emitterRes->fieldConvergenceData  = NULL;
    emitterRes->fieldPosAddData       = NULL;
    emitterRes->fieldCurlNoiseData    = NULL;
    emitterRes->fieldCustomData       = NULL;
    for( u32 i = 0; i < EFT_EMITTER_ANIM_MAX; i++ )
    {
        emitterRes->emitterAnimArray[i] = NULL;
    }

    // フィールドが存在するエミッタかどうか？
    bool isUseField = false;
    emitterRes->fieldUniformBlock     = NULL;
    emitterRes->customShaderParam     = NULL;
    emitterRes->customDataParam       = NULL;
    emitterRes->customActionParam     = NULL;
//    ResAnim8KeyParamSetAdapter adapter;

    emitterRes->emitterPluginIndex      = 0;
    emitterRes->emitterPluginData       = NULL;

    emitterRes->isUseEmitterAnim        = false;
    emitterRes->isEmitterSRTAnim        = false;

    // カスタムシェーダパラメータのヘッダのフリップフラグを立てるため外出し
    BinaryData* csdpSub = NULL;
#ifndef EFT_BI_ENDIAN
    NW_UNUSED_VARIABLE(csdpSub);
#endif

    if ( resEmitter->GetSubData() )
    {
        BinaryData* sub = resEmitter->GetSubData();

        while( sub )
        {
            switch ( sub->GetBinaryTag() )
            {
            // フィールド：ランダム（新方式）
            case EFT_MAKE_TAG( 'F', 'R', 'N', 'D' ):
                emitterRes->fieldRandomData = sub->GetBinaryDataWithFlip<ResFieldRandom>();
                isUseField = true;
                break;
            // フィールド：ランダム（FE1互換）
            case EFT_MAKE_TAG( 'F', 'R', 'N', '1' ):
                emitterRes->fieldRandomDataFe1 = sub->GetBinaryDataWithFlip<ResFieldRandomFe1>();
                isUseField = true;
                break;
            // フィールド：磁力
            case EFT_MAKE_TAG( 'F', 'M', 'A', 'G' ):
                emitterRes->fieldMagnetData = sub->GetBinaryDataWithFlip<ResFieldMagnet>();
                isUseField = true;
                break;
            // フィールド：スピン
            case EFT_MAKE_TAG( 'F', 'S', 'P', 'N' ):
                emitterRes->fieldSpinData = sub->GetBinaryDataWithFlip<ResFieldSpinData>();
                isUseField = true;
                break;
            // フィールド：シンプルコリジョン
            case EFT_MAKE_TAG( 'F', 'C', 'O', 'L' ):
                emitterRes->fieldCollisionData = sub->GetBinaryDataWithFlip<ResFieldCollisionData>();
                isUseField = true;
                break;
            // フィールド：収束
            case EFT_MAKE_TAG( 'F', 'C', 'O', 'V' ):
                emitterRes->fieldConvergenceData = sub->GetBinaryDataWithFlip<ResFieldConvergenceData>();
                isUseField = true;
                break;
            // フィールド：カールノイズ
            case EFT_MAKE_TAG( 'F', 'C', 'L', 'N' ):
                emitterRes->fieldCurlNoiseData = sub->GetBinaryDataWithFlip<ResFieldCurlNoiseData>();
                isUseField = true;
                break;
            // フィールド：位置に加算
            case EFT_MAKE_TAG( 'F', 'P', 'A', 'D' ):
                emitterRes->fieldPosAddData = sub->GetBinaryDataWithFlip<ResFieldPosAddData>();
                isUseField = true;
                break;
            // フィールド：カスタム
            case EFT_MAKE_TAG( 'F', 'C', 'S', 'F' ):
                emitterRes->fieldCustomData = sub->GetBinaryDataWithFlip<ResFieldCustom>();
                isUseField = true;
                break;

            // エミッタプラグイン / 連結式ストライプ
            case EFT_MAKE_TAG( 'E', 'P', '0', '1' ):
                {
                    ResStripeConnection* data = sub->GetBinaryDataWithFlip<ResStripeConnection>();
                    emitterRes->emitterPluginData = data;
                    // プラグインインデックス
                    emitterRes->emitterPluginIndex = ConnectionStripeSystem::PluginId;
                }
                break;
            // エミッタプラグイン / 履歴式ストライプ
            case EFT_MAKE_TAG( 'E', 'P', '0', '2' ):
                {
                    ResStripeHistory* data = sub->GetBinaryDataWithFlip<ResStripeHistory>();
                    emitterRes->emitterPluginData = data;
                    // プラグインインデックス
                    emitterRes->emitterPluginIndex = StripeSystem::PluginId;
                }
                break;
            // エミッタプラグイン / スーパーストライプ
            case EFT_MAKE_TAG( 'E', 'P', '0', '3' ):
                {
                    ResStripeSuper* data = sub->GetBinaryDataWithFlip<ResStripeSuper>();
                    emitterRes->emitterPluginData = data;
                    // プラグインインデックス
                    emitterRes->emitterPluginIndex = SuperStripeSystem::PluginId;
                }
                break;

            // エミッタプラグイン / 範囲内ループ
            case EFT_MAKE_TAG( 'E', 'P', '0', '4' ):
                {
                    ResEPAreaLoop* data = sub->GetBinaryDataWithFlip<ResEPAreaLoop>();
                    emitterRes->emitterPluginData = data;
                    // プラグインインデックス
                    emitterRes->emitterPluginIndex = AreaLoopSystem::PluginId;
                }
                break;

            // カスタムデータパラメータ
            case EFT_MAKE_TAG( 'C', 'U', 'D', 'P' ):
                {
                    emitterRes->customDataParam = sub->GetBinaryData();
                    if( sub->IsNeedFlip() )
                    {
                        u32 size = static_cast< u32 >( sub->GetBinarySize() - 32 );
                        {
                            u32* temp = reinterpret_cast< u32* >( emitterRes->customDataParam );
                            int count = static_cast< int >( size  / 4 );
                            for( int i = 0; i < count; i++ )
                            {
                                EndianUtil::Flip( temp );
                                temp++;
                            }
                        }
                        sub->flipped = static_cast<u8>( EFT_BINARY_DATA_FLIPPED );
                    }
                }
                break;

            // カスタムアクションパラメータ
            case EFT_MAKE_TAG( 'C', 'A', 'D', 'P' ):
                {
                    emitterRes->customActionParam = sub->GetBinaryData();
                    if ( sub->IsNeedFlip() )
                    {
                        u32 size = sub->GetBinarySize() - 32;
                        {
                            u32* temp = reinterpret_cast< u32* >( emitterRes->customActionParam );
                            int count = static_cast< int >( size  / 4 );
                            for( int i = 0; i < count; i++ )
                            {
                                EndianUtil::Flip( temp );
                                temp++;
                            }
                        }
                        sub->flipped = static_cast<u8>(EFT_BINARY_DATA_FLIPPED);
                    }
                }
                break;

            // カスタムシェーダパラメータ
            case EFT_MAKE_TAG( 'C', 'S', 'D', 'P' ):
                {
                    emitterRes->customShaderParam       = sub->GetBinaryData();
                    emitterRes->customShaderParamSize   = sub->GetBinarySize() - 32;
                    csdpSub = sub;
                }
                break;

            // エミッタ時間アニメ : パーティクルスケールアニメ配列
            case EFT_MAKE_TAG( 'E', 'A', 'S', 'L' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_PTCL_SCALE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : エミッタスケールアニメ配列
            case EFT_MAKE_TAG( 'E', 'A', 'E', 'S' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_SCALE] = GetEmitterAnimationTable( sub );
                    emitterRes->isEmitterSRTAnim = true;
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : エミッタ回転アニメ配列
            case EFT_MAKE_TAG( 'E', 'A', 'E', 'R' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_ROTATE] = GetEmitterAnimationTable( sub );
                    emitterRes->isEmitterSRTAnim = true;
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : エミッタ位置アニメ配列
            case EFT_MAKE_TAG( 'E', 'A', 'E', 'T' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_TRANS] = GetEmitterAnimationTable( sub );
                    emitterRes->isEmitterSRTAnim = true;
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : 放出レート
            case EFT_MAKE_TAG( 'E', 'A', 'T', 'R' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_EMISSION_RATE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : 初速／全方向初速
            case EFT_MAKE_TAG( 'E', 'A', 'O', 'V' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_ALL_DIR_VEL] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : 初速／指定方向初速
            case EFT_MAKE_TAG( 'E', 'A', 'D', 'V' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_DESIGNATED_DIR_SCALE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : その他運動設定／重力
            case EFT_MAKE_TAG( 'E', 'A', 'G', 'V' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_GRAVITY_SCALE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : パーティクル寿命／寿命
            case EFT_MAKE_TAG( 'E', 'A', 'P', 'L' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_PTCL_LIFE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : グローバルカラー0
            case EFT_MAKE_TAG( 'E', 'A', 'C', '0' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_COLOR0] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : グローバルカラー1
            case EFT_MAKE_TAG( 'E', 'A', 'C', '1' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_COLOR1] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : グローバルアルファ0
            case EFT_MAKE_TAG( 'E', 'A', 'A', '0' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_ALPHA0] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : グローバルアルファ1
            case EFT_MAKE_TAG( 'E', 'A', 'A', '1' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_ALPHA1] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;

            // エミッタ時間アニメ : エミッタボリュームスケール
            case EFT_MAKE_TAG( 'E', 'A', 'S', 'S' ):
                {
                    emitterRes->emitterAnimArray[EFT_EMITTER_ANIM_VOLUME_SCALE] = GetEmitterAnimationTable( sub );
                    emitterRes->isUseEmitterAnim = true;
                }
                break;
            }

            sub = sub->GetNextData();
        }
    }

    // フィールドのUBO用ワークを確保
    if ( isUseField )
    {
        emitterRes->fieldUniformBlock = reinterpret_cast<EmitterFieldUniformBlock*>(
                                            m_ResHeap.Alloc( sizeof(EmitterFieldUniformBlock), 0x100 ) );
        m_UboAllocedSize += sizeof(EmitterFieldUniformBlock);
        MemUtil::FillZero( emitterRes->fieldUniformBlock, sizeof(EmitterFieldUniformBlock) );
    }

    // リソースアップデート
    emitterRes->Update();

    // エミッタ静的ユニフォームブロックを初期化
    ResCommon tempCommon;
    MemUtil::Copy( &tempCommon, emitterRes->emitterData, sizeof( ResCommon ) );
    {
        emitterRes->emitterStaticUbo.Set( emitterRes->emitterData, sizeof( ResCommon ) + sizeof( EmitterStaticUniformBlock ) );
        emitterRes->emitterStaticUbo.Validate();
    }
    MemUtil::Copy( emitterRes->emitterData, &tempCommon, sizeof( ResCommon ) );


    if ( emitterRes->fieldUniformBlock )
    {
        emitterRes->fieldStaticUbo.Set( emitterRes->fieldUniformBlock, sizeof( EmitterFieldUniformBlock ) );
        emitterRes->fieldStaticUbo.Validate();
    }

    // ユーザー定義データのエンディアンフリップを行う
#ifdef  EFT_BI_ENDIAN
    if ( emitterRes->customShaderParam && csdpSub != NULL )
    {
        if ( csdpSub->IsNeedFlip() )
        {
            u32 id = emitterRes->emitterData->shader.customShaderIndex + EFT_CUSTOM_ACTION_CALLBACK_ID_7;
            CallbackSet* cb = m_System->GetCallbackSet( static_cast<CallBackID>( id ) );
            if ( cb )
            {
                EndianFlipArg arg;
                arg.customShaderParam       = emitterRes->customShaderParam;
                arg.customShaderParamSize   = emitterRes->customShaderParamSize;

                if ( cb->endianFlip )
                {
                    cb->endianFlip( arg );
                }
                else
                {
                    _EndianFlipCallback( arg );
                }
            }

            csdpSub->flipped = EFT_BINARY_DATA_FLIPPED;
        }
    }
#endif//EFT_BI_ENDIAN
}


//------------------------------------------------------------------------------
//  EmitterResource の終了処理をする
//------------------------------------------------------------------------------
void Resource::FinalizeEmitterResource( EmitterResource* emitterRes )
{
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_0].Finalize();
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_1].Finalize();
    emitterRes->textureSampler[EFT_TEXTURE_SLOT_2].Finalize();

    if ( emitterRes->fieldUniformBlock )
    {
        m_ResHeap.Free( emitterRes->fieldUniformBlock );
    }

#if defined(EFT_IS_CAFE)
    ResCommon tempCommon;
    MemUtil::Copy( &tempCommon, emitterRes->emitterData, sizeof( ResCommon ) );
    {
        emitterRes->emitterStaticUbo.Set( emitterRes->emitterData, sizeof( ResCommon ) + sizeof( EmitterStaticUniformBlock ) );
        emitterRes->emitterStaticUbo.Validate();
    }
    MemUtil::Copy( emitterRes->emitterData, &tempCommon, sizeof( ResCommon ) );
#endif
}


//------------------------------------------------------------------------------
//  テクスチャを検索する
//------------------------------------------------------------------------------
TextureResource* Resource::GetTexture( u64 guid )
{
    if ( guid == static_cast<u64>( EFT_INVALID_GLOBAL_ID ) ) return NULL;

    for ( u32 i = 0; i < m_TextureCount; i++ )
    {
        if ( m_TextureArray[i].GetGlobalID() == guid )
        {
            return &m_TextureArray[i];
        }
    }

    if ( m_pResidentResource )
    {
        TextureResource* retTexture = m_pResidentResource->GetTexture( guid );
        if ( retTexture )
        {
            return retTexture;
        }
        else
        {
            OutputLog( "Appointed Texture is not found from Resident Resource.\n" );
        }
    }

    return NULL;
}

//------------------------------------------------------------------------------
//  プリミティブを検索する
//------------------------------------------------------------------------------
Primitive* Resource::GetPrimitive( u64 guid )
{
    if ( guid == static_cast<u64>( EFT_INVALID_GLOBAL_ID ) ) return NULL;

    for ( u32 i = 0; i < m_PrimitiveCount; i++ )
    {
        if ( m_PrimitiveArray[i].GetGlobalID() == guid )
        {
            return &m_PrimitiveArray[i];
        }
    }

    if ( m_pResidentResource )
    {
        Primitive* retPrimitive = m_pResidentResource->GetPrimitive( guid );
        if ( retPrimitive )
        {
            return retPrimitive;
        }
        else
        {
            OutputLog( "Appointed Primitive is not found from Resident Resource.\n" );
        }
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  エミッタセットIDの検索を行います。
//---------------------------------------------------------------------------
s32 Resource::SearchEmitterSetID( const char* name ) const
{
    for ( u32 i = 0; i < m_EmitterSetCount; i++ )
    {
        if( strcmp( name, m_EmitterSetResArray[i].emitterSetRes->name ) == 0 ) return i;
    }

    return EFT_INVALID_EMITTER_SET_ID;
}

//---------------------------------------------------------------------------
//  sourceResで指定されるリソースに繋ぎ換えます。
//---------------------------------------------------------------------------
bool Resource::BindResource( s32 targetSetID, EmitterSetResource* emitterSetResource )
{
    if ( m_EmitterSetCount <= static_cast<u32>( targetSetID ) ) return false;

    m_BindEmitterSetResArray[targetSetID] = emitterSetResource;

    // 該当エミッタセットを再生成する
    bool ret = m_System->ReCreateEmitterSet( m_ResourceID , targetSetID );
    if ( !ret ) return false;

    return true;
}

//---------------------------------------------------------------------------
//  targetSetIDで指定されるエミッタセットリソースの繋ぎ換えを解除します。
//---------------------------------------------------------------------------
bool Resource::UnbindResource( s32 targetSetID )
{
    if ( m_EmitterSetCount <= static_cast<u32>( targetSetID ) ) return false;

    m_BindEmitterSetResArray[targetSetID] = NULL;
    // 該当エミッタセットを再生成する
    bool ret = m_System->ReCreateEmitterSet( m_ResourceID , targetSetID );
    if ( !ret ) return false;

    return true;
}

//---------------------------------------------------------------------------
//  チャイルドエミッタが存在するかどうか。
//---------------------------------------------------------------------------
bool Resource::IsExistChildEmitter( s32 targetSetID )
{
    EmitterSetResource* emSetRes = GetEmitterSetResource( targetSetID );
    if ( !emSetRes ) return false;

    bool ret = false;
    for ( u32 i = 0; i < emSetRes->emitterNum; i++ )
    {
        if ( emSetRes->emitterResSet[i].childEmitterResSet[0] )
        {
            ret = true;
        }
    }
    return ret;
}

//---------------------------------------------------------------------------
//  指定EmitterSetが無限寿命、または、ループエフェクトかを取得する。
//---------------------------------------------------------------------------
bool Resource::IsNeedFade( s32 emitterSetID ) const
{
    EmitterSetResource* emSetRes = GetEmitterSetResource( emitterSetID );
    if ( !emSetRes ) return false;

    return ( emSetRes->isLoopEffect || emSetRes->isLifeInfinity );
}

#if EFT_GX2
//---------------------------------------------------------------------------
//  シェーダバイナリを配置前に戻す。
//---------------------------------------------------------------------------
void Resource::Unrelocate( void* pData )
{
    void* treeTop = ( reinterpret_cast<u8*>(pData) + sizeof( ResHeader ) );
    BinaryData* temp = reinterpret_cast<BinaryData*>(treeTop);

    while( temp )
    {
        u32 binarySize = temp->GetBinarySize();
        u32 binaryOffset = temp->GetNextBinaryOffset();

        // ShaderArray
        if ( temp->GetBinaryTag() == EFT_MAKE_TAG( 'S', 'H', 'D', 'A' ) )
        {
            BinaryData* shaderData = temp->GetDirectChildData( EFT_MAKE_TAG( 'S', 'H', 'D', 'B' ) );
            ShaderManager::Unrelocate( shaderData->GetBinaryData() );
        }

        temp = temp->GetNextData();
    }
}
#endif
//---------------------------------------------------------------------------
//  リソースサイズの詳細をログ出力します。
//---------------------------------------------------------------------------
void Resource::OutputResourceInfo()
{
    OutputLog( "-----ResourceInfo:%d-----\n", m_ResourceID );
    OutputLog( "0-EmitterSet     Name:%s\n",  m_EmitterSetResArray[0].emitterSetRes->name );
    OutputLog( "EmitterSet       Num :%8d\n", m_EmitterSetCount );
    OutputLog( "Emitter          Num :%8d\n", m_EmitterCount );
    OutputLog( "                 Size:%8d\n", m_EmitterResourceSize );
    OutputLog( "Shader Variation Num :%8d\n", m_ShaderCount );
#if EFT_IS_CAFE
    OutputLog( "Vertex Shader    Num :%8d\n", m_ShaderManager.GetVertexShaderNum() );
    OutputLog( "                 Size:%8d\n", m_ShaderManager.GetVertexShaderSize() );
    OutputLog( "Fragment Shader  Num :%8d\n", m_ShaderManager.GetFragmentShaderNum() );
    OutputLog( "                 Size:%8d\n", m_ShaderManager.GetFragmentShaderSize() );
#else
    OutputLog( "ShaderCode       Size:%8d\n", m_ShaderCodeResourceSize );
#endif
    OutputLog( "Texture          Num :%8d\n", m_TextureCount );
    OutputLog( "                 Size:%8d\n", m_TextureResourceSize );
    OutputLog( "Primitive        Num :%8d\n", m_PrimitiveCount );
    OutputLog( "                 Size:%8d\n", m_PrimitiveResourceSize );
    OutputLog( "------------------------\n" );
    OutputLog( "------ResHeap Info------\n",  m_ResourceID );
    OutputLog( "Allocate         Size:%8d\n", m_ResHeap.GetAllocatedSize() );
    OutputLog( "Allocate        Count:%8d\n", m_ResHeap.GetAllocatedCount() );
    OutputLog( "-TexAlloc        Size:%8d\n", m_ResTexAllocedSize );
    OutputLog( "-PrimAlloc       Size:%8d\n", m_ResPrimAllocedSize );
    OutputLog( "-ShdBinAlloc     Size:%8d\n", m_ResShaderBinAllocedSize );
    OutputLog( "-ShdCrdAlloc     Size:%8d\n", m_ResShaderCodeAllocedSize );
    OutputLog( "-EmSetAlloc      Size:%8d\n", m_ResEmitterSetAllocedSize );
    OutputLog( "-UboAlloc        Size:%8d\n", m_UboAllocedSize );
    OutputLog( "-ShdMngAlloc     Size:%8d\n", m_ShaderManagerAllocedSize );
    OutputLog( "------------------------\n" );
}




} // namespace eft2
} // namespace nw




