﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_RenderContext.h>


namespace nw   {
namespace eft2 {


//---------------------------------------------------------------------------
//  ResRenderStateの内容をセットする
//---------------------------------------------------------------------------
void Rendercontext::SetRenderStateResource( ResRenderState* resRenderState, bool colorBufferEnable )
{
    SetBlendType(   ( resRenderState->isBlendEnable != 0 ), resRenderState->blendType, colorBufferEnable );
    SetZBufferCtrl( ( resRenderState->isDepthTest   != 0 ), ( resRenderState->isDepthMask != 0 ), resRenderState->depthFunc );
    SetAlphaTest(   ( resRenderState->isAlphaTest   != 0 ), resRenderState->alphaThreshold, resRenderState->alphaFunc );
    SetDisplaySideType( resRenderState->displaySide );
}

//---------------------------------------------------------------------------
//  ブレンドタイプのセットアップ
//---------------------------------------------------------------------------
void Rendercontext::SetBlendType( bool blend, BlendType blendType, bool colorBufferEnable )
{
#ifdef EFT_OGL
    {
        EFT_UNUSED_VARIABLE( colorBufferEnable );

        if ( blend )
        {
            glEnable(  GL_BLEND );
            glBlendFunc( GL_SRC_ALPHA , GL_ONE );
        }
        else
        {
            glDisable( GL_BLEND );
        }

        switch( blendType )
        {
        case EFT_BLEND_TYPE_NORMAL:
            glBlendEquation( GL_FUNC_ADD );
            glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
            break;

        case EFT_BLEND_TYPE_ADD:
            glBlendEquation( GL_FUNC_ADD );
            glBlendFunc( GL_SRC_ALPHA, GL_ONE );
            break;

        case EFT_BLEND_TYPE_SUB:
            glBlendEquation( GL_FUNC_REVERSE_SUBTRACT );
            glBlendFunc( GL_SRC_ALPHA, GL_ONE );
            break;

        case EFT_BLEND_TYPE_SCREEN:
            glBlendEquation( GL_FUNC_ADD );
            glBlendFunc( GL_ONE_MINUS_DST_COLOR, GL_ONE );
            break;

        case EFT_BLEND_TYPE_MULT:
            glBlendEquation( GL_FUNC_ADD );
            glBlendFunc( GL_ZERO, GL_SRC_COLOR );
            break;
        }

        EFT_GLERR_CHECK();
    }
#endif

#if EFT_GX2
    {
        GX2SetColorControl( GX2_LOGIC_OP_COPY, static_cast<GX2Boolean>( blend ), GX2_DISABLE, static_cast<GX2Boolean>( colorBufferEnable ) );

        switch( blendType )
        {
        case EFT_BLEND_TYPE_NORMAL:
            GX2SetBlendControl( GX2_RENDER_TARGET_0,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE_MINUS_SRC_ALPHA,
                GX2_BLEND_COMBINE_ADD,
                GX2_DISABLE,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE_MINUS_SRC_ALPHA,
                GX2_BLEND_COMBINE_ADD
                );
            break;

        case EFT_BLEND_TYPE_ADD:
            GX2SetBlendControl( GX2_RENDER_TARGET_0,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_ADD,
                GX2_DISABLE,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_ADD
                );
            break;

        case EFT_BLEND_TYPE_SUB:
            GX2SetBlendControl( GX2_RENDER_TARGET_0,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_DST_MINUS_SRC,
                GX2_DISABLE,
                GX2_BLEND_SRC_ALPHA,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_DST_MINUS_SRC
                );
            break;

        case EFT_BLEND_TYPE_SCREEN:
            GX2SetBlendControl( GX2_RENDER_TARGET_0,
                GX2_BLEND_ONE_MINUS_DST_COLOR,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_ADD,
                GX2_DISABLE,
                GX2_BLEND_ONE_MINUS_DST_COLOR,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_ADD
                );
            break;

        case EFT_BLEND_TYPE_MULT:
            GX2SetBlendControl( GX2_RENDER_TARGET_0,
                GX2_BLEND_ZERO,
                GX2_BLEND_SRC_COLOR ,
                GX2_BLEND_COMBINE_ADD,
                GX2_DISABLE,
                GX2_BLEND_ONE_MINUS_DST_COLOR,
                GX2_BLEND_ONE,
                GX2_BLEND_COMBINE_ADD
                );
            break;
        }
    }
#endif
}

//---------------------------------------------------------------------------
//  Ｚバッファのセットアップ
//---------------------------------------------------------------------------
void Rendercontext::SetZBufferCtrl( bool test, bool write, CompareFunc func )
{
#ifdef EFT_OGL
    {
        glDepthMask( write );

        if ( test )
        {
            glEnable(  GL_DEPTH_TEST );
            switch ( func )
            {
            case nw::eft2::EFT_COMPARE_NEVER:   glDepthFunc( GL_NEVER ); break;
            case nw::eft2::EFT_COMPARE_LESS:    glDepthFunc( GL_LESS ); break;
            case nw::eft2::EFT_COMPARE_EQUAL:   glDepthFunc( GL_EQUAL ); break;
            case nw::eft2::EFT_COMPARE_LEQUAL:  glDepthFunc( GL_LEQUAL ); break;
            case nw::eft2::EFT_COMPARE_GREATER: glDepthFunc( GL_GREATER ); break;
            case nw::eft2::EFT_COMPARE_NOTEQUAL:glDepthFunc( GL_NOTEQUAL ); break;
            case nw::eft2::EFT_COMPARE_GEQUAL:  glDepthFunc( GL_GEQUAL ); break;
            case nw::eft2::EFT_COMPARE_ALWAYS:  glDepthFunc( GL_ALWAYS ); break;
            default: break;
            }
        }
        else
        {
            glDisable( GL_DEPTH_TEST );
        }
        EFT_GLERR_CHECK();
    }
#endif

#if EFT_GX2
    {
        GX2SetDepthOnlyControl( static_cast<GX2Boolean>( test  ),
                                static_cast<GX2Boolean>( write ), static_cast<GX2CompareFunction>( func ) );
    }
#endif
}

//---------------------------------------------------------------------------
//  αテストタイプのセットアップ
//---------------------------------------------------------------------------
void Rendercontext::SetAlphaTest( bool enable, f32 alphaThreshold, CompareFunc func )
{
    EFT_UNUSED_VARIABLE( func );

#if EFT_OPENGL
    {
        if ( enable )
        {
            glEnable(  GL_ALPHA_TEST );
            switch ( func )
            {
            case nw::eft2::EFT_COMPARE_NEVER:   glAlphaFunc( GL_NEVER,    alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_LESS:    glAlphaFunc( GL_LESS,     alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_EQUAL:   glAlphaFunc( GL_EQUAL,    alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_LEQUAL:  glAlphaFunc( GL_LEQUAL,   alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_GREATER: glAlphaFunc( GL_GREATER,  alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_NOTEQUAL:glAlphaFunc( GL_NOTEQUAL, alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_GEQUAL:  glAlphaFunc( GL_GEQUAL,   alphaThreshold ); break;
            case nw::eft2::EFT_COMPARE_ALWAYS:  glAlphaFunc( GL_ALWAYS,   alphaThreshold ); break;
            default: break;
            }
        }
        else
        {
            glDisable( GL_ALPHA_TEST );
        }
        EFT_GLERR_CHECK();
    }
#elif EFT_GX2
    {
        GX2SetAlphaTest( static_cast<GX2Boolean>( enable ),
                         static_cast<GX2CompareFunction>( func ), alphaThreshold );
    }
#else

    NW_UNUSED_VARIABLE( enable );
    NW_UNUSED_VARIABLE( alphaThreshold );

#endif
}

//------------------------------------------------------------------------------
//  表示面のセットアップ
//------------------------------------------------------------------------------
void Rendercontext::SetDisplaySideType( DisplaySideType displaySideType )
{
#ifdef EFT_OGL
    {
        switch( displaySideType )
        {
        case EFT_DISPLAYSIDETYPE_BOTH:
            glDisable( GL_CULL_FACE );
            break;

        case EFT_DISPLAYSIDETYPE_FRONT:
            glEnable( GL_CULL_FACE );
            glCullFace( GL_BACK );
            break;

        case EFT_DISPLAYSIDETYPE_BACK:
            glEnable( GL_CULL_FACE );
            glCullFace( GL_FRONT );
            break;
        }
    }

#elif EFT_GX2

    {
        switch( displaySideType )
        {
        case EFT_DISPLAYSIDETYPE_BOTH:  GX2SetCullOnlyControl( GX2_FRONT_FACE_CCW, GX2_FALSE, GX2_FALSE ); break;
        case EFT_DISPLAYSIDETYPE_FRONT: GX2SetCullOnlyControl( GX2_FRONT_FACE_CCW, GX2_FALSE, GX2_TRUE );  break;
        case EFT_DISPLAYSIDETYPE_BACK:  GX2SetCullOnlyControl( GX2_FRONT_FACE_CCW, GX2_TRUE, GX2_FALSE );  break;
        }
    }

#else

    NW_UNUSED_VARIABLE( displaySideType );

#endif
}


} // namespace eft2
} // namespace nw

