﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_CurlNoiseData.h>
#include <nw/eft/eft2_FieldRandom.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_System.h>

namespace nw   {
namespace eft2 {

//------------------------------------------------------------------------------
//  8キーアニメの計算（フィールド）
//------------------------------------------------------------------------------
void CalculateField8KeyAnim(
    nw::math::VEC3* dstAnim,
    Emitter* emitter,
    const nw::eft2::ResAnim8KeyParamSet& anim,
    const f32 time,
    const f32 life )
{
    EFT_UNUSED_VARIABLE( emitter );
    EFT_UNUSED_VARIABLE( anim );
    EFT_UNUSED_VARIABLE( time );
    EFT_UNUSED_VARIABLE( life );

    nw::math::VEC3 start    = nw::math::VEC3( 0, 0, 0 );
    nw::math::VEC3 end      = nw::math::VEC3( 0, 0, 0 );
    int  term   = 0;
    float dis   = 0;
    const int lastKeyIndex = anim.keyNum - 1;

    // キーなし
    if ( anim.keyNum == 0 )
    {
        *dstAnim = nw::math::VEC3( 0, 0, 0 );
        return;
    }
    // キーが一つのみ
    if ( anim.keyNum == 1 )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.keyValue[0] );
        //return nw::math::VEC3( anim.keyValue[0].x, anim.keyValue[0].y, anim.keyValue[0].z );
        return;
    }

    // ループを加味した、再生タイミング[0, 1]に計算し直す
    float localTime = 0;
    if( anim.loop > 0 )
    {
        localTime = time + anim.startRandom * emitter->particleAttr->random.x * anim.loopNum;
        localTime = ( static_cast<f32>(fmod( localTime, anim.loopNum )) ) / static_cast<f32>( anim.loopNum );
    }
    else
    {
        localTime = time / static_cast<f32>(emitter->emitterData->ptcl.life);
    }
    //f32 rate = ( anim.loop ) ? ( 100.0f / anim.loopNum ) : 1.0f;
    //f32 localTime = (time / life) * rate + static_cast<f32>( anim.startRandom ) * emitter->particleAttr->random.x;
    //終端処理（ループ終端を0.0には飛ばさない）
    //localTime = ( localTime == rate ) ? 1.0f : fmod( localTime, 1.0f );

    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if ( anim.keyValue[ lastKeyIndex ].w <= localTime )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.keyValue[lastKeyIndex] );
        //return nw::math::VEC3( anim.keyValue[lastKeyIndex].x, anim.keyValue[lastKeyIndex].y, anim.keyValue[lastKeyIndex].z );
        return;
    }
    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if ( localTime < anim.keyValue[0].w )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.keyValue[0] );
        //return nw::math::VEC3( anim.keyValue[0].x, anim.keyValue[0].y, anim.keyValue[0].z );
        return;
    }

    // TODO：平たく書いた方がよいかも。
    for ( term = 0; term < anim.keyNum; term++ )
    {
        s32 index1 = term;
        s32 index2 = term + 1;
        f32 t1 = anim.keyValue[index1].w;
        f32 t2 = anim.keyValue[index2].w;

        if ( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            EFT_F32_VEC3_COPY( &start, &anim.keyValue[index1] );
            EFT_F32_VEC3_COPY( &end, &anim.keyValue[index2] );
            //start.Set( anim.keyValue[index1].x, anim.keyValue[index1].y, anim.keyValue[index1].z );
            //end.Set( anim.keyValue[index2].x, anim.keyValue[index2].y, anim.keyValue[index2].z );
            dis   = t2 - t1;

            nw::math::VEC3 ret = ( start + ( end - start ) / dis * ( localTime - t1 ) );
            EFT_F32_VEC3_COPY( dstAnim, &ret );
            return;
        }
    }

    // ここには来ないはず
    dstAnim->x = 0.0f;
    dstAnim->y = 0.0f;
    dstAnim->z = 0.0f;
    return;
}

//------------------------------------------------------------------------------
//  8キーアニメの計算
//------------------------------------------------------------------------------
void Calculate8KeyAnim(
    nw::math::VEC3*                   dstAnim,
    const nw::eft2::ResAnim8KeyParam& anim,
    const s32                         keyNum,
    const f32                         random,
    const f32                         time,
    const f32                         loopRate,
    const f32                         isLoopRandom,
    const f32                         life )
{
    nw::math::VEC3 start    = nw::math::VEC3( 0, 0, 0 );
    nw::math::VEC3 end      = nw::math::VEC3( 0, 0, 0 );
    int  term   = 0;
    float dis   = 0;
    const int lastKeyIndex = keyNum - 1;

    // キーなし
    if ( keyNum == 0 )
    {
        dstAnim->x = 1.f;
        dstAnim->y = 1.f;
        dstAnim->z = 1.f;
        return;
    }
    // キーが一つのみ
    if ( keyNum == 1 )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.value[0] );
        //dstAnim->x = anim.value[0].x;
        //dstAnim->y = anim.value[0].y;
        //dstAnim->z = anim.value[0].z;
        return;
    }

    // ループを加味した、再生タイミング[0, 1]に計算し直す
    float localTime = 0;
    if( loopRate > 0 )
    {
        localTime = time + isLoopRandom * random * loopRate;
        localTime = ( fmod( localTime, loopRate ) ) / static_cast<f32>( loopRate );
    }
    else
    {
        localTime = time / static_cast<f32>( life );
    }

    //f32 rate = ( 100.0f / loopRate );
    //f32 localTime = (time / life) * rate + isLoopRandom * emitter->particleAttr->random.x;
    ////終端処理（ループ終端を0.0には飛ばさない）
    //localTime = ( localTime == rate ) ? 1.0f : fmod( localTime, 1.0f );

    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if ( anim.value[ lastKeyIndex ].w <= localTime )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.value[lastKeyIndex] );
        //dstAnim->x = anim.value[lastKeyIndex].x;
        //dstAnim->y = anim.value[lastKeyIndex].y;
        //dstAnim->z = anim.value[lastKeyIndex].z;
        return;
    }
    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if ( localTime < anim.value[0].w )
    {
        EFT_F32_VEC3_COPY( dstAnim, &anim.value[0] );
        //dstAnim->x = anim.value[0].x;
        //dstAnim->y = anim.value[0].y;
        //dstAnim->z = anim.value[0].z;
        return;
    }

    for ( term = 0; term < keyNum; term++ )
    {
        s32 index1 = term;
        s32 index2 = term + 1;
        f32 t1 = anim.value[index1].w;
        f32 t2 = anim.value[index2].w;

        if ( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            EFT_F32_VEC3_COPY( &start, &anim.value[index1] );
            EFT_F32_VEC3_COPY( &end, &anim.value[index2] );
            //start.Set( anim.value[index1].x, anim.value[index1].y, anim.value[index1].z );
            //end.Set( anim.value[index2].x, anim.value[index2].y, anim.value[index2].z );
            dis   = t2 - t1;
            f32 divp = ( localTime - t1 );
            dstAnim->x = ( start.x + ( end.x - start.x ) / dis * divp );
            dstAnim->y = ( start.y + ( end.y - start.y ) / dis * divp );
            dstAnim->z = ( start.z + ( end.z - start.z ) / dis * divp );
            return;
        }
    }

    // 処理が行われなかった。

    dstAnim->x = 0.0f;
    dstAnim->y = 0.0f;
    dstAnim->z = 0.0f;
    return;
}

//---------------------------------
//  フィールド：カスタムフィールド
//---------------------------------
void _CalcParticleBehavior_CustomField( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr, s32 particleIndex )
{
    CustomFieldCallback Field = emitter->GetEmitterSet()->GetSystem()->GetCustomFieldCallback();
    if( Field )
    {
        const ResFieldCustom* resFieldCustom = emitter->emitterRes->fieldCustomData;
        Field( emitter, pos, vec, attr, resFieldCustom, particleIndex );
    }
}

//---------------------------------
//  フィールド：新ランダム（仮
//---------------------------------
void _CalcParticleBehavior_FieldRandom( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE( pos );
    EFT_UNUSED_VARIABLE( attr );

    f32 time        = emitter->time - vec->w;
    ResFieldRandom* resFieldRandom = emitter->emitterRes->fieldRandomData;

    nw::math::VEC3 randomVel;
    if( resFieldRandom->animParam.enable )
    {
        CalculateField8KeyAnim( &randomVel, emitter, resFieldRandom->animParam, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
    }
    else
    {
        randomVel = resFieldRandom->randomVel;
    }

    CalcFieldRandom(emitter, pos, vec, attr, randomVel);
}


//---------------------------------
//  フィールド：ランダム（FE1のバージョン）
//---------------------------------
void _CalcParticleBehavior_FieldRandomFe1( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE( pos );
    EFT_UNUSED_VARIABLE( attr );

    ResFieldRandomFe1* resFieldRandom = emitter->emitterRes->fieldRandomDataFe1;
    f32 ptclTime = emitter->time - vec->w;
    nw::math::VEC3 fieldRandomVelAdd;

    nw::math::VEC3 randomVel;
    if( resFieldRandom->animParam.enable )
    {
        CalculateField8KeyAnim( &randomVel, emitter, resFieldRandom->animParam, ptclTime, static_cast<f32>(emitter->emitterData->ptcl.life) );
    }
    else
    {
        randomVel = resFieldRandom->randomVel;
    }

    if( ( static_cast<u32>( ptclTime ) % resFieldRandom->blank ) == 0 )
    {
        const nw::math::VEC3 &rndVec3 = emitter->random.GetVec3();

        vec->x += rndVec3.x * randomVel.x;
        vec->y += rndVec3.y * randomVel.y;
        vec->z += rndVec3.z * randomVel.z;
    }
}

enum FieldMagnetFlg
{
    EFT_MAGNET_FLAG_X = ( 1 << 0 ) ,                        //!< X軸に適応
    EFT_MAGNET_FLAG_Y = ( 1 << 1 ) ,                        //!< Y軸に適応
    EFT_MAGNET_FLAG_Z = ( 1 << 2 ) ,                        //!< Z軸に適応
    EFT_MAGNET_FLAG_FORCE_4BYTE = 0x80000000
};

//------------------------------------------------------------------------------
//  磁力
//------------------------------------------------------------------------------
void _CalcParticleBehavior_FieldMagnet( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE(emitter);
    EFT_UNUSED_VARIABLE(pos);
    EFT_UNUSED_VARIABLE(vec);
    EFT_UNUSED_VARIABLE(attr);

    ResFieldMagnet* resFieldMagnet = emitter->emitterRes->fieldMagnetData;

    // ８キーアニメ適用
    nw::math::VEC3 fieldMagnetPower;
    {
        f32 time        = emitter->time - vec->w;
        const ResAnim8KeyParamSet& anim = emitter->emitterRes->fieldMagnetData->animParam;

        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldMagnetPower, emitter, anim, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
        }
        else
        {
            fieldMagnetPower.x = resFieldMagnet->fieldMagnetPower;
        }
    }

    if ( resFieldMagnet->isFollowEmitter )
    {
        // エミッタを追従

        nw::math::VEC3 local;
        nw::math::VEC3 src;
        //src.x = emitter->emitterData->emitter.trans.x;
        //src.y = emitter->emitterData->emitter.trans.y;
        //src.z = emitter->emitterData->emitter.trans.z;
        src.x = emitter->matrixRT.m[0][3];
        src.y = emitter->matrixRT.m[1][3];
        src.z = emitter->matrixRT.m[2][3];

        // ローカルのベクトルを算出
        //emitter->TransformLocalVec( &local, &src );
        {
            nw::math::VEC3* pDst = &local;
            const nw::math::VEC3* pSrc = &src;

            nw::math::MTX34 invMat;
            if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_NONE )
            {
                // 追従しない
                nw::math::MTX34 mtx;
                mtx.v[0] = attr->emitterMat0;
                mtx.v[1] = attr->emitterMat1;
                mtx.v[2] = attr->emitterMat2;
                MTX34Inverse( &invMat, &mtx );
            }
            else
            {
                // 完全追従 or 位置のみ追従
                MTX34Inverse( &invMat, &emitter->matrixSRT );
                VEC3Transform( pDst, &invMat, pSrc );
            }
            VEC3Transform( pDst, &invMat, pSrc );
        }

        // 速度を更新
        if( resFieldMagnet->fieldMagnetFlagX  ) vec->x += ( local.x + resFieldMagnet->fieldMagnetPos.x - pos->x - vec->x) * fieldMagnetPower.x;
        if( resFieldMagnet->fieldMagnetFlagY  ) vec->y += ( local.y + resFieldMagnet->fieldMagnetPos.y - pos->y - vec->y) * fieldMagnetPower.x;
        if( resFieldMagnet->fieldMangetFlagZ  ) vec->z += ( local.z + resFieldMagnet->fieldMagnetPos.z - pos->z - vec->z) * fieldMagnetPower.x;
    }
    else
    {
        // エミッタを追従しない

        // 速度を更新
        if( resFieldMagnet->fieldMagnetFlagX  ) vec->x += ( resFieldMagnet->fieldMagnetPos.x - pos->x - vec->x) * fieldMagnetPower.x;
        if( resFieldMagnet->fieldMagnetFlagY  ) vec->y += ( resFieldMagnet->fieldMagnetPos.y - pos->y - vec->y) * fieldMagnetPower.x;
        if( resFieldMagnet->fieldMangetFlagZ  ) vec->z += ( resFieldMagnet->fieldMagnetPos.z - pos->z - vec->z) * fieldMagnetPower.x;
    }
}


//------------------------------------------------------------------------------
//  スピン
//------------------------------------------------------------------------------
void _CalcParticleBehavior_FieldSpin( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE(emitter);
    EFT_UNUSED_VARIABLE(pos);
    EFT_UNUSED_VARIABLE(vec);
    EFT_UNUSED_VARIABLE(attr);

    ResFieldSpinData* dat = emitter->emitterRes->fieldSpinData;
    float sinV , cosV;

    f32 frameRate = emitter->frameRate;
    f32 dynamicsRand = attr->scale.w;

    f32 time        = emitter->time - vec->w;

    // ８キーアニメ適用
    nw::math::VEC3 fieldSpinRotate;
    {
        const ResAnim8KeyParamSet& anim = emitter->emitterRes->fieldSpinData->rotateAnimParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldSpinRotate, emitter, anim, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
            fieldSpinRotate = fieldSpinRotate / 180.0f * math::F_PI;    //ラジアンに変換
        }
        else
        {
            fieldSpinRotate.x = dat->fieldSpinRotate;
        }
    }
    nw::math::VEC3 fieldSpinOuter;
    {
        const ResAnim8KeyParamSet& anim = emitter->emitterRes->fieldSpinData->outerAnimParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldSpinOuter, emitter, anim, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
            fieldSpinOuter = fieldSpinOuter / 180.0f * math::F_PI;    //ラジアンに変換
        }
        else
        {
            fieldSpinOuter.x = dat->fieldSpinOuter;
        }
    }

    nw::math::SinCosRad( &sinV , &cosV , fieldSpinRotate.x * dynamicsRand * frameRate );

    switch ( dat->fieldSpinAxis )
    {
    case 0://X軸
    {
        f32 v0 = pos->y *  cosV  + pos->z * sinV;
        f32 v1 = pos->y * -sinV  + pos->z * cosV;
        pos->y = v0;
        pos->z = v1;

        // 拡散速度
        if( fieldSpinOuter.x == 0.0f )
        {
            break;
        }

        f32 length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        f32 r = 1.0f / nw::math::FSqrt( length2 ) * fieldSpinOuter.x * dynamicsRand * frameRate;
        pos->y += ( v0 * r );
        pos->z += ( v1 * r );
    }
    break;

    case 1://Y軸
    {
        f32 v0 = pos->z *  cosV  + pos->x * sinV;
        f32 v1 = pos->z * -sinV  + pos->x * cosV;
        pos->z = v0;
        pos->x = v1;

        // 拡散速度
        if( fieldSpinOuter.x == 0.0f )
        {
            break;
        }

        f32 length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        f32 r = 1.0f / nw::math::FSqrt( length2 ) * fieldSpinOuter.x * dynamicsRand * frameRate;
        pos->z += ( v0 * r );
        pos->x += ( v1 * r );
    }
    break;

    case 2://Z軸
    {
        f32 v0 = pos->x *  cosV  + pos->y * sinV;
        f32 v1 = pos->x * -sinV  + pos->y * cosV;
        pos->x = v0;
        pos->y = v1;

        // 拡散速度
        if( fieldSpinOuter.x == 0.0f )
        {
            break;
        }

        f32 length2 = v0 * v0 + v1 * v1;
        if( length2 <= 0.0f )
        {
            break;
        }

        f32 r = 1.0f / nw::math::FSqrt( length2 ) * fieldSpinOuter.x * dynamicsRand * frameRate;
        pos->x += ( v0 * r );
        pos->y += ( v1 * r );
    }
    break;
    }
}

enum FieldCollisionReaction
{
    EFT_FIELD_COLLISION_REACTION_REFLECTION = 0,               //!< 反射
    EFT_FIELD_COLLISION_REACTION_CESSER ,               //!< 消滅
    EFT_FIELD_COLLISION_REACTION_CUSTOM ,                   //!< カスタム
    EFT_FIELD_COLLISION_REACTION_FORCE_4BYTE = 0x80000000
};

static const f32 EPSILON = 0.0001f;

//------------------------------------------------------------------------------
//  コリジョン
//------------------------------------------------------------------------------
void _CalcParticleBehavior_FieldCollision( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, ParticleAttribute* attr, ParticleData* ptclData )
{
    ResFieldCollisionData* dat = emitter->emitterRes->fieldCollisionData;

    // コリジョン回数判定
    if ( dat->fieldCollisionCnt != -1 && dat->fieldCollisionCnt <= static_cast<s32>( ptclData->collisionCount ) )
    {
        return;
    }

    // XZ平面のY座標
    f32 y = dat->fieldCollisionCoord;
    {
        // データ上のコリジョン平面を利用する

        //----------------------------------------------------------
        //      ワールド座標系
        //----------------------------------------------------------
        if( dat->fieldCollisionIsWorld )
        {
            // エミッタ座標系へ
             nw::math::VEC3 worldPos;

            // エミッタ→ワールド変換行列
            nw::math::MTX34 *matEmitter;
            nw::math::MTX34 mtx;

            if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_ALL )
            {
                matEmitter = &emitter->matrixSRT;
            }
            else
            {
                mtx.m[0][0] = attr->emitterMat0.x;
                mtx.m[0][1] = attr->emitterMat0.y;
                mtx.m[0][2] = attr->emitterMat0.z;
                mtx.m[0][3] = attr->emitterMat0.w;

                mtx.m[1][0] = attr->emitterMat1.x;
                mtx.m[1][1] = attr->emitterMat1.y;
                mtx.m[1][2] = attr->emitterMat1.z;
                mtx.m[1][3] = attr->emitterMat1.w;

                mtx.m[2][0] = attr->emitterMat2.x;
                mtx.m[2][1] = attr->emitterMat2.y;
                mtx.m[2][2] = attr->emitterMat2.z;
                mtx.m[2][3] = attr->emitterMat2.w;

                matEmitter = &mtx;
            }

            {
                nw::math::VEC3 temp(pos->x, pos->y, pos->z);
                VEC3Transform(&worldPos, matEmitter, &temp);
            }

            switch( dat->fieldCollisionType )
            {
                // 消滅
            case EFT_FIELD_COLLISION_REACTION_CESSER:

                // 上から下への侵入を許さない
                if( worldPos.y < y )
                {
                    worldPos.y = y;
                    f32 ptclTime = emitter->time - vec->w;
                    // 寿命を縮める
                    pos->w = ptclTime;
                    ptclData->life = ptclTime;
                }
                break;

                // 反射
            case EFT_FIELD_COLLISION_REACTION_REFLECTION:

                // 上から下への侵入を許さない
                if( worldPos.y < y  )
                {
                    worldPos.y = y + EPSILON;

                    // 速度をワールドに持っていく
                     nw::math::VEC3 worldVel;
                    {
                        // TODO : mathクラス置き換えの暫定処理なので関数化
                        f32 tx = vec->x;
                        f32 ty = vec->y;
                        f32 tz = vec->z;
                        nw::math::MTX34 &m = *matEmitter;
                        worldVel.x = m.m[0][0] * tx + m.m[0][1] * ty + m.m[0][2] * tz;
                        worldVel.y = m.m[1][0] * tx + m.m[1][1] * ty + m.m[1][2] * tz;
                        worldVel.z = m.m[2][0] * tx + m.m[2][1] * ty + m.m[2][2] * tz;
                    }

                    // 反射
                    worldVel.y *= -dat->fieldCollisionCoef;

                    // 逆行列
                    nw::math::MTX34 matEmitterInv;
                    matEmitterInv.SetInverse( *matEmitter );

                    // 速度と位置をエミッタ座標系に持っていく
                    f32 nowLife = pos->w;
                    nw::math::VEC4 temp(worldPos.x, worldPos.y, worldPos.z, 1); // 寿命は一旦除く
                    pos->SetTransform( matEmitterInv , temp );
                    pos->w = nowLife;   // 寿命は上記変換時には抜かないと計算結果が崩れるので注意。
                    {
                        f32 tx = worldVel.x;
                        f32 ty = worldVel.y;
                        f32 tz = worldVel.z;
                        nw::math::MTX34 &m = matEmitterInv;
                        vec->x = m.m[0][0] * tx + m.m[0][1] * ty + m.m[0][2] * tz;
                        vec->y = m.m[1][0] * tx + m.m[1][1] * ty + m.m[1][2] * tz;
                        vec->z = m.m[2][0] * tx + m.m[2][1] * ty + m.m[2][2] * tz;

                        vec->x *= dat->fieldCollisionRegist;
                        vec->y *= dat->fieldCollisionRegist;
                        vec->z *= dat->fieldCollisionRegist;
                    }

                    ptclData->collisionCount += 1; //コリジョン回数を増やす
                }
                break;
            }
        }

        //----------------------------------------------------------
        //      エミッタ座標系
        //----------------------------------------------------------
        else
        {
            switch( dat->fieldCollisionType )
            {
                // 消滅
            case EFT_FIELD_COLLISION_REACTION_CESSER:

                // 上から下への侵入を許さない
                if( pos->y < y )
                {
                    pos->y = y;

                    f32 ptclTime = emitter->time - vec->w;
                    // 寿命を縮める
                    pos->w = ptclTime;
                    ptclData->life = ptclTime;
                }
                break;

                // 反射
            case EFT_FIELD_COLLISION_REACTION_REFLECTION:

                // 上から下への侵入を許さない
                if( pos->y < y )
                {
                    pos->y = y;
                    vec->y *= -dat->fieldCollisionCoef;

                    vec->x *= dat->fieldCollisionRegist;
                    vec->y *= dat->fieldCollisionRegist;
                    vec->z *= dat->fieldCollisionRegist;

                    ptclData->collisionCount += 1; //コリジョン回数を増やす
                }
                break;
            }
        }
    }

}

enum FieldConvergenceType
{
    EFT_FIELD_CONVERGENCE_TYPE_ASSIGNED_POSITION = 0,           //!< 指定された位置に収束
    EFT_FIELD_CONVERGENCE_TYPE_EMITTER_POSITION,                //!< エミッタ位置に収束
    EFT_FIELD_CONVERGENCE_TYPE_FORCE_4BYTE = 0x80000000
};


//------------------------------------------------------------------------------
//  収束
//------------------------------------------------------------------------------
void _CalcParticleBehavior_FieldConvergence( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE(vec);
    EFT_UNUSED_VARIABLE(attr);

    // 仮打ちのパラメータ
    ResFieldConvergenceData* dat = emitter->emitterRes->fieldConvergenceData;

    nw::math::VEC3 fieldConvergenceRatio;
    f32 frameRate = emitter->frameRate;
    f32 dynamicsRand = attr->scale.w;
    //dynamicsRand = 1.0f;

    // ８キーアニメ適用
    {
        f32 time        = emitter->time - vec->w;
        const ResAnim8KeyParamSet& anim = emitter->emitterRes->fieldConvergenceData->animParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldConvergenceRatio, emitter, anim, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
        }
        else
        {
            fieldConvergenceRatio.x = dat->fieldConvergenceRatio;
        }
    }

    if ( dat->fieldConvergenceType == EFT_FIELD_CONVERGENCE_TYPE_ASSIGNED_POSITION )
    {
        pos->x += ( dat->fieldConvergencePos.x - pos->x ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
        pos->y += ( dat->fieldConvergencePos.y - pos->y ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
        pos->z += ( dat->fieldConvergencePos.z - pos->z ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
    }
    else
    {
        nw::math::VEC3 local;
        nw::math::VEC3 src;

        src.x = emitter->matrixRT.m[0][3];
        src.y = emitter->matrixRT.m[1][3];
        src.z = emitter->matrixRT.m[2][3];

        //emitter->TransformLocalVec( &local, &src );
        {
            nw::math::VEC3* pDst = &local;
            const nw::math::VEC3* pSrc = &src;

            nw::math::MTX34 invMat;
            if( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_NONE )
            {
                // 追従しない
                nw::math::MTX34 mtx;
                mtx.v[0] = attr->emitterMat0;
                mtx.v[1] = attr->emitterMat1;
                mtx.v[2] = attr->emitterMat2;
                MTX34Inverse( &invMat, &mtx );
            }
            else
            {
                // 完全追従 or 位置のみ追従
                MTX34Inverse( &invMat, &emitter->matrixSRT );
                VEC3Transform( pDst, &invMat, pSrc );
            }
            VEC3Transform( pDst, &invMat, pSrc );
        }
        pos->x += ( local.x + dat->fieldConvergencePos.x - pos->x ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
        pos->y += ( local.y + dat->fieldConvergencePos.y - pos->y ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
        pos->z += ( local.z + dat->fieldConvergencePos.z - pos->z ) * fieldConvergenceRatio.x * dynamicsRand * frameRate;
    }
}



//------------------------------------------------------------------------------
//  位置に加算
//------------------------------------------------------------------------------
void _CalcParticleBehavior_FieldPosAdd( Emitter* emitter, nw::math::VEC4* pos, nw::math::VEC4* vec, const ParticleAttribute* attr )
{
    EFT_UNUSED_VARIABLE(vec);
    EFT_UNUSED_VARIABLE(attr);

    // 仮打ちパラメータ
    ResFieldPosAddData* dat = emitter->emitterRes->fieldPosAddData;

    f32 frameRate = emitter->frameRate;
    f32 dynamicsRand = attr->scale.w;
    //dynamicsRand = 1.0f;
    nw::math::VEC3 fieldPosAdd;

    // ８キーアニメ適用
    {
        f32 time        = emitter->time - vec->w;
        const ResAnim8KeyParamSet& anim = emitter->emitterRes->fieldPosAddData->animParam;
        if( anim.enable )
        {
            CalculateField8KeyAnim( &fieldPosAdd, emitter, anim, time, static_cast<f32>(emitter->emitterData->ptcl.life) );
        }
        else
        {
            EFT_F32_VEC3_COPY( &fieldPosAdd, &dat->fieldPosAdd );
            //fieldPosAdd.x = dat->fieldPosAdd.x;
            //fieldPosAdd.y = dat->fieldPosAdd.y;
            //fieldPosAdd.z = dat->fieldPosAdd.z;
        }
    }

    if( !dat->isFieldPosAddGlobal )
    {
        pos->x += ( fieldPosAdd.x * dynamicsRand * frameRate );
        pos->y += ( fieldPosAdd.y * dynamicsRand * frameRate );
        pos->z += ( fieldPosAdd.z * dynamicsRand * frameRate );
    }
    else
    {
        nw::math::MTX34* mrt  =  &emitter->matrixRT;   //単位行列（仮）

         f32 posAddX = fieldPosAdd.x * dynamicsRand * frameRate;
         f32 posAddY = fieldPosAdd.y * dynamicsRand * frameRate;
         f32 posAddZ = fieldPosAdd.z * dynamicsRand * frameRate;

        pos->x +=  posAddX * mrt->m[0][0] + posAddY * mrt->m[1][0] + posAddZ * mrt->m[2][0];
        pos->y +=  posAddX * mrt->m[0][1] + posAddY * mrt->m[1][1] + posAddZ * mrt->m[2][1];
        pos->z +=  posAddX * mrt->m[0][2] + posAddY * mrt->m[1][2] + posAddZ * mrt->m[2][2];
    }
}

/*
//---------------------------------------------------
//  CurlNoiseフィールド コールバック
//---------------------------------------------------
const void *EmitterCalc::_ptclField_CurlNoise( EmitterInstance * __restrict e, PtclInstance * __restrict ptcl , const void *fieldData, f32 frameRate )
{
    NW_UNUSED_VARIABLE( frameRate );

    const FieldCurlNoiseData *dat = reinterpret_cast< const FieldCurlNoiseData * >( fieldData );

    // カールノイズ
    f32 time    = (f32)e->cnt;
    f32 sizeMul = dat->fieldCurlNoiseScale;              // 渦のスケール
    f32 spdMulX = dat->fieldCurlNoiseInfluence.x;        // 影響の強さX
    f32 spdMulY = dat->fieldCurlNoiseInfluence.y;        // 影響の強さY
    f32 spdMulZ = dat->fieldCurlNoiseInfluence.z;        // 影響の強さZ
    f32 ofsXSpd = dat->fieldCurlNoiseSpeed.x;            // 渦アニメ速度X
    f32 ofsYSpd = dat->fieldCurlNoiseSpeed.y;            // 渦アニメ速度Y
    f32 ofsZSpd = dat->fieldCurlNoiseSpeed.z;            // 渦アニメ速度Z
    f32 ofsBase = dat->fieldCurlNoiseBase;               // ノイズのオフセット
    if ( dat->isFieldCurlNoiseBaseRandom & 0x01 ) ofsBase *= ptcl->random[0];  // ノイズのオフセットランダム
    s32 ipType  = (dat->isFieldCurlNoiseInterpolation & 0x01);                 // 補間タイプ(0:補間なし 1以上:補間あり)

    // パーティクル位置から受ける力を計算
    nw::math::Vector3 ofs;
    ofs.x = ptcl->pos.x;
    ofs.y = ptcl->pos.y;
    ofs.z = ptcl->pos.z;

    ofs.x *= sizeMul;
    ofs.y *= sizeMul;
    ofs.z *= sizeMul;
    ofs.x += ofsXSpd*time;
    ofs.y += ofsYSpd*time;
    ofs.z += ofsZSpd*time;
    nw::math::Vector3 add;

    if( ipType == 0 ){
        add = g_CurlNoiseTbl32.GetCurlNoiseS8( (s32)(ofs.x+ofsBase), (s32)(ofs.y+ofsBase), (s32)(ofs.z+ofsBase) );
    }else{
        add = g_CurlNoiseTbl32.GetCurlNoiseS8Interpolate( ofs.x+ofsBase, ofs.y+ofsBase, ofs.z+ofsBase );
    }
    add.x *= spdMulX;
    add.y *= spdMulY;
    add.z *= spdMulZ;

    // 力を速度に加算
    ptcl->vel.x += add.x;
    ptcl->vel.y += add.y;
    ptcl->vel.z += add.z;

    return ( dat + 1 );
}
*/

//------------------------------------------------------------------------------
// 揺らぎ : Sin波
//------------------------------------------------------------------------------
f32 _CalcFluctuation_Sin( f32 time, f32 amplitude, f32 cycle, f32 phaseInit, f32 phaseRandom, const nw::math::VEC4* random )
{
    const f32 t = ( ( time + phaseInit ) / cycle ) + ( phaseRandom * random->x );                       //現在フレーム位置（割合）
    const f32 flucValue = 1.0f - ( ( cos( t * nw::math::F_PI * 2.0f ) + 1.0f ) * 0.5f * amplitude );    //揺らぎ量
    return flucValue;
}

//------------------------------------------------------------------------------
// 揺らぎ : のこぎり波
//------------------------------------------------------------------------------
f32 _CalcFluctuation_SawTooth( f32 time, f32 amplitude, f32 cycle, f32 phaseInit, f32 phaseRandom, const nw::math::VEC4* random )
{
    const f32 t = (( time + phaseInit ) / cycle ) + ( phaseRandom * random->x );            //現在フレーム位置（割合）
    const f32 flucValue = fabs( ( 1.0f - fract( t ) * amplitude ) );                        //揺らぎ量
    return flucValue;
}

//------------------------------------------------------------------------------
// 揺らぎ : 矩形波
//------------------------------------------------------------------------------
f32 _CalcFluctuation_Rect( f32 time, f32 amplitude, f32 cycle, f32 phaseInit, f32 phaseRandom, const nw::math::VEC4* random )
{
    const f32 t = (( time + phaseInit ) / cycle ) + ( phaseRandom * random->x );            //現在フレーム位置（割合）
    const f32 flucValue = fabs( 1.0f - ( 1.0f - step( 0.5f, fract( t ) ) ) * amplitude );   //揺らぎ量
    return flucValue;
}



//------------------------------------------------------------------------------
//  テクスチャーパターンアニメ計算
//------------------------------------------------------------------------------
#ifdef EFT_DEGRADATION_SPEC
s32 _CalcFluctuation_TexPatternAnim( f32 time, f32 life, f32 random, const ResTexPtnAnim* texPatternAnimData, const ResTextureAnim* texAnimData )
{
    s32 timeS        = static_cast<s32>( time );
    s32 lifeS        = static_cast<s32>( life );
    s32 tableNum     = static_cast<s32>( texPatternAnimData->ptnTableNum );
    s32 patternIndex = 0;

    TexturePtnAnimType animType = texAnimData->patternAnimType;

    if ( animType == EFT_TEXTURE_PTN_ANIM_TYPE_LIFE_FIT )
    {
        patternIndex = timeS * tableNum / lifeS;
    }

    if ( animType == EFT_TEXTURE_PTN_ANIM_TYPE_LOOP )
    {
        patternIndex = timeS / static_cast<s32>( texPatternAnimData->ptnFreq );

        if ( texAnimData->isPatAnimLoopRandom )
        {
            patternIndex += static_cast<s32>( random * tableNum );
        }
    }

    if ( animType == EFT_TEXTURE_PTN_ANIM_TYPE_CLAMP )
    {
        patternIndex = timeS / static_cast<s32>( texPatternAnimData->ptnFreq );

        if ( tableNum < patternIndex )
        {
            patternIndex = tableNum-1;
        }
    }

    if ( animType == EFT_TEXTURE_PTN_ANIM_TYPE_RANDOM )
    {
        patternIndex = static_cast<s32>( random * tableNum );
    }

    patternIndex = texPatternAnimData->ptnTable[ patternIndex % tableNum ];

    return patternIndex;
}
#endif



//---------------------------------------------------------------------------
//  パーティクルの挙動計算を行う 積算版
//---------------------------------------------------------------------------
void _CalcParticleBehavior_Integrating( Emitter* emitter, s32 index, nw::math::VEC4* pos, nw::math::VEC4* vec, const nw::math::VEC4* posBack, const nw::math::VEC4* vecBack, const f32 time )
{
    EFT_NULL_ASSERT( emitter->emitterRes->resEmitterStaticUniformBlock );

    const f32 frameRate       = emitter->frameRate;

    ParticleAttribute*  ptclAttr = &emitter->particleAttr[index];
    ParticleData*       ptclData = &emitter->particleData[index];

    // 運動量ランダム x フレームレート
    const f32 rndRatio = ptclAttr->scale.w * frameRate;
    if( time > 0 )
    {
        pos->x = posBack->x + ( vecBack->x * rndRatio );
        pos->y = posBack->y + ( vecBack->y * rndRatio );
        pos->z = posBack->z + ( vecBack->z * rndRatio );
    }
    else
    {
        // 初回フレームだけはフロントバッファの値を利用して更新
        pos->x += ( vec->x * rndRatio );
        pos->y += ( vec->y * rndRatio );
        pos->z += ( vec->z * rndRatio );
    }

    // 空気抵抗
    if ( emitter->emitterRes->resEmitterStaticUniformBlock->airRegist < 1.0f )
    {
        const f32 airRegist = pow( emitter->emitterRes->resEmitterStaticUniformBlock->airRegist, frameRate );
        if( time > 0 )
        {
            vec->x = vecBack->x * airRegist;
            vec->y = vecBack->y * airRegist;
            vec->z = vecBack->z * airRegist;
        }
        else
        {
            // 初回フレームだけはフロントバッファの値を利用して更新
            vec->x *= airRegist;
            vec->y *= airRegist;
            vec->z *= airRegist;
        }
    }
    else
    {
        // 空気抵抗が無ければ値をコピー
        if( time > 0 )
        {
            // 初回フレームのみ
            EFT_F32_VEC3_COPY( vec, vecBack );
        }
    }
    // この時点でpos,vecのフロントバッファへの更新処理が完了。

    // 重力
    const f32 gravScaleAnim = emitter->GetEmitterAnimValue().gravityScale;
    if ( gravScaleAnim > 0.f )
    {
        const f32 ubGravityX  = emitter->emitterData->emission.gravityDir.x * gravScaleAnim;
        const f32 ubGravityY  = emitter->emitterData->emission.gravityDir.y * gravScaleAnim;
        const f32 ubGravityZ  = emitter->emitterData->emission.gravityDir.z * gravScaleAnim;

        if( emitter->emitterData->emission.isWorldGravity )
        {
            if ( emitter->emitterData->emitter.followType == EFT_EMITTER_FOLLOW_TYPE_NONE )
            {
                vec->x += frameRate * (ubGravityX * ptclAttr->emitterRTMat0.x + ubGravityY * ptclAttr->emitterRTMat1.x + ubGravityZ * ptclAttr->emitterRTMat2.x);
                vec->y += frameRate * (ubGravityX * ptclAttr->emitterRTMat0.y + ubGravityY * ptclAttr->emitterRTMat1.y + ubGravityZ * ptclAttr->emitterRTMat2.y);
                vec->z += frameRate * (ubGravityX * ptclAttr->emitterRTMat0.z + ubGravityY * ptclAttr->emitterRTMat1.z + ubGravityZ * ptclAttr->emitterRTMat2.z);
            }
            else
            {
                vec->x += frameRate * (ubGravityX * emitter->matrixRT.m[0][0] + ubGravityY * emitter->matrixRT.m[1][0] + ubGravityZ * emitter->matrixRT.m[2][0]);
                vec->y += frameRate * (ubGravityX * emitter->matrixRT.m[0][1] + ubGravityY * emitter->matrixRT.m[1][1] + ubGravityZ * emitter->matrixRT.m[2][1]);
                vec->z += frameRate * (ubGravityX * emitter->matrixRT.m[0][2] + ubGravityY * emitter->matrixRT.m[1][2] + ubGravityZ * emitter->matrixRT.m[2][2]);
            }
        }
        else
        {
            vec->x += frameRate * ubGravityX;
            vec->y += frameRate * ubGravityY;
            vec->z += frameRate * ubGravityZ;
        }
    }


    // フィールド
    if ( emitter->emitterRes->fieldUniformBlock )
    {
        if ( emitter->emitterRes->fieldRandomData )      _CalcParticleBehavior_FieldRandom( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldRandomDataFe1 )   _CalcParticleBehavior_FieldRandomFe1( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldMagnetData )      _CalcParticleBehavior_FieldMagnet( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldSpinData )        _CalcParticleBehavior_FieldSpin( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldCollisionData )   _CalcParticleBehavior_FieldCollision( emitter, pos, vec, ptclAttr, ptclData );
        if ( emitter->emitterRes->fieldConvergenceData ) _CalcParticleBehavior_FieldConvergence( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldCurlNoiseData )   _CalcParticleBehavior_FieldCurlNoise( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldPosAddData )      _CalcParticleBehavior_FieldPosAdd( emitter, pos, vec, ptclAttr );
        if ( emitter->emitterRes->fieldCustomData )      _CalcParticleBehavior_CustomField( emitter, pos, vec, ptclAttr, index );
    }

#ifdef EFT_DEGRADATION_SPEC
    // 機能削減版では、CPUで位置・速度以外のパラメータも計算する

    ResEmitter* emitterData = emitter->emitterRes->emitterData;
    f32 life = pos->w;

    nw::math::VEC4 iniVec4;
    iniVec4.x = iniVec4.y = iniVec4.z = iniVec4.w = 1.0f;
    nw::math::VEC3 iniVec3;
    iniVec3.x = iniVec3.y = iniVec3.z = 1.0f;

    // カラー0
    if( emitterData->ptclColor.color0Type != EFT_PARTICLE_COLOR_CALC_TYPE_FIXED || emitterData->ptclColor.alpha0Type != EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        CalcPtclColor0Vec( &ptclAttr->initColor0, emitter->emitterRes, &ptclAttr->random, &iniVec4, &iniVec3, 1.0f, life, time );
    }

    // カラー1
    if( emitterData->ptclColor.color1Type != EFT_PARTICLE_COLOR_CALC_TYPE_FIXED || emitterData->ptclColor.alpha1Type != EFT_PARTICLE_COLOR_CALC_TYPE_FIXED )
    {
        CalcPtclColor1Vec( &ptclAttr->initColor1, emitter->emitterRes, &ptclAttr->random, &iniVec4, &iniVec3, 1.0f, life, time );
    }

    // スケール
    if( emitterData->staticUbo.scaleAnimKeyNum > 1 || emitterData->ptclFluctuation.isApplayScale )
    {
        CalcPtclScaleVecPerFrame( &ptclAttr->scale, emitter->emitterRes, &ptclData->initScale, &ptclAttr->random, life, time );
    }

    // 回転
    if ( emitterData->ptcl.isRotateX || emitterData->ptcl.isRotateY || emitterData->ptcl.isRotateZ )
    {
        nw::math::VEC4 rotateInit;
        rotateInit.x = rotateInit.y = rotateInit.z = rotateInit.w = 0.0f;
        CalcRotationMatrix( &ptclAttr->initRotate, emitter->emitterRes, &ptclAttr->random, time );
    }

    // テクスチャ0 パタンアニメ
    if ( emitterData->textureAnim0.patternAnimType != EFT_TEXTURE_PTN_ANIM_TYPE_NONE )
    {
        f32 index = static_cast<f32>(
            _CalcFluctuation_TexPatternAnim( time, life, ptclAttr->random.x, &emitterData->staticUbo.texPtnAnim0, &emitterData->textureAnim0 ) );
        ptclAttr->initRotate.w = index * 0.01f + 0.005f;
    }
    else
    {
        ptclAttr->initRotate.w = 0.0f;
    }

    // テクスチャ1 パタンアニメ
    if ( emitterData->textureAnim1.patternAnimType != EFT_TEXTURE_PTN_ANIM_TYPE_NONE )
    {
        ptclAttr->initRotate.w += static_cast<f32>(
            _CalcFluctuation_TexPatternAnim( time, life, ptclAttr->random.x, &emitterData->staticUbo.texPtnAnim1, &emitterData->textureAnim1 ) );
    }
#endif
}

//------------------------------------------------------------------------------
// 現在のパーティクルスケールを計算する
//------------------------------------------------------------------------------
void CalcPtclScaleVecFromTime( nw::math::VEC4*         dstScale,
                              const EmitterResource*   emitterRes,
                              const nw::math::VEC4*    initScale,
                              const nw::math::VEC4*    random,
                              f32                      life,
                              f32                      time )
{
    EFT_F32_VEC3_COPY( dstScale, initScale );
    dstScale->w = 1.0f;

    if ( !emitterRes->resEmitterStaticUniformBlock ) return;

    //アニメ
    if( emitterRes->resEmitterStaticUniformBlock->scaleAnimKeyNum > 1 )
    {
        f32 loopRate;
        if( emitterRes->emitterData->ptcl.scaleAnimIsLoop )
        {
            loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.scaleAnimLoopRate );
        }
        else
        {
            loopRate = 0.0f;
        }

        nw::math::VEC3 sAnim;
        Calculate8KeyAnim( &sAnim,
                           emitterRes->resEmitterStaticUniformBlock->scale8keyAnim,
                           emitterRes->resEmitterStaticUniformBlock->scaleAnimKeyNum,
                           random->x, time, loopRate,
                           static_cast<f32>( emitterRes->emitterData->ptcl.scaleAnimIsLoopInitRandom ),
                           life );

        dstScale->x *= sAnim.x;
        dstScale->y *= sAnim.y;
        dstScale->z *= sAnim.z;
    }
    else
    {
        dstScale->x *= emitterRes->resEmitterStaticUniformBlock->scale8keyAnim.value[0].x;
        dstScale->y *= emitterRes->resEmitterStaticUniformBlock->scale8keyAnim.value[0].y;
        dstScale->z *= emitterRes->resEmitterStaticUniformBlock->scale8keyAnim.value[0].z;
    }

    // 揺らぎ
    // TODO: Z？
    if( emitterRes->emitterData->ptclFluctuation.isApplayScale )
    {
        f32 flucX = 1.0f;
        f32 flucY = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.x;
        fluc.cycle.x       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.x;
        fluc.phaseRnd.x    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.x;

        fluc.amplitude.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.y;
        fluc.cycle.y       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.y;
        fluc.phaseRnd.y    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.y;

        const u32 index = emitterRes->emitterData->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case EFT_FLUCTUATION_WAVE_TYPE_SIN:
            flucX = _CalcFluctuation_Sin( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
            flucX = _CalcFluctuation_SawTooth( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_RECT:
            flucX = _CalcFluctuation_Rect( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }
        flucY = flucX;

        if( emitterRes->emitterData->ptclFluctuation.isApplayScaleY )
        {
            //軸を個別に適用
            switch( index )
            {
            case EFT_FLUCTUATION_WAVE_TYPE_SIN:
                flucY = _CalcFluctuation_Sin( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
                flucY = _CalcFluctuation_SawTooth( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case EFT_FLUCTUATION_WAVE_TYPE_RECT:
                flucY = _CalcFluctuation_Rect( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            default:
                break;
            }
        }

        // 揺らぎは乗算処理
        dstScale->x *= flucX;
        dstScale->y *= flucY;
    }

    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクルスケールを計算する
//------------------------------------------------------------------------------
void CalcPtclScaleVecPerFrame( nw::math::VEC4*          scaleParam,
                               const EmitterResource*   emitterRes,
                               const nw::math::VEC4*    initScale,
                               const nw::math::VEC4*    random,
                               f32                      life,
                               f32                      time )
{
    scaleParam->x = initScale->x;
    scaleParam->y = initScale->y;
    scaleParam->z = initScale->z;
    scaleParam->w = 1.0f;

    if ( !emitterRes->resEmitterStaticUniformBlock ) return;

    //アニメ
    if( emitterRes->resEmitterStaticUniformBlock->scaleAnimKeyNum > 1 )
    {
        f32 loopRate;
        if( emitterRes->emitterData->ptcl.scaleAnimIsLoop )
        {
            loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.scaleAnimLoopRate );
        }
        else
        {
            loopRate = 0.0f;
        }

        nw::math::VEC3 sAnim;
        Calculate8KeyAnim( &sAnim,
            emitterRes->resEmitterStaticUniformBlock->scale8keyAnim,
            emitterRes->resEmitterStaticUniformBlock->scaleAnimKeyNum,
            random->x, time, loopRate,
            static_cast<f32>( emitterRes->emitterData->ptcl.scaleAnimIsLoopInitRandom ),
            life );

        scaleParam->x *= sAnim.x;
        scaleParam->y *= sAnim.y;
        scaleParam->z *= sAnim.z;
    }

    // 揺らぎ
    if( emitterRes->emitterData->ptclFluctuation.isApplayScale )
    {
        f32 flucX = 1.0f;
        f32 flucY = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.x;
        fluc.cycle.x       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.x;
        fluc.phaseRnd.x    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.x;

        fluc.amplitude.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.y;
        fluc.cycle.y       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.y;
        fluc.phaseRnd.y    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.y;

        const u32 index = emitterRes->emitterData->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case EFT_FLUCTUATION_WAVE_TYPE_SIN:
            flucX = _CalcFluctuation_Sin( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
            flucX = _CalcFluctuation_SawTooth( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_RECT:
            flucX = _CalcFluctuation_Rect( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }
        flucY = flucX;

        if( emitterRes->emitterData->ptclFluctuation.isApplayScaleY )
        {
            //軸を個別に適用
            switch( index )
            {
            case EFT_FLUCTUATION_WAVE_TYPE_SIN:
                flucY = _CalcFluctuation_Sin( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
                flucY = _CalcFluctuation_SawTooth( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            case EFT_FLUCTUATION_WAVE_TYPE_RECT:
                flucY = _CalcFluctuation_Rect( time, fluc.amplitude.y, fluc.cycle.y, fluc.phaseInit.y, fluc.phaseRnd.y, random );
                break;
            default:
                break;
            }
        }

        // 揺らぎは乗算処理
        scaleParam->x *= flucX;
        scaleParam->y *= flucY;
    }

    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクルカラー0を計算する
//------------------------------------------------------------------------------
void CalcPtclColor0Vec( nw::math::VEC4*         dstColor,
                        const EmitterResource*  emitterRes,
                        const nw::math::VEC4*   random,
                        const nw::math::VEC4*   emitterColor,
                        const nw::math::VEC3*   emitterAnimColor,
                        f32                     emitterAnimAlpha,
                        f32                     life,
                        f32                     time )
{
    // 基本値
    EFT_F32_VEC3_COPY( dstColor, &emitterRes->emitterData->ptclColor.color0 );
    dstColor->w = emitterRes->emitterData->ptclColor.alpha0;
    if ( !emitterRes->resEmitterStaticUniformBlock ) return;

    // ランダム
    if( emitterRes->emitterData->ptclColor.color0Type == EFT_PARTICLE_COLOR_CALC_TYPE_RANDOM )
    {
        int index = int ( random->x * emitterRes->resEmitterStaticUniformBlock->color0AnimKeynum );
        EFT_F32_VEC3_COPY( dstColor, &emitterRes->resEmitterStaticUniformBlock->color0Anim.value[index] );
    }

    // 8キーアニメ
    if( emitterRes->emitterData->ptclColor.color0Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY )
    {
        if( emitterRes->resEmitterStaticUniformBlock->color0AnimKeynum > 0 )
        {
            f32 loopRate;
            if(emitterRes->emitterData->ptcl.color0AnimIsLoop)
            {
                loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.color0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nw::math::VEC3 c;
            Calculate8KeyAnim( &c,
                               emitterRes->resEmitterStaticUniformBlock->color0Anim,
                               emitterRes->resEmitterStaticUniformBlock->color0AnimKeynum,
                               random->x, time, loopRate,
                               static_cast<f32>( emitterRes->emitterData->ptcl.color0AnimIsLoopInitRandom ),
                               life );
            EFT_F32_VEC3_COPY( dstColor, &c );
        }
    }
    if( emitterRes->emitterData->ptclColor.alpha0Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY )
    {
        if( emitterRes->resEmitterStaticUniformBlock->alpha0AnimKeyNum > 0 )
        {
            f32 loopRate;
            if(emitterRes->emitterData->ptcl.alpha0AnimIsLoop)
            {
                loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.alpha0AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nw::math::VEC3 a;
            Calculate8KeyAnim( &a,
                               emitterRes->resEmitterStaticUniformBlock->alpha0Anim,
                               emitterRes->resEmitterStaticUniformBlock->alpha0AnimKeyNum,
                               random->x, time, loopRate,
                               static_cast<f32>( emitterRes->emitterData->ptcl.alpha0AnimIsLoopInitRandom ),
                               life );
            dstColor->w = a.x;
        }
    }

    // エミッタカラー / カラースケール
    const f32 colorScale = emitterRes->resEmitterStaticUniformBlock->colorScale;
    dstColor->x *= ( emitterColor->x * emitterAnimColor->x * colorScale );
    dstColor->y *= ( emitterColor->y * emitterAnimColor->y * colorScale );
    dstColor->z *= ( emitterColor->z * emitterAnimColor->z * colorScale );
    dstColor->w *= ( emitterColor->w * emitterAnimAlpha );

    // アルファ揺らぎ
    if( emitterRes->emitterData->ptclFluctuation.isApplyAlpha )
    {
        f32 flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.x;
        fluc.cycle.x       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.x;
        fluc.phaseRnd.x    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.x;

        fluc.amplitude.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.y;
        fluc.cycle.y       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.y;
        fluc.phaseRnd.y    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.y;

        const u32 index = emitterRes->emitterData->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case EFT_FLUCTUATION_WAVE_TYPE_SIN:
            flucX = _CalcFluctuation_Sin( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
            flucX = _CalcFluctuation_SawTooth( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_RECT:
            flucX = _CalcFluctuation_Rect( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        dstColor->w *= flucX;
        if( dstColor->w < 0 ){ dstColor->w = 0; }
        if( dstColor->w > 1 ){ dstColor->w = 1; }
    }

    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクルカラー１を計算する
//------------------------------------------------------------------------------
void CalcPtclColor1Vec( nw::math::VEC4*         dstColor,
                        const EmitterResource*  emitterRes,
                        const nw::math::VEC4*   random,
                        const nw::math::VEC4*   emitterColor,
                        const nw::math::VEC3*   emitterAnimColor,
                        f32                     emitterAnimAlpha,
                        f32                     life,
                        f32                     time )
{
    // 基本値
    EFT_F32_VEC3_COPY( dstColor, &emitterRes->emitterData->ptclColor.color1 );
    dstColor->w = emitterRes->emitterData->ptclColor.alpha1;
    if ( !emitterRes->resEmitterStaticUniformBlock ) return;

    // ランダム
    if( emitterRes->emitterData->ptclColor.color1Type == EFT_PARTICLE_COLOR_CALC_TYPE_RANDOM )
    {
        int index = int ( random->x * emitterRes->resEmitterStaticUniformBlock->color1AnimKeynum );
        EFT_F32_VEC3_COPY( dstColor, &emitterRes->resEmitterStaticUniformBlock->color1Anim.value[index] );
    }
    // 8キーアニメ
    if( emitterRes->emitterData->ptclColor.color1Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY )
    {

        if( emitterRes->resEmitterStaticUniformBlock->color1AnimKeynum > 0 )
        {
            f32 loopRate;
            if( emitterRes->emitterData->ptcl.color1AnimIsLoop )
            {
                loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.color1AnimLoopRate );
            }
            else
            {
                loopRate = 0.0f;
            }

            nw::math::VEC3 c;
            Calculate8KeyAnim(
                &c,
                emitterRes->resEmitterStaticUniformBlock->color1Anim,
                emitterRes->resEmitterStaticUniformBlock->color1AnimKeynum,
                random->x, time, loopRate,
                static_cast<f32>( emitterRes->emitterData->ptcl.color1AnimIsLoopInitRandom ),
                life );

            EFT_F32_VEC3_COPY( dstColor, &c );
        }
    }
    if( emitterRes->emitterData->ptclColor.alpha1Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY )
    {
        if( emitterRes->resEmitterStaticUniformBlock->alpha1AnimKeyNum > 0 )
        {

            f32 loopRate;
            if( emitterRes->emitterData->ptcl.alpha1AnimIsLoop)
            {
                loopRate = static_cast<f32>( emitterRes->emitterData->ptcl.alpha1AnimLoopRate);
            }
            else
            {
                loopRate = 0.0f;
            }

            nw::math::VEC3 a;
            Calculate8KeyAnim(
                &a,
                emitterRes->resEmitterStaticUniformBlock->alpha1Anim,
                emitterRes->resEmitterStaticUniformBlock->alpha1AnimKeyNum,
                random->x, time, loopRate,
                static_cast<f32>( emitterRes->emitterData->ptcl.alpha1AnimIsLoopInitRandom ),
                life );

            dstColor->w = a.x;
        }
    }

    // エミッタカラー / カラースケール
    const f32 colorScale = emitterRes->resEmitterStaticUniformBlock->colorScale;
    dstColor->x *= ( emitterColor->x * emitterAnimColor->x * colorScale );
    dstColor->y *= ( emitterColor->y * emitterAnimColor->y * colorScale );
    dstColor->z *= ( emitterColor->z * emitterAnimColor->z * colorScale );
    dstColor->w *= ( emitterColor->w * emitterAnimAlpha );

    // アルファ揺らぎ
    if( emitterRes->emitterData->ptclFluctuation.isApplyAlpha )
    {
        f32 flucX = 1.0f;
        FluctuationData fluc;

        fluc.amplitude.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.x;
        fluc.cycle.x       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.x;
        fluc.phaseRnd.x    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.x;
        fluc.phaseInit.x   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.x;

        fluc.amplitude.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.amplitude.y;
        fluc.cycle.y       = emitterRes->resEmitterStaticUniformBlock->fluctuationData.cycle.y;
        fluc.phaseRnd.y    = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseRnd.y;
        fluc.phaseInit.y   = emitterRes->resEmitterStaticUniformBlock->fluctuationData.phaseInit.y;

        const u32 index = emitterRes->emitterData->ptclFluctuation.isWaveType >> 4;
        switch( index )
        {
        case EFT_FLUCTUATION_WAVE_TYPE_SIN:
            flucX = _CalcFluctuation_Sin( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH:
            flucX = _CalcFluctuation_SawTooth( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        case EFT_FLUCTUATION_WAVE_TYPE_RECT:
            flucX = _CalcFluctuation_Rect( time, fluc.amplitude.x, fluc.cycle.x, fluc.phaseInit.x, fluc.phaseRnd.x, random );
            break;
        default:
            break;
        }

        dstColor->w *= flucX;
        if( dstColor->w < 0 ){ dstColor->w = 0; }
        if( dstColor->w > 1 ){ dstColor->w = 1; }
    }

    return;
}

//------------------------------------------------------------------------------
// 現在のパーティクル回転値を計算する
//------------------------------------------------------------------------------
void CalcRotationMatrix( nw::math::VEC4*        dstRotate,
                         const EmitterResource* emitterRes,
                         const nw::math::VEC4*  initRotate,
                         const nw::math::VEC4*  random,
                         f32                    time )
{
    dstRotate->x = initRotate->x;
    dstRotate->y = initRotate->y;
    dstRotate->z = initRotate->z;
    if ( !emitterRes->resEmitterStaticUniformBlock ) return;

    const nw::math::VEC4& rInitRand = emitterRes->resEmitterStaticUniformBlock->rotateInitRand;
    const nw::math::VEC3& rAdd      = emitterRes->resEmitterStaticUniformBlock->rotateAdd;
    const nw::math::VEC3& rAddRand  = emitterRes->resEmitterStaticUniformBlock->rotateAddRand;
    const f32 rotVelRegist          = emitterRes->resEmitterStaticUniformBlock->rotateRegist;

    f32 rotInitX    = initRotate->x; // 初期値 X
    f32 rotInitY    = initRotate->y; // 初期値 Y
    f32 rotInitZ    = initRotate->z; // 初期値 Z
    f32 rotInitRndX = rInitRand.x;   // 初期値ランダム X
    f32 rotInitRndY = rInitRand.y;   // 初期値ランダム Y
    f32 rotInitRndZ = rInitRand.z;   // 初期値ランダム Z
    f32 rotVelX     = rAdd.x;        // 加算値 X
    f32 rotVelY     = rAdd.y;        // 加算値 Y
    f32 rotVelZ     = rAdd.z;        // 加算値 Z
    f32 rotVelRandX = rAddRand.x;    // 加算値ランダム X
    f32 rotVelRandY = rAddRand.y;    // 加算値ランダム Y
    f32 rotVelRandZ = rAddRand.z;    // 加算値ランダム Z

    // 乱数
    f32 ptclRandomX = random->x;
    f32 ptclRandomY = random->y;
    f32 ptclRandomZ = random->z;

    // 乱数を増やす
    f32 tempRandX = ( ptclRandomX + ptclRandomY ) / 2;
    f32 tempRandY = ( ptclRandomY + ptclRandomZ ) / 2;
    f32 tempRandZ = ( ptclRandomZ + ptclRandomX ) / 2;

    // 回転速度（ランダム量込み）
    f32 rVelX = rotVelX + ( tempRandX * rotVelRandX );
    f32 rVelY = rotVelY + ( tempRandY * rotVelRandY );
    f32 rVelZ = rotVelZ + ( tempRandZ * rotVelRandZ );

    // 回転方向ランダム
    if( emitterRes->emitterData->ptcl.rotRevRandX && ( ptclRandomY >= 0.5f ) )
    {
        rVelX    = -rVelX;
        rotInitX = -rotInitX;
    }
    if( emitterRes->emitterData->ptcl.rotRevRandY && ( ptclRandomZ >= 0.5f ) )
    {
        rVelY    = -rVelY;
        rotInitY = -rotInitY;
    }
    if( emitterRes->emitterData->ptcl.rotRevRandZ && ( ptclRandomX >= 0.5f ) )
    {
        rVelZ    = -rVelZ;
        rotInitZ = -rotInitZ;
    }

    // 回転初期値（ランダム量込み）
    f32 rotX = rotInitX + ( ptclRandomX * rotInitRndX );
    f32 rotY = rotInitY + ( ptclRandomY * rotInitRndY );
    f32 rotZ = rotInitZ + ( ptclRandomZ * rotInitRndZ );

    // 減衰率を計算
    f32 rotRegist = pow( rotVelRegist, time );
    f32 rotTime = ( rotVelRegist == 1.0 ) ? time : ( 1 - rotRegist )/( 1 - rotVelRegist );

    // 現在時刻の回転量を計算
    dstRotate->x = ( rVelX * rotTime ) + rotX;
    dstRotate->y = ( rVelY * rotTime ) + rotY;
    dstRotate->z = ( rVelZ * rotTime ) + rotZ;

    return;
}

//---------------------------------------------------
//! @brief  現在のパーティクル回転値を計算する
//---------------------------------------------------
void CalcRotationMatrix( nw::math::VEC4* rotateParam, const EmitterResource* emitterRes, const nw::math::VEC4* random, f32 time )
{
    EFT_UNUSED_VARIABLE( random );

    const nw::math::VEC4& rInit     = emitterRes->resEmitterStaticUniformBlock->rotateInit;
    const nw::math::VEC4& rInitRand = emitterRes->resEmitterStaticUniformBlock->rotateInitRand;
    const nw::math::VEC3& rAdd      = emitterRes->resEmitterStaticUniformBlock->rotateAdd;
    const nw::math::VEC3& rAddRand  = emitterRes->resEmitterStaticUniformBlock->rotateAddRand;
    const f32 rotVelRegist          = emitterRes->resEmitterStaticUniformBlock->rotateRegist;

    f32 rotInitRndX = rInitRand.x;   // 初期値ランダム X
    f32 rotInitRndY = rInitRand.y;   // 初期値ランダム Y
    f32 rotInitRndZ = rInitRand.z;   // 初期値ランダム Z
    f32 rotVelX     = rAdd.x;        // 加算値 X
    f32 rotVelY     = rAdd.y;        // 加算値 Y
    f32 rotVelZ     = rAdd.z;        // 加算値 Z
    f32 rotVelRandX = rAddRand.x;    // 加算値ランダム X
    f32 rotVelRandY = rAddRand.y;    // 加算値ランダム Y
    f32 rotVelRandZ = rAddRand.z;    // 加算値ランダム Z

    // 乱数
    f32 ptclRandomX = random->x;
    f32 ptclRandomY = random->y;
    f32 ptclRandomZ = random->z;

    // 乱数を増やす
    f32 tempRandX = ( ptclRandomX + ptclRandomY ) / 2;
    f32 tempRandY = ( ptclRandomY + ptclRandomZ ) / 2;
    f32 tempRandZ = ( ptclRandomZ + ptclRandomX ) / 2;

    // 回転初期値（ランダム量込み）
    f32 rotX = rInit.x + ( ptclRandomX * rotInitRndX );
    f32 rotY = rInit.y + ( ptclRandomY * rotInitRndY );
    f32 rotZ = rInit.z + ( ptclRandomZ * rotInitRndZ );

    // 回転速度（ランダム量込み）
    f32 rVelX = rotVelX + ( tempRandX * rotVelRandX );
    f32 rVelY = rotVelY + ( tempRandY * rotVelRandY );
    f32 rVelZ = rotVelZ + ( tempRandZ * rotVelRandZ );

    // 減衰率を計算
    f32 rotRegist = pow( rotVelRegist, time );
    f32 rotTime = ( rotVelRegist == 1.0 ) ? time : ( 1 - rotRegist )/( 1 - rotVelRegist );

    // 現在時刻の回転量を計算
    rotateParam->x = ( rVelX * rotTime ) + rotX;
    rotateParam->y = ( rVelY * rotTime ) + rotY;
    rotateParam->z = ( rVelZ * rotTime ) + rotZ;
}


//------------------------------------------------------------------------------
// XYZ 回転行列を生成します。
//------------------------------------------------------------------------------
void MakeRotationMatrixXYZ( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate )
{
    //回転の軸は常にWorld空間に固定されます

    f32 sinX = sin( rotate.x );
    f32 cosX = cos( rotate.x );

    f32 sinY = sin( rotate.y );
    f32 cosY = cos( rotate.y );

    f32 sinZ = sin( rotate.z );
    f32 cosZ = cos( rotate.z );

    f32 opt1 = cosX * sinZ;
    f32 opt2 = cosX * cosZ;
    f32 opt3 = sinX * sinZ;
    f32 opt4 = sinX * cosZ;

    dstMatrix->m[0][0] = cosY * cosZ;         // (cosY * cosZ)
    dstMatrix->m[0][1] = cosY * sinZ;         // (cosY * sinZ)
    dstMatrix->m[0][2] = -sinY;               //-(sinY)
    dstMatrix->m[0][3] = 0.0;

    dstMatrix->m[1][0] = opt4 * sinY - opt1;  // (sinX * sinY * cosZ) - (cosX * sinZ)
    dstMatrix->m[1][1] = opt3 * sinY + opt2;  // (sinX * sinY * sinZ) + (cosX * cosZ)
    dstMatrix->m[1][2] = sinX * cosY;         // (sinX * cosY)
    dstMatrix->m[1][3] = 0.0;

    dstMatrix->m[2][0] = opt2 * sinY + opt3;  // (cosX * sinY * cosZ) + (sinX * sinZ)
    dstMatrix->m[2][1] = opt1 * sinY - opt4;  // (cosX * sinY * sinZ) - (sinX * sinZ)
    dstMatrix->m[2][2] = cosX * cosY;         // (cosX * cosY)
    dstMatrix->m[2][3] = 0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;
}



#if 0
//------------------------------------------------------------------------------
// 回転行列生成 ( YZX順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixYZX( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate )
{
    //回転の軸は常にWorld空間に固定されます

    nw::math::MTX44 rotMat;

    float sinX = sin( -rotate.x );
    float cosX = cos( -rotate.x );

    float sinY = sin( -rotate.y );
    float cosY = cos( -rotate.y );

    float sinZ = sin( -rotate.z );
    float cosZ = cos( -rotate.z );

    float opt1 = sinX * sinY;
    float opt2 = cosX * sinY;
    float opt3 = sinX * cosY;
    float opt4 = cosX * cosY;

    dstMatrix->m[0][0] =  cosY * cosZ;        // (cosY * cosZ)
    dstMatrix->m[0][1] =  opt4 * sinZ + opt1; // (cosX * cosY * sinZ) + (sinX * sinY)
    dstMatrix->m[0][2] =  opt3 * sinZ - opt2; // (sinX * cosY * sinZ) - (cosX * sinY)
    dstMatrix->m[0][3] =  0.0;

    dstMatrix->m[1][0] = -sinZ;               //-(sinZ)
    dstMatrix->m[1][1] =  cosZ * cosX;        // (cosX * cosZ)
    dstMatrix->m[1][2] =  cosZ * sinX;        // (sinX * cosZ)
    dstMatrix->m[1][3] =  0.0;

    dstMatrix->m[2][0] =  sinY * cosZ;        // (sinY * cosZ)
    dstMatrix->m[2][1] =  opt2 * sinZ - opt3; // (cosX * sinY * sinZ) - (sinX * cosY)
    dstMatrix->m[2][2] =  opt1 * sinZ + opt4; //-(sinX * sinY * sinZ) + (cosX * cosY)
    dstMatrix->m[2][3] =  0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;

    return;
}

//------------------------------------------------------------------------------
// 回転行列生成 ( XYZ順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixXYZ( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate )
{
    //回転の軸は常にWorld空間に固定されます

    nw::math::MTX44 rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosX * sinZ;
    float opt2 = cosX * cosZ;
    float opt3 = sinX * sinZ;
    float opt4 = sinX * cosZ;

    dstMatrix->m[0][0] = cosY * cosZ;         // (cosY * cosZ)
    dstMatrix->m[0][1] = cosY * sinZ;         // (cosY * sinZ)
    dstMatrix->m[0][2] = -sinY;               //-(sinY)
    dstMatrix->m[0][3] = 0.0;

    dstMatrix->m[1][0] = opt4 * sinY - opt1;  // (sinX * sinY * cosZ) - (cosX * sinZ)
    dstMatrix->m[1][1] = opt3 * sinY + opt2;  // (sinX * sinY * sinZ) + (cosX * cosZ)
    dstMatrix->m[1][2] = sinX * cosY;         // (sinX * cosY)
    dstMatrix->m[1][3] = 0.0;

    dstMatrix->m[2][0] = opt2 * sinY + opt3;  // (cosX * sinY * cosZ) + (sinX * sinZ)
    dstMatrix->m[2][1] = opt1 * sinY - opt4;  // (cosX * sinY * sinZ) - (sinX * sinZ)
    dstMatrix->m[2][2] = cosX * cosY;         // (cosX * cosY)
    dstMatrix->m[2][3] = 0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;

    return;
}

//------------------------------------------------------------------------------
// 回転行列生成 ( ZXY順の回転 )
//------------------------------------------------------------------------------
void MakeRotationMatrixZXY( nw::math::MTX44* dstMatrix, const nw::math::VEC3& rotate )
{
    //回転の軸は常にWorld空間に固定されます
    nw::math::MTX44 rotMat;

    float sinX = sin( rotate.x );
    float cosX = cos( rotate.x );

    float sinY = sin( rotate.y );
    float cosY = cos( rotate.y );

    float sinZ = sin( rotate.z );
    float cosZ = cos( rotate.z );

    float opt1 = cosY * cosZ;
    float opt2 = cosY * sinZ;
    float opt3 = sinY * cosZ;
    float opt4 = sinY * sinZ;

    dstMatrix->m[0][0] = sinX * opt4 + opt1;  // (sinX * sinY * sinZ) + (cosY * cosZ)
    dstMatrix->m[0][1] = cosX * sinZ;         // (cosX * sinZ)
    dstMatrix->m[0][2] = sinX * opt2 - opt3;  // (sinX * cosY * sinZ) - (sinY * cosZ)
    dstMatrix->m[0][3] = 0.0;

    dstMatrix->m[1][0] = sinX * opt3 - opt2;  // (sinX * sinY * cosZ) - (cosY * sinZ)
    dstMatrix->m[1][1] = cosX * cosZ;         // (cosX * cosZ)
    dstMatrix->m[1][2] = sinX * opt1 + opt4;  // (sinX * cosY * cosZ) + (sinY * sinZ)
    dstMatrix->m[1][3] = 0.0;

    dstMatrix->m[2][0] = cosX * sinY;         // (cosX * sinY)
    dstMatrix->m[2][1] = -sinX;               //-(sinX)
    dstMatrix->m[2][2] = cosX * cosY;         // (cosX * cosY)
    dstMatrix->m[2][3] = 0.0;

    dstMatrix->m[3][0] = 0.0;
    dstMatrix->m[3][1] = 0.0;
    dstMatrix->m[3][2] = 0.0;
    dstMatrix->m[3][3] = 1.0;

    return;
}
#endif


} // namespace eft2
} // namespace nw
