﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Heap.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_EmitterSet.h>
#include <nw/eft/eft2_Emitter.h>

namespace nw   {
namespace eft2 {


u32                     g_StaticHeapAllocedSize     = 0;
Heap*                   g_StaticHeap                = NULL;
Heap*                   g_DynamicHeap               = NULL;
bool                    g_SuppressLog               = false;
OutputMessageCallback   g_LogCallback               = NULL;
OutputMessageCallback   g_WarningCallback           = NULL;
OutputMessageCallback   g_ErrorCallback             = NULL;


// 動的ヒープから遅延解放するアドレス情報
#define  _DELAY_FREE_TIME   (3)

void**   g_DelayFreeList[EFT_CPU_CORE_MAX][_DELAY_FREE_TIME];
u32      g_DelayFreeListNum[EFT_CPU_CORE_MAX][_DELAY_FREE_TIME];
u32      g_DelayFreeListMax  = 0;
u32      g_DelayFreeListCntr = 0;
u32      g_DelayFreeTime     = 3;

u32      g_Warning           = 0;

//---------------------------------------------------------------------------
//  静的ヒープからメモリを設定します。
//---------------------------------------------------------------------------
void SetStaticHeap( Heap* staticHeap )
{
    if ( staticHeap && g_StaticHeap )
    {
        OutputError( "StaticHeap is Initialized." );
    }

    g_StaticHeap = staticHeap;
}

//---------------------------------------------------------------------------
//  静的ヒープからメモリを確保します。
//---------------------------------------------------------------------------
void* AllocFromStaticHeap( u32 size, u32 align )
{
    u32 allocSize =  nw::eft2::RoundUp( size, 32 );
    void* ptr = g_StaticHeap->Alloc( allocSize, align );
    EFT_NULL_ASSERT( ptr );
    g_StaticHeapAllocedSize += allocSize;
    return ptr;
}

//---------------------------------------------------------------------------
//  静的ヒープからメモリを解放します。
//---------------------------------------------------------------------------
void FreeFromStaticHeap( void* ptr )
{
    return g_StaticHeap->Free( ptr );
}

//---------------------------------------------------------------------------
//  静的ヒープから確保されたメモリサイズを取得します。
//---------------------------------------------------------------------------
u32 GetAllocedSizeFromStaticHeap()
{
    return g_StaticHeapAllocedSize;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを設定します。
//---------------------------------------------------------------------------
void SetDynamicHeap( Heap* dynamicHeap )
{
    if ( dynamicHeap && g_DynamicHeap )
    {
        OutputError( "DynamicHeap is Initialized." );
    }

    g_DynamicHeap = dynamicHeap;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを確保します。
//---------------------------------------------------------------------------
void* AllocFromDynamicHeap( u32 size, u32 align )
{
    u32 allocSize =  nw::eft2::RoundUp( size, 0x100 );
    void* ptr = g_DynamicHeap->Alloc( allocSize, align );
    if ( !ptr )
    {
        OutputWarning( "DynamicHeap Alloc Failed. size : %d Byte\n", allocSize );
    }
    return ptr;
}

//---------------------------------------------------------------------------
//  動的ヒープからメモリを解放します。
//---------------------------------------------------------------------------
void FreeFromDynamicHeap( void* ptr, bool immediate )
{
    if ( immediate )
    {
        return g_DynamicHeap->Free( ptr );
    }
    else
    {
        // 動的ヒープ free リストへつなぐ
        return AddFreeListForDynamicHeap( ptr );
    }
}

//---------------------------------------------------------------------------
//  遅延解放用のリストを初期化します。
//---------------------------------------------------------------------------
void InitializeDelayFreeList( u32 freeListNum, u32 delayTime )
{
    if ( !g_StaticHeap )
    {
        OutputError( "StaticHeap is not Initialized." );
    }

    g_DelayFreeListMax = freeListNum;
    g_DelayFreeTime    = delayTime;

    for( u32 i = 0; i < EFT_CPU_CORE_MAX; i++ )
    {
        for( u32 j = 0; j < _DELAY_FREE_TIME; j++ )
        {
            g_DelayFreeList[i][j] = ( void** )AllocFromStaticHeap( sizeof( void* ) * g_DelayFreeListMax );
            MemUtil::FillZero( g_DelayFreeList[i][j], sizeof( void* ) * g_DelayFreeListMax );
            g_DelayFreeListNum[i][j] = 0;
        }
    }
}

//---------------------------------------------------------------------------
//  遅延解放用のリストを破棄します。
//---------------------------------------------------------------------------
void FinalizeDelayFreeList()
{
    for( u32 i = 0; i < EFT_CPU_CORE_MAX; i++ )
    {
        for( u32 j = 0; j < _DELAY_FREE_TIME; j++ )
        {
            for( u32 k = 0; k < g_DelayFreeListNum[i][j]; k++ )
            {
                g_DynamicHeap->Free( g_DelayFreeList[i][j][k] );
                g_DelayFreeList[i][j][k] = NULL;
            }

            g_StaticHeap->Free( g_DelayFreeList[i][j] );
            g_DelayFreeList[i][j] = NULL;
            g_DelayFreeListNum[i][j] = 0;
        }
    }
}

//---------------------------------------------------------------------------
//  遅延解放の為のフリーリストへアドレスを追加する
//---------------------------------------------------------------------------
void AddFreeListForDynamicHeap( void* ptr )
{
#if EFT_IS_CAFE
    CpuCore core = (CpuCore)OSGetCoreId();
#else
    CpuCore core = EFT_CPU_CORE_1;
#endif

    u32 index = g_DelayFreeListNum[core][g_DelayFreeListCntr];
    g_DelayFreeList[core][g_DelayFreeListCntr][index] = ptr;
    g_DelayFreeListNum[core][g_DelayFreeListCntr]++;
    EFT_ASSERT( g_DelayFreeListNum[core][g_DelayFreeListCntr] <= g_DelayFreeListMax );
}

//---------------------------------------------------------------------------
//  遅延解放用のリスト内アドレスを解放します。
//---------------------------------------------------------------------------
void FlushDelayFreeList()
{
    g_DelayFreeListCntr++;
    if ( g_DelayFreeListCntr == g_DelayFreeTime ) g_DelayFreeListCntr = 0;

    for( u32 core = 0; core < EFT_CPU_CORE_MAX; core++ )
    {
        for( u32 j = 0; j < g_DelayFreeListNum[core][g_DelayFreeListCntr]; j++ )
        {
            g_DynamicHeap->Free( g_DelayFreeList[core][g_DelayFreeListCntr][j] );
            g_DelayFreeList[core][g_DelayFreeListCntr][j] = NULL;
        }
        g_DelayFreeListNum[core][g_DelayFreeListCntr] = 0;
    }
}

//---------------------------------------------------------------------------
//  ランタイムログ出力を抑制行います。
//---------------------------------------------------------------------------
void SetSuppressOutputLog( bool flag )
{
    g_SuppressLog = flag;
}

//---------------------------------------------------
//  ログ出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputLogCallBack( OutputMessageCallback cb )
{
    g_LogCallback = cb;
}

//---------------------------------------------------
//  ランタイム警告出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputWarningCallBack( OutputMessageCallback cb )
{
    g_WarningCallback = cb;
}

//---------------------------------------------------
//! @brief  ランタイムエラー出力時コールバックを設定します。
//---------------------------------------------------
void SetOutputErrorCallBack( OutputMessageCallback cb )
{
    g_ErrorCallback = cb;
}

//---------------------------------------------------------------------------
//  ランタイムログ出力を行います。
//---------------------------------------------------------------------------
void OutputLog( const char* format, ... )
{
#ifndef NW_RELEASE
    if ( g_SuppressLog ) return;

    if ( g_LogCallback )
    {
        va_list vargs;
        va_start(vargs, format);
        g_LogCallback( "[EFT] ", vargs );
        g_LogCallback( format, vargs );
        va_end(vargs);
        return;
    }

    EFT_PRINT( "[EFT] " );
    va_list vargs;
    va_start(vargs, format);
    VPrintf( format, vargs );
    va_end(vargs);
#else
    EFT_UNUSED_VARIABLE( format );
#endif
}

//---------------------------------------------------------------------------
//  ランタイムログ警告出力を行います。
//---------------------------------------------------------------------------
void OutputWarning( const char* format, ... )
{
#ifndef NW_RELEASE
    if ( g_WarningCallback )
    {
        va_list vargs;
        va_start(vargs, format);
        g_WarningCallback( "[EFT] Warning!! ", vargs );
        g_WarningCallback( format, vargs );
        va_end(vargs);
        return;
    }

    EFT_PRINT( "[EFT] Warning!! " );
    va_list vlist;
    va_start(vlist, format);
    VPrintf( format, vlist );
    va_end(vlist);

#else
    EFT_UNUSED_VARIABLE( format );
#endif
}

//---------------------------------------------------------------------------
//  ランタイムログ警告出力を行います。
//---------------------------------------------------------------------------
void Warning( void* context, RuntimeWarning warningId )
{
#ifndef NW_RELEASE

    g_Warning |= warningId;

    switch ( warningId )
    {
        case EFT_WARNING_PARTICLE_EMISSION_FAILED:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "Particle Emission Error.\n" );
                OutputWarning( "  %s - %s\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name );
            }
            break;

        case EFT_WARNING_PARTICLE_MAX_NUM_IS_ZERO:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "Particle Max Num is Zero. PLife:%d/Interval/EmissionRate",
                    emitter->emitterData->ptcl.life, emitter->emitterData->emission.interval, emitter->emitterData->emission.rate );
                OutputWarning( "  %s - %s - GroupId:%d\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name, emitter->groupID );
            }
            break;

        case EFT_WARNING_PARTICLE_IS_DIRTY:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "Particle UserData or PluginData is Dirty. \n" );
                OutputWarning( "  %s - %s - GroupId:%d\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name, emitter->groupID );
            }
            break;

        case EFT_WARNING_CUSTOMACTION_PARAM_IS_NOT_EXIST:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "CustomAction Parameter isn't Exist.\n" );
                OutputWarning( "  %s\n", emitter->GetEmitterSet()->GetName() );
            }
            break;

        case EFT_WARNING_SHADER_IS_NOT_EXIST:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "Shader isn't Exist.\n" );
                OutputWarning( "  %s - %s - GroupId:%d\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name, emitter->groupID );
            }
            break;

        case EFT_WARNING_PARENT_PARTICLE_IS_NOT_EXIST:
            {
                nw::eft2::Emitter* emitter = reinterpret_cast<nw::eft2::Emitter*>( context );
                OutputWarning( "Parent Particle isn't Exist.\n" );
                OutputWarning( "  %s - %s - GroupId:%d\n", emitter->GetEmitterSet()->GetName(), emitter->emitterData->name, emitter->groupID );
            }
            break;

        case EFT_WARNING_UNSAFE_PRIMITIVE_LOAD:
            OutputWarning( "Unsafe Primitive Load.\n" );
            break;

        case EFT_WARNING_NOT_SET_MANUAL_EMIT_MODE:
            {
                nw::eft2::EmitterSet* emitterSet = reinterpret_cast<nw::eft2::EmitterSet*>( context );
                OutputWarning( "Manual Emit Mode is not Set.\n" );
                OutputWarning( "  EmitterSet:%s\n", emitterSet->GetName() );
            }
            break;

        case EFT_WARNING_NONE:
            break;

        default:
            EFT_ASSERT(0);
            break;
    }
#else
    EFT_UNUSED_VARIABLE( context );
    EFT_UNUSED_VARIABLE( warningId );
#endif
}

//---------------------------------------------------------------------------
//  ランタイム警告のビットフィールドを取得します。
//---------------------------------------------------------------------------
u32 GetWarnings()
{
    return g_Warning;
}

//---------------------------------------------------------------------------
//  ランタイムエラー出力を行います。
//---------------------------------------------------------------------------
void OutputError( const char* format, ... )
{
#ifndef NW_RELEASE
    if ( g_ErrorCallback )
    {
        va_list vargs;
        va_start(vargs, format);
        g_ErrorCallback( "[EFT] Error!! ", vargs );
        g_ErrorCallback( format, vargs );
        va_end(vargs);
        EFT_ERR( "[EFT] Error!! Stopped." );
        return;
    }

    EFT_PRINT( "[EFT] Error!! " );
    va_list vargs;
    va_start(vargs, format);
    EFT_ERR( format, vargs );
    va_end(vargs);
#else
    EFT_UNUSED_VARIABLE( format );
#endif
}

//---------------------------------------------------------------------------
//  文字列出力を行います。
//---------------------------------------------------------------------------
void VPrintf(const char *fmt, va_list vlist)
{
#if ( EFT_IS_WIN || EFT_IS_CAFE)
     nw::ut::VPrintf( fmt, vlist );
#endif
}

} // namespace eft
} // namespace nw
