﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_Random.h>
#include <nw/eft/eft2_Emitter.h>

// 等分割球の定義データをインクルード
#include <nw/eft/eft2_SameDivideSphereData.h>

namespace nw   {
namespace eft2 {

//------------------------------------------------------------------------------
//      関数テーブル
//------------------------------------------------------------------------------
EmitterCalc::EmitFunction EmitterCalc::m_EmitFunctions[] =
{
    EmitterCalc::CalcEmitPoint,                                 //!< 点エミッタ
    EmitterCalc::CalcEmitCircle,                                //!< 円
    EmitterCalc::CalcEmitCircleEqualDivision,                   //!< 円(分割)
    EmitterCalc::CalcEmitCircleFill,                            //!< 円(フィル)
    EmitterCalc::CalcEmitSphere,                                //!< 球
    EmitterCalc::CalcEmitSphereEqualDivision32,                 //!< 球(分割)
    EmitterCalc::CalcEmitSphereEqualDivision64,                 //!< 球(分割64)
    EmitterCalc::CalcEmitSphereFill,                            //!< 球(フィル)
    EmitterCalc::CalcEmitCylinder,                              //!< 円柱
    EmitterCalc::CalcEmitCylinderFill,                          //!< 円柱(フィル)
    EmitterCalc::CalcEmitBox,                                   //!< 立方体
    EmitterCalc::CalcEmitBoxFill,                               //!< 立方体(フィル)
    EmitterCalc::CalcEmitLine,                                  //!< 線
    EmitterCalc::CalcEmitLineEqualDivision,                     //!< 線(分割)
    EmitterCalc::CalcEmitRectangle,                             //!< 矩形
    EmitterCalc::CalcEmitPrimitive,                             //!< プリミティブ
};

namespace detail {

//------------------------------------------------------------------------------
//  弧の開始角度、角度幅を求めます。（パラメータ式）
//------------------------------------------------------------------------------
inline void CalcVolumeSweepParams( f32* rotWidth, f32* rotStart, const f32 sweepParam, const f32 sweepStart, const u8 isSweepStartRandom, const f32 randomValue )
{
    //単位：Radian
    *rotWidth = sweepParam;
    *rotStart = ( isSweepStartRandom ) ? ( randomValue * nw::math::F_PI * 2.0f ) : ( sweepStart );
}

//------------------------------------------------------------------------------
//  弧の位置を計算
//------------------------------------------------------------------------------
inline f32 CalcRotateForSweep( const f32 random, const f32 rotWidth, const f32 rotStart )
{
    return random * rotWidth + rotStart - (rotWidth / 2);
}


//------------------------------------------------------------------------------
//  円周上の位置を求めます。
//------------------------------------------------------------------------------
inline void CalcSinCosRad( const ResEmitterVolume* __restrict res, Random* rnd, const f32 rotWidth, const f32 rotStart, f32* sinV, f32* cosV )
{
    // 単位：Radian

    // 円の場合
    if( res->arcType == 1 )
    {
        f32 rot = rnd->GetF32() * nw::math::F_PI * 2.0f;
        nw::math::SinCosRad( sinV, cosV, rot );
    }
    // 弧の場合
    else
    {
        f32 rot = CalcRotateForSweep( rnd->GetF32(), rotWidth, rotStart );
        nw::math::SinCosRad( sinV, cosV, rot );
    }
}

//------------------------------------------------------------------------------
//  Y軸上の位置を求めます。
//------------------------------------------------------------------------------
inline f32 CalcPosOnAxisY( const ResEmitterVolume* __restrict res, Random* rnd )
{
    if( res->arcType == 1 )
    {
        // 緯度の場合
        return 1.0f - ( 1.0f - nw::math::CosRad( res->sweepLatitude ) ) * rnd->GetF32();
    }
    else
    {
        // 経度の場合
        return rnd->GetF32() * 2.0f - 1.0f;
    }
}

//------------------------------------------------------------------------------
//  球の緯度の範囲内かどうか判定します。
//------------------------------------------------------------------------------
inline bool IsLatitudeInside( const ResEmitterVolume* __restrict res, const nw::math::VEC3& dir )
{
    // ほぼ 180 度の場合は true を返す
    if( nw::math::F_PI - 0.0001f < res->sweepLatitude )
    {
        return true;
    }

    f32 y = nw::math::CosRad( res->sweepLatitude );
    return ( y < dir.y );
}

//------------------------------------------------------------------------------
//  球を緯度の軸に合うように回転します。
//------------------------------------------------------------------------------
inline void RotateDirection( const ResEmitterVolume* __restrict res, nw::math::VEC3* dir )
{
    enum
    {
        LATITUDE_DIR_PLUS_X  = 0,
        LATITUDE_DIR_MINUS_X = 1,
        LATITUDE_DIR_PLUS_Y  = 2,
        LATITUDE_DIR_MINUS_Y = 3,
        LATITUDE_DIR_PLUS_Z  = 4,
        LATITUDE_DIR_MINUS_Z = 5,
    };

    nw::math::VEC3 latitudeDir( 0, 0, 0 );
    switch( res->volumeLatitudeDir )
    {
    case LATITUDE_DIR_PLUS_X:  { latitudeDir.x =  1.0f; break; }
    case LATITUDE_DIR_MINUS_X: { latitudeDir.x = -1.0f; break; }
    case LATITUDE_DIR_PLUS_Y:  { latitudeDir.y =  1.0f; break; }
    case LATITUDE_DIR_MINUS_Y: { latitudeDir.y = -1.0f; break; }
    case LATITUDE_DIR_PLUS_Z:  { latitudeDir.z =  1.0f; break; }
    case LATITUDE_DIR_MINUS_Z: { latitudeDir.z = -1.0f; break; }
    default: NW_ASSERT(0);
    }

    if( latitudeDir.x != 0.0f || latitudeDir.y != 1.0f || latitudeDir.z != 0.0f )
    {
        // (0, 1, 0)を指定方向に回転させる回転行列を生成し適用する
        nw::math::QUAT q;
        nw::math::QUATMakeVectorRotation( &q, nw::math::VEC3( 0.0f, 1.0f, 0.0f ), latitudeDir );
        nw::math::MTX34 qmat;
        nw::math::QUATToMTX34( &qmat, q );
        nw::math::VEC3 src;
        src.Set( dir->x, dir->y, dir->z );
        nw::math::VEC3Transform( dir, &qmat, &src );
    }
}

}

//------------------------------------------------------------------------------
//  点エミッタ
//------------------------------------------------------------------------------
bool EmitterCalc::CalcEmitPoint( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );
    EFT_UNUSED_VARIABLE( emitterAnim );

    // ローカル位置
    localPos->x = 0.0f;
    localPos->y = 0.0f;
    localPos->z = 0.0f;

    // ローカル速度
    nw::math::VEC3 rvec3 = emitter->random.GetNormalizedVec3();
    f32 allv = emitterAnim->allVelDir;
    localVec->x = rvec3.x * allv;
    localVec->y = rvec3.y * allv;
    localVec->z = rvec3.z * allv;

    return true;
}

//---------------------------------------------------------------------------
//  円エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitCircle( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    // 弧の開始角度、角度幅
    f32 rotWidth;
    f32 rotStart;
    const ResEmitterVolume* res = &emitter->emitterData->volume;
    detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLongitude, res->sweepStart, res->sweepStartRandom, randomValue );

    f32 sinV, cosV;
    f32 emitterRotate = detail::CalcRotateForSweep( emitter->random.GetF32(), rotWidth, rotStart );
    nw::math::SinCosRad( &sinV, &cosV, emitterRotate );

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // ローカル位置
    localPos->x = sinV * scaleX;
    localPos->z = cosV * scaleZ;
    localPos->y = 0.0f;

    // ローカル速度
    localVec->x = sinV * emitterAnim->allVelDir;
    localVec->z = cosV * emitterAnim->allVelDir;
    localVec->y = 0.0f;

    return true;
}

//------------------------------------------------------------------------------
//  等分割円エミッタ
//------------------------------------------------------------------------------
bool EmitterCalc::CalcEmitCircleEqualDivision( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    // 単位：Radian
    EFT_UNUSED_VARIABLE( emitMax );

    const ResEmitterVolume* res = &emitter->emitterData->volume;

    // 弧の初期角度を決定
    f32 sweepStart = emitter->emitterData->volume.sweepStart;
    if( emitter->emitterData->volume.sweepStartRandom )
    {
        sweepStart = randomValue * nw::math::F_PI * 2.0f;
    }

    f32 rot     = sweepStart;

    // 弧の幅＆分割数から、インデックスごとの角度増分を決定
    u32 numDivision = res->numDivideCircle;
    if ( emitter->emitterRes->emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
    {
        numDivision -= static_cast<u32>( randomValue * res->numDivideCircleRandom * 0.01f * res->numDivideCircle );
    }
    if ( emitter->emitterData->volume.sweepLongitude != nw::math::F_PI * 2.0f )
    {
        // 円でない場合は先端と末端が重ならないので「N点使って等分割（≒従ってN-1分割）」になる
        if( numDivision > 1 )
        {
            numDivision--;
        }
    }
    u32 emitFixedIndex = emitIndex;

    switch ( emitter->emitterRes->emitterData->volume.primEmitType )
    {
    case EFT_EMIT_FROM_PRIMITIVE_UNISON:// 一斉放出：何もしない
        break;
    case EFT_EMIT_FROM_PRIMITVE_RANDOM:// ランタイム放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->random.GetF32() * numDivision );
        break;
    case EFT_EMIT_FROM_PRIMITIVE_INDEX_ORDER:// インデックス放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->primEmitCounter ) % numDivision;
        emitter->primEmitCounter++;
        if( emitter->primEmitCounter >= numDivision )
        {
            emitter->primEmitCounter = 0;
        }
        break;
    default:
        EFT_ASSERT( 0 );
    }

    const f32 longitude = emitter->emitterData->volume.sweepLongitude;    // 弧の幅
    const f32 rotPlus = longitude / numDivision;

    // 角度
    rot += emitFixedIndex * rotPlus - (longitude / 2.0f) + ( ( emitter->random.GetF32() - 0.5f ) * 2 ) * emitter->emitterData->volume.volumeSurfacePosRand;
    f32 sinV, cosV;
    nw::math::SinCosRad( &sinV, &cosV, rot );

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // ローカル位置
    localPos->x = sinV * scaleX;
    localPos->z = cosV * scaleZ;
    localPos->y = 0.0f;

    // ローカル速度
    localVec->x = sinV * emitterAnim->allVelDir;
    localVec->z = cosV * emitterAnim->allVelDir;
    localVec->y = 0.0f;

    return true;
}

//---------------------------------------------------------------------------
//  円フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitCircleFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    // 弧の開始角度、角度幅
    f32 rotWidth;
    f32 rotStart;
    const ResEmitterVolume* res = &emitter->emitterData->volume;
    detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLongitude, res->sweepStart, res->sweepStartRandom, randomValue );

    f32 sinV, cosV;
    f32 emitterRotate = detail::CalcRotateForSweep( emitter->random.GetF32(), rotWidth, rotStart );
    nw::math::SinCosRad( &sinV, &cosV, emitterRotate );

    // 方向
    f32 rand = emitter->random.GetF32();
    f32 inner = 1.0f - emitter->emitterData->volume.caliberRatio;

    // MEMO: 半径上の位置をここで乱数で補間して動かすが、
    //       線形補間すると、円の中心の密度が外周側と比べて大きくなるので、二乗したモノの線形補間のrootを取ることで、
    //       円内に均一に分布するようにする。（※外周側の方がより多く選ばれる）
    rand = SafeSqrt( rand + inner * inner * ( 1.0f - rand ) );
    f32 dirx = rand * sinV;
    f32 dirz = rand * cosV;

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // ローカル位置
    localPos->x = dirx * scaleX;
    localPos->z = dirz * scaleZ;
    localPos->y = 0.0f;

    // ローカル速度
    // MEMO: エミッタ形状に沿って歪ませたあと、一度正規化し全方向初速を乗算する。
    nw::math::VEC3 vel = nw::math::VEC3( dirx * rand, 0, dirz * rand );
    if( scaleX > scaleZ )
    {
        const f32 ratio = scaleZ / scaleX;
        vel.z *= ratio;
    }
    else
    {
        const f32 ratio = scaleX / scaleZ;
        vel.x *= ratio;
    }
    if( vel.LengthSquare() > 0.0f )
    {
        vel.Normalize();
    }
    else
    {
        // 零ベクトルの場合（≒乱数で 0 を引いた場合）、デフォルトの方向に決め打ちする。
        vel = nw::math::VEC3( 0, 0, 1 );
    }
    vel *= emitterAnim->allVelDir;
    EFT_F32_VEC3_COPY( localVec, &vel );

    return true;
}

//---------------------------------------------------------------------------
//  球エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitSphere( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    // 参照を取得
    const ResEmitterVolume* res = &emitter->emitterData->volume;
    Random* rnd = &emitter->random;

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;


    // 弧の開始角度、角度幅
    f32 rotWidth;
    f32 rotStart;
    if( res->arcType == 1 )
    {
        // 緯度
        detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLatitude, res->sweepStart, res->sweepStartRandom, randomValue );
    }
    else
    {
        // 経度
        detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLongitude, res->sweepStart, res->sweepStartRandom, randomValue );
    }

    // 円周上の位置
    f32 sinV;
    f32 cosV;
    detail::CalcSinCosRad(res, rnd, rotWidth, rotStart, &sinV, &cosV);

    // 高さ
    f32 y = detail::CalcPosOnAxisY(res, rnd);

    // 半径
    f32 r = SafeSqrt( 1.0f - y * y );

    // 方向
    nw::math::VEC3 dir;
    dir.x = r * sinV;
    dir.z = r * cosV;
    dir.y = y;

    // 緯度の方向回転
    if( res->arcType == 1 )
    {
        detail::RotateDirection( res, &dir );
    }

    // 位置
    localPos->x = dir.x * scaleX;
    localPos->y = dir.y * scaleY;
    localPos->z = dir.z * scaleZ;

    // 速度
    f32 figureVel = emitterAnim->allVelDir;
    localVec->x = dir.x * figureVel;
    localVec->y = dir.y * figureVel;
    localVec->z = dir.z * figureVel;

    return true;
}

//---------------------------------------------------------------------------
//  球 32等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitSphereEqualDivision32( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    // 参照を取得
    const ResEmitterVolume* res = &emitter->emitterData->volume;

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // UI上のIndex値から放出レートにマッピング。
    const u32 pattern[] =
    {
        2, 3, 4, 6, 8, 12, 20, 32
    };

    const u32 vIndex = res->volumeTblIndex;
    const f32 *sphere_tbl = gSameDivideSphereTbl[ vIndex ];

    u32 emitFixedIndex = emitIndex;
    const u32 numDivision = pattern[ vIndex ];
    switch ( emitter->emitterRes->emitterData->volume.primEmitType )
    {
    case EFT_EMIT_FROM_PRIMITIVE_UNISON:// 一斉放出
        break;
    case EFT_EMIT_FROM_PRIMITVE_RANDOM:// ランタイム放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->random.GetF32() * numDivision );
        break;
    case EFT_EMIT_FROM_PRIMITIVE_INDEX_ORDER:// インデックス放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->primEmitCounter ) % numDivision;
        emitter->primEmitCounter++;
        if( emitter->primEmitCounter >= numDivision )
        {
            emitter->primEmitCounter = 0;
        }
        break;
    default:
        EFT_ASSERT( 0 );
    }

    // 方向
    nw::math::VEC3 dir;
    const u32 index = emitFixedIndex * 3;
    dir.x = sphere_tbl[ index ];
    dir.y = sphere_tbl[ index + 1 ];
    dir.z = sphere_tbl[ index + 2 ];

    // 経度は使わない
    {
        // 緯度でフィルタ
        if( !detail::IsLatitudeInside( res, dir ) )
        {
            return false;
        }

        // 緯度の方向回転
        detail::RotateDirection( res, &dir );
    }

    // 位置
    localPos->x = dir.x * scaleX;
    localPos->y = dir.y * scaleY;
    localPos->z = dir.z * scaleZ;

    // 速度
    f32 figureVel = emitterAnim->allVelDir;
    localVec->x = dir.x * figureVel;
    localVec->y = dir.y * figureVel;
    localVec->z = dir.z * figureVel;

    return true;
}

//---------------------------------------------------------------------------
//  球 64等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitSphereEqualDivision64( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    // 参照を取得
    const ResEmitterVolume* res = &emitter->emitterData->volume;

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    const u32 vIndex = res->volumeTblIndex64 - 4;   // 分割数からインデックス値に修正
    const u32 tblIndex = vIndex + 2;
    const f32 *sphere_tbl = gSameDivideSphere64Tbl[ tblIndex ];

    u32 emitFixedIndex = emitIndex;
    const u32 numDivision = res->volumeTblIndex64;
    switch ( emitter->emitterRes->emitterData->volume.primEmitType )
    {
    case EFT_EMIT_FROM_PRIMITIVE_UNISON:// 一斉放出
        break;
    case EFT_EMIT_FROM_PRIMITVE_RANDOM:// ランタイム放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->random.GetF32() * numDivision );
        break;
    case EFT_EMIT_FROM_PRIMITIVE_INDEX_ORDER:// インデックス放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->primEmitCounter ) % numDivision;
        emitter->primEmitCounter++;
        if( emitter->primEmitCounter >= numDivision )
        {
            emitter->primEmitCounter = 0;
        }
        break;
    default:
        EFT_ASSERT( 0 );
    }

    // 方向
    nw::math::VEC3 dir;
    const u32 index = emitFixedIndex * 3;
    dir.x = sphere_tbl[index + 0];
    dir.y = sphere_tbl[index + 1];
    dir.z = sphere_tbl[index + 2];

    // 経度は使わない
    {
        // 緯度でフィルタ
        if( !detail::IsLatitudeInside( res, dir ) )
        {
            localPos->w = emitter->time + emitter->emitterData->ptcl.life;
            return false;
        }

        // 緯度の方向回転
        detail::RotateDirection( res, &dir );
    }

    // 位置
    localPos->x = dir.x * scaleX;
    localPos->y = dir.y * scaleY;
    localPos->z = dir.z * scaleZ;

    // 速度
    f32 figureVel = emitterAnim->allVelDir;
    localVec->x = dir.x * figureVel;
    localVec->y = dir.y * figureVel;
    localVec->z = dir.z * figureVel;

    return true;
}

//---------------------------------------------------------------------------
//  球 フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitSphereFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );

    // 参照を取得
    const ResEmitterVolume* res = &emitter->emitterData->volume;
    Random* rnd = &emitter->random;

    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // 弧の開始角度、角度幅
    f32 rotWidth;
    f32 rotStart;
    if( res->arcType == 1 )
    {
        detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLatitude, res->sweepStart, res->sweepStartRandom, randomValue );
    }
    else
    {
        detail::CalcVolumeSweepParams( &rotWidth, &rotStart,res->sweepLongitude, res->sweepStart, res->sweepStartRandom, randomValue );
    }

    // 円周上の位置
    f32 sinV;
    f32 cosV;
    detail::CalcSinCosRad(res, rnd, rotWidth, rotStart, &sinV, &cosV);

    // 高さ
    f32 y = detail::CalcPosOnAxisY(res, rnd);

    // 半径
    f32 r = SafeSqrt( 1.0f - y * y );

    // 空洞状の分布
    f32 radius = rnd->GetF32();
    radius = SafeSqrt( radius );
    radius = radius * res->caliberRatio + 1.0f - res->caliberRatio;

    // 方向
    nw::math::VEC3 dir;
    dir.x = r * sinV;
    dir.z = r * cosV;
    dir.y = y;

    // 緯度の方向回転
    if( res->arcType == 1 )
    {
        detail::RotateDirection( res, &dir );
    }

    // 位置
    localPos->x = dir.x * scaleX * radius;
    localPos->y = dir.y * scaleY * radius;
    localPos->z = dir.z * scaleZ * radius;

    // 速度
    f32 figureVel = emitterAnim->allVelDir;
    localVec->x = dir.x * figureVel;
    localVec->y = dir.y * figureVel;
    localVec->z = dir.z * figureVel;

    return true;
}

//---------------------------------------------------------------------------
//  シリンダー エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitCylinder( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    bool result = CalcEmitCircle( localPos, localVec, emitter, emitIndex, emitMax, randomValue, emitterAnim );
    if( !result ){ return false; }

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;

    f32 y = emitter->random.GetF32() * 2.0f - 1.0f;
    localPos->y = y * scaleY;

    return true;
}

//---------------------------------------------------------------------------
//  シリンダーフィル エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitCylinderFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    bool result = CalcEmitCircleFill( localPos, localVec, emitter, emitIndex, emitMax, randomValue, emitterAnim );
    if( !result ){ return false; }

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;

    f32 y = emitter->random.GetF32() * 2.0f - 1.0f;
    localPos->y = y * scaleY;

    return true;
}

//---------------------------------------------------------------------------
//  ボックス エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitBox( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    u32 coord   = emitter->random.GetU32Direct();
    u32 sign    = emitter->random.GetU32Direct();

    nw::math::VEC3 radRnd;
    radRnd.x = emitter->random.GetF32() * 2.0f - 1.0f;
    radRnd.y = emitter->random.GetF32() * 2.0f - 1.0f;
    radRnd.z = emitter->random.GetF32() * 2.0f - 1.0f;

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    // xy
    if( coord < 0xffffffff / 3 * 1 )
    {
        if( sign < 0xffffffff / 2 )
        {
            localPos->x = scaleX * radRnd.x;
            localPos->y = scaleY * radRnd.y;
            localPos->z = scaleZ;

        }
        else
        {
            localPos->x =  scaleX * radRnd.x;
            localPos->y =  scaleY * radRnd.y;
            localPos->z = -scaleZ;
        }
    }

    // xz
    else if( coord < 0xffffffff / 3 * 2 )
    {
        if( sign < 0xffffffff / 2 )
        {
            localPos->x = scaleX * radRnd.x;
            localPos->y = scaleY;
            localPos->z = scaleZ * radRnd.z;

        }
        else
        {
            localPos->x =   scaleX * radRnd.x;
            localPos->y =  -scaleY;
            localPos->z =   scaleZ * radRnd.z;
        }
    }

    // yz
    else
    {
        if( sign < 0xffffffff / 2 )
        {
            localPos->x = scaleX;
            localPos->y = scaleY * radRnd.y;
            localPos->z = scaleZ * radRnd.z;

        }
        else
        {
            localPos->x =  -scaleX;
            localPos->y =   scaleY * radRnd.y;
            localPos->z =   scaleZ * radRnd.z;
        }
    }

    // ローカル速度
    nw::math::VEC3 dir;
    if ( !localPos->IsZero() )
    {
        dir.x = localPos->x;
        dir.y = localPos->y;
        dir.z = localPos->z;
        if( !dir.IsZero() )
        {
            dir.Normalize();
        }
    }
    localVec->x = dir.x * emitterAnim->allVelDir;
    localVec->y = dir.y * emitterAnim->allVelDir;
    localVec->z = dir.z * emitterAnim->allVelDir;

    return true;
}

//---------------------------------------------------------------------------
//  ボックス フィルエミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitBoxFill( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleY = res->volumeRadius.y * emitterAnim->vvolumeScale.y;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;


    //if( emitter->emitterData->volume.caliberRatio == 0.0f )
    //{
    //    localPos->x = scaleX * ( emitter->random.GetF32() * 2.0f - 1.0f );
    //    localPos->y = scaleY * ( emitter->random.GetF32() * 2.0f - 1.0f );
    //    localPos->z = scaleZ * ( emitter->random.GetF32() * 2.0f - 1.0f );
    //}
    //else
    {
        // 空洞状の分布
        f32 r = 1.0f - emitter->emitterData->volume.caliberRatio;
        if( r == 1.0f ){ r = 0.999f; }

        f32 v = emitter->random.GetF32();
        f32 r2 = r * r;
        f32 r3 = r2 * r;

        f32 volumeA = 1.0f - r;
        f32 volumeB = volumeA * r;

        v = v * ( 1.0f - r3 );

        if( v < volumeA )
        {
            localPos->x = emitter->random.GetF32();
            localPos->y = emitter->random.GetF32() * ( 1.0f - r ) + r;
            localPos->z = emitter->random.GetF32();

        }
        else if ( v < volumeA + volumeB )
        {
            localPos->x = emitter->random.GetF32() * ( 1.0f - r ) + r;
            localPos->y = emitter->random.GetF32() * r;
            localPos->z = emitter->random.GetF32();
        }
        else
        {
            localPos->x = emitter->random.GetF32() * r;
            localPos->y = emitter->random.GetF32() * r;
            localPos->z = emitter->random.GetF32() * ( 1.0f - r ) + r;
        }

        if( emitter->random.GetF32() < 0.5f )
        {
            localPos->x *= -1.0f;
        }

        if( emitter->random.GetF32() < 0.5f )
        {
            localPos->y *= -1.0f;
        }

        if( emitter->random.GetF32() < 0.5f )
        {
            localPos->z *= -1.0f;
        }

        localPos->x *= scaleX;
        localPos->y *= scaleY;
        localPos->z *= scaleZ;
    }

    // ローカル速度
    nw::math::VEC3 dir;
    if ( !localPos->IsZero() )
    {
        dir.x = localPos->x;
        dir.y = localPos->y;
        dir.z = localPos->z;

        if( dir.IsZero() )
        {
            // この形状ではパーティクルが(0,0,0)に配置される可能性がある
            // 向きをランダムに与える
            dir.x = emitter->random.GetF32();
            dir.y = emitter->random.GetF32();
            dir.z = emitter->random.GetF32();
        }
        dir.Normalize();
    }

    // ローカル速度
    localVec->x = dir.x * emitterAnim->allVelDir;
    localVec->y = dir.y * emitterAnim->allVelDir;
    localVec->z = dir.z * emitterAnim->allVelDir;

    return true;
}

//---------------------------------------------------------------------------
//  ライン エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitLine( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleZ = res->lineLength * emitterAnim->vvolumeScale.z;

    // ローカル位置
    localPos->x = 0.0f;
    localPos->y = 0.0f;
    localPos->z = emitter->random.GetF32() * scaleZ
                        - ( scaleZ + emitter->emitterData->volume.lineCenter * scaleZ ) / 2.0f;

    // ローカル速度
    localVec->x = 0.0f;
    localVec->y = 0.0f;
    localVec->z = emitterAnim->allVelDir;

    return true;
}

//---------------------------------------------------------------------------
//  ライン 等分割エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitLineEqualDivision( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitMax );

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleZ = res->lineLength * emitterAnim->vvolumeScale.z;

    f32 t      = 0.0f;

    u32 emitFixedIndex = emitIndex;
    u32 numDivision = res->numDivideLine;

    switch ( emitter->emitterRes->emitterData->volume.primEmitType )
    {
    case EFT_EMIT_FROM_PRIMITIVE_UNISON:// 一斉放出
        numDivision -= static_cast<u32>( randomValue * res->numDivideLineRandom * 0.01f * res->numDivideLine );
        break;
    case EFT_EMIT_FROM_PRIMITVE_RANDOM:// ランタイム放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->random.GetF32() * numDivision );
        break;
    case EFT_EMIT_FROM_PRIMITIVE_INDEX_ORDER:// インデックス放出タイプ
        emitFixedIndex = static_cast<u32>( emitter->primEmitCounter ) % numDivision;
        emitter->primEmitCounter++;
        if( emitter->primEmitCounter >= numDivision )
        {
            emitter->primEmitCounter = 0;
        }
        break;
    default:
        EFT_ASSERT( 0 );
    }

    f32 t_plus = 1.0f / numDivision;
    if( numDivision == 1 )
    {
        t      = 0.5f;
        t_plus = 0.0f;
    }
    else
    {
        t      = 0.0f;
        t_plus = 1.0f / ( numDivision - 1 );
    }

    t = t_plus * emitFixedIndex + t;

    // ローカル位置
    localPos->x = 0.0f;
    localPos->y = 0.0f;
    localPos->z = t * scaleZ
                        - ( scaleZ + emitter->emitterData->volume.lineCenter * scaleZ ) / 2.0f;

    // ローカル速度
    localVec->x = 0.0f;
    localVec->y = 0.0f;
    localVec->z = emitterAnim->allVelDir;

    return true;
}

//---------------------------------------------------------------------------
//  矩形 エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitRectangle( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    EFT_UNUSED_VARIABLE( emitIndex );
    EFT_UNUSED_VARIABLE( emitMax );
    EFT_UNUSED_VARIABLE( randomValue );

    const ResEmitterVolume* res = &emitter->emitterData->volume;
    f32 scaleX = res->volumeRadius.x * emitterAnim->vvolumeScale.x;
    f32 scaleZ = res->volumeRadius.z * emitterAnim->vvolumeScale.z;

    u32 coord   = emitter->random.GetU32Direct();
    u32 sign    = emitter->random.GetU32Direct();

    nw::math::VEC3 radRnd;
    radRnd.x = emitter->random.GetF32() * 2.0f - 1.0f;
    radRnd.z = emitter->random.GetF32() * 2.0f - 1.0f;

    // xy
    if( coord < 0xffffffff / 2 )
    {
        if( sign < 0xffffffff / 2 )
        {
            localPos->x = scaleX * radRnd.x;
            localPos->y = 0;
            localPos->z = scaleZ;

        }
        else
        {
            localPos->x =  scaleX * radRnd.x;
            localPos->y =  0;
            localPos->z = -scaleZ;
        }
    }

    // yz
    else
    {
        if( sign < 0xffffffff / 2 )
        {
            localPos->x = scaleX;
            localPos->y = 0;
            localPos->z = scaleZ * radRnd.z;

        }
        else
        {
            localPos->x =  -scaleX;
            localPos->y =   0;
            localPos->z =   scaleZ * radRnd.z;
        }
    }

    // ローカル速度
    nw::math::VEC3 dir;
    if ( !localPos->IsZero() )
    {
        dir.x = localPos->x;
        dir.y = localPos->y;
        dir.z = localPos->z;
        if( !dir.IsZero() )
        {
            dir.Normalize();
        }
    }

    localVec->x = dir.x * emitterAnim->allVelDir;
    localVec->y = dir.y * emitterAnim->allVelDir;
    localVec->z = dir.z * emitterAnim->allVelDir;

    return true;
}

//---------------------------------------------------------------------------
//  プリミティブ エミッタ計算処理を行います。
//---------------------------------------------------------------------------
bool EmitterCalc::CalcEmitPrimitive( nw::math::VEC4* localPos, nw::math::VEC4* localVec, Emitter* emitter, u32 emitIndex, u32 emitMax, f32 randomValue, EmitterAnimValue* emitterAnim )
{
    if ( !emitter->emitterRes->volumePrimitive )
    {
        return CalcEmitPoint( localPos, localVec, emitter, emitIndex, emitMax, randomValue, emitterAnim );
    }

    Primitive* primitive = emitter->emitterRes->volumePrimitive;

    nw::math::VEC4* pos = reinterpret_cast<nw::math::VEC4*>( primitive->GetPositionVertexBuffer().GetVertexBuffer() );
    nw::math::VEC4* nor = reinterpret_cast<nw::math::VEC4*>( primitive->GetNoramlVertexBuffer().GetVertexBuffer() );

    u32 emitFixedIndex = emitIndex;

    switch ( emitter->emitterRes->emitterData->volume.primEmitType )
    {
        // ランタイム放出タイプ
        case 1:
            emitFixedIndex = static_cast<u32>( emitter->random.GetF32() * primitive->GetVertexNum() );
            break;

        // インデックス放出タイプ
        case 2:
            emitFixedIndex = static_cast<u32>( emitter->primEmitCounter ) % primitive->GetVertexNum();
            emitter->primEmitCounter++;
            break;

    }

    localPos->x = pos[emitFixedIndex].x * emitterAnim->vvolumeScale.x;
    localPos->y = pos[emitFixedIndex].y * emitterAnim->vvolumeScale.y;
    localPos->z = pos[emitFixedIndex].z * emitterAnim->vvolumeScale.z;

    if ( nor )
    {
        localVec->x = nor[emitFixedIndex].x * emitterAnim->allVelDir;
        localVec->y = nor[emitFixedIndex].y * emitterAnim->allVelDir;
        localVec->z = nor[emitFixedIndex].z * emitterAnim->allVelDir;
    }
    else
    {
        localVec->x = 0.0f;
        localVec->y = 0.0f;
        localVec->z = 0.0f;
    }

    return true;
}

} // namespace eft2
} // namespace nw
