﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_EmitterSet.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Callback.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_StripeConnection.h>
#include <nw/eft/eft2_SuperStripe.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  初期化処理を行います。
//---------------------------------------------------------------------------
bool EmitterSet::Initialize( s32 emitterSetID, u32 createID, u32 resourceID, u8 groupID, u32 particleMax, Heap* heap )
{
    // リソースを取得
    Resource* resource = m_System->GetResource( resourceID );
    if( !resource )
    {
        OutputWarning( "Resource %d is not Exist.\n", resourceID );
        return false;
    }

    // リソースセットを取得
    m_EmitterSetRes = resource->GetEmitterSetResource( emitterSetID );
    if( !m_EmitterSetRes )
    {
        OutputWarning( "EmitterSetResource %d is not Exist.\n", emitterSetID );
        return false;
    }

    m_EmitterSetID                   = emitterSetID;
    m_EmitterSetCreateID             = createID;
    m_ResourceID                     = resourceID;
    m_GroupID                        = groupID;
    m_DrawPriority                   = EFT_DEFAULT_DRAW_PRIORITY;
    m_Next                           = NULL;
    m_Prev                           = NULL;
    m_EmitterHead                    = NULL;
    m_EmitterTail                    = NULL;
    m_IsLoopEffect                   = false;
    m_IsFade                         = false;
    m_IsDraw                         = true;
    m_IsCalc                         = true;
    m_IsSetDirectional               = false;
    m_IsDelayCreate                  = false;
    m_Heap                           = heap;
    m_DrawPath                       = 0;
    m_StartFrame                     = 0;
    m_RuntimeUserData                = 0;
    m_EmitterFirstNum                = 0;
    m_EmitterCreateID                = 0;
    m_EmitterVolumeScale.Set( 1.0f, 1.0f, 1.0f );
    m_AutoCalcScale.Set( 1.0f, 1.0f, 1.0f );
    m_ParticleScaleForCalc.Set( 1.0f, 1.0f, 1.0f );
    m_InitialRoate.Set( 0.0f, 0.0f, 0.0f );
    m_DirectionalVel                 = 1.0f;
    m_Directional.Set( 0.0f, 0.0f, 0.0f );
    m_DrawViewFlag                   = EFT_DRAW_VIEW_FLAG_ALL;
    m_ProcessingEmitterNum           = 0;
    m_ProcessingCpuEmitterNum        = 0;
    m_ProcessingGpuEmitterNum        = 0;
    m_ProcessingGpuSoEmitterNum      = 0;
    m_ProcessingStripeNum            = 0;
    m_ProcessingSuperStripeNum       = 0;
    m_CalcSkipEmitterNum             = 0;
    m_EmissionRatioScale             = 1.0f;
    m_EmissionIntervalScale          = 1.0f;
    m_ParticleLifeScale              = 1.0f;
    m_ReqFrameBufferTexturePath      = 0;
    m_ReqDepthBufferTexturePath      = 0;
    m_IsManualEmission               = false;
    m_FrameRate                      = 1.0f;
    m_InitializeCallback             = GetSystem()->GetEmitterSetInitializeCallback();
    m_FinalizeCallback               = GetSystem()->GetEmitterSetFinalizeCallback();
    m_IsEmitterSRTDirty              = 0;
    m_ManualEmissionAssignNum        = particleMax;
    m_IsBufferFliped                 = false;
    m_ResidentEmitterTime            = -1;
    m_IsAnyEmitterFade               = false;

    // 共通のリセット処理
    ResetCommon();

    m_MatrixSRT.SetIdentity();
    m_MatrixRT.SetIdentity();

    m_Color.Set( 1.0f, 1.0f, 1.0f, 1.0f );
    m_ParticleScale.Set( 1.0f, 1.0f, 1.0f );
    m_ParticleEmissionScale.Set( 1.0f, 1.0f, 1.0f );

    m_FigureVel          = 1.0f;
    m_DirectionalVel     = 1.0f;
    m_RandomVel          = 1.0f;
    m_VelAdd.Set( 0.0f, 0.0f, 0.0f );

    m_IsEmitterSRTDirty = 1;

    // エミッタセット初期化コールバックの呼び出し
    if ( m_InitializeCallback )
    {
        EmitterSetInitializeArg arg;
        arg.pSystem = GetSystem();
        arg.pEmitterSet = this;

        m_InitializeCallback( arg );
    }

    return true;
}

//---------------------------------------------------------------------------
//  EmitterSet::Reset() と EmitterSet::Initialize() の共通のリセット処理です
//---------------------------------------------------------------------------
void EmitterSet::ResetCommon()
{
    // 親エミッタ数
    m_EmitterNum = m_EmitterSetRes->emitterNum;

    // エミッタ間共通の乱数の種を生成
    // セット間共通の乱数
    RandomGenerator* rgenerator = Random::GetGlobalRandom();
    m_RandomSeed = rgenerator->GetU32();

    EmitterResource* emRes = m_EmitterSetRes->emitterResSet;
    while( emRes )
    {
        // エミッタ生成処理
        Emitter* emitter = CreateEmitter( emRes, m_ManualEmissionAssignNum );
        if ( !emitter )
        {
            emRes = emRes->next;
            continue;
        }
        m_EmitterFirstNum++;
        m_ProcessingEmitterNum++;

        // リソースが保持するシェーダを受け取る
        emitter->shader[EFT_SHADER_TYPE_NORMAL] = m_EmitterSetRes->shaderManager->GetShader( emRes->shaderIndex );
        EFT_NULL_ASSERT( emitter->shader );
        if ( emRes->drawPathShader1 != EFT_INVALID_SHADER_ID )
        {
            emitter->shader[EFT_SHADER_TYPE_PATH_DEF1] = m_EmitterSetRes->shaderManager->GetShader( emRes->drawPathShader1 );
        }
        if ( emRes->drawPathShader2 != EFT_INVALID_SHADER_ID )
        {
            emitter->shader[EFT_SHADER_TYPE_PATH_DEF2] = m_EmitterSetRes->shaderManager->GetShader( emRes->drawPathShader2 );
        }

        // ループエフェクトを含むかどうか。
        if ( !emRes->emitterData->emission.isOneTime )
        {
            m_IsLoopEffect = true;
        }

        emRes = emRes->next;
    }

    m_IsUsage = true;
}

//---------------------------------------------------------------------------
//  リセット処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Reset()
{
    // 再生中のエフェクトを終了する
    FinalizeEmitterList( m_EmitterHead );

    m_EmitterHead            = NULL;
    m_EmitterTail            = NULL;

    // Dirty立て直し
    m_IsEmitterSRTDirty = 1;

    // リソースを取得しなおす
    Resource* resource = m_System->GetResource( m_ResourceID );
    EFT_NULL_ASSERT( resource );
    m_EmitterSetRes = resource->GetEmitterSetResource( m_EmitterSetID );
    EFT_NULL_ASSERT( m_EmitterSetRes );

    // 共通のリセット処理
    ResetCommon();
}

//---------------------------------------------------------------------------
//  終了処理を行います。
//---------------------------------------------------------------------------
bool EmitterSet::Finalize()
{
    bool ret = FinalizeEmitterList( m_EmitterHead );

    // エミッタセット破棄コールバックの呼び出し
    {
        if( m_FinalizeCallback )
        {
            EmitterSetFinalizeArg arg;
            arg.system = GetSystem();
            arg.emitterSet = this;

            m_FinalizeCallback( arg );
            m_FinalizeCallback = NULL;
        }
    }

    m_IsUsage            = false;
    m_EmitterSetCreateID = 0xFFFFFFFF;

    return ret;
}

bool EmitterSet::FinalizeEmitterList( Emitter* pEmitterHead )
{
    bool ret = true;

    Emitter* emitter = pEmitterHead;
    while( emitter )
    {
        Emitter* next = emitter->next;

        if( emitter->emitterSetCreateID == m_EmitterSetCreateID &&
            emitter->emitterCalc &&
            emitter->IsAlive() )
        {
            // 子エミッタの終了処理
            for ( u32 i = 0; i < emitter->emitterRes->childEmitterResCount; i++ )
            {
                Emitter* child = emitter->childHead[i];
                while( child )
                {
                    Emitter* childNext = child->next;

                    if( child->emitterSetCreateID == m_EmitterSetCreateID &&
                        child->emitterCalc &&
                        child->IsAlive() )
                    {
                        // エミッタの終了処理
                        child->Finalize();

                        // システム依存のエミッタ終了処理
                        m_System->FinalizeEmitter( child );
                    }

                    child = childNext;
                }
            }

            // エミッタの終了処理
            emitter->Finalize();

            // システム依存のエミッタ終了処理
            m_System->FinalizeEmitter( emitter );
        }

        emitter = next;
    }

    return ret;
}

//---------------------------------------------------------------------------
//  エミッタセットの計算を指定回数回す。
//---------------------------------------------------------------------------
void EmitterSet::ForceCalc( u32 numLoop )
{
    if ( numLoop == 0 )
    {
        return;
    }

    //---------------------------------------------------------------------------
    // 指定回数分 Calc を回す。エミッタ更新処理＋CPUパーティクルの計算処理が走る。
    //---------------------------------------------------------------------------
    Calc( 1.0f, true, NULL );

    for ( u32 i = 0; i < numLoop-1; i++ )
    {
        Calc( 1.0f, true, NULL );
    }

    //---------------------------------------------------------------------------
    // GPU+SO の場合、ここで初期位置を SO バッファにコピーしておく。
    // TODO: コピーする前に、パーティクルの位置を↑の時間分進める。
    //---------------------------------------------------------------------------
    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        const EmitterCalcType calcType = emitter->GetCalcType();
        if( calcType == EFT_EMITTER_CALC_TYPE_GPU_SO )
        {
            // SO バッファを強制的に更新
            emitter->ForceUpdateForStreamOutBuffer();
        }
        else if( calcType == EFT_EMITTER_CALC_TYPE_CPU )
        {
            // 現状、親エミッタは CPU エミッタ限定なので絞っておく
            // チャイルドも含めて走査する
            for ( u32 i = 0; i < emitter->emitterRes->childEmitterResCount; i++ )
            {
                Emitter* child = emitter->childHead[i];
                while( child )
                {
                    const EmitterCalcType childCalcType = child->GetCalcType();
                    if( childCalcType == EFT_EMITTER_CALC_TYPE_GPU_SO )
                    {
                        // SO バッファを強制的に更新
                        child->ForceUpdateForStreamOutBuffer();
                    }
                    child = child->next;
                }
            }
        }
        emitter = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  エミッタセットの計算処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Calc( f32 frameRate, bool swapBuffer, EmitterCalcLodCallback emitterCalcLod )
{
    if ( !m_IsCalc )
    {
        return;
    }

    m_ProcessingEmitterNum       = 0;
    m_CalcSkipEmitterNum         = 0;
    m_ProcessingCpuParticleNum   = 0;
    m_ProcessingGpuParticleNum   = 0;
    m_ProcessingGpuSoParticleNum = 0;
    m_ProcessingEmitterAnimNum   = 0;
    m_ProcessingCpuEmitterNum    = 0;
    m_ProcessingGpuEmitterNum    = 0;
    m_ProcessingGpuSoEmitterNum  = 0;
    m_ProcessingStripeNum        = 0;
    m_ProcessingSuperStripeNum   = 0;
    m_AllocedFromDynamicHeapSize = 0;
    m_DrawPath                   = 0;
    m_ReqFrameBufferTexturePath  = 0;
    m_ReqDepthBufferTexturePath  = 0;
    m_FrameRate                  = frameRate;

    Emitter* emitter             = m_EmitterHead;
    bool enableIsAnyEmitterFade = false;

    while( emitter )
    {
        Emitter* next = emitter->next;

        // 親エミッタの処理
        // マニュアル放出モード時は既に放出済みの場合のみ処理
        if ( m_IsManualEmission && emitter->ptclNum == 0 && emitter->manualEmissionNum == 0 )
        {
            emitter->time += frameRate;
            m_ProcessingEmitterNum++;
        }
        else
        {
            EmitterCalculationResult ret = CalcEmitter( emitter, swapBuffer, emitterCalcLod );

            // 子エミッタの処理
            emitter->aliveChildren = false;

            for ( u32 i = 0; i < emitter->emitterRes->childEmitterResCount; i++ )
            {
                Emitter* child = emitter->childHead[i];
                while( child )
                {
                    Emitter* childNext = child->next;
                    const EmitterCalculationResult childCalcResult = CalcEmitter( child, swapBuffer, emitterCalcLod );

                    if ( childCalcResult == EmitterCalculationResult_Kill )
                    {
                        KillEmitter( child );
                    }
                    else
                    {
                        emitter->aliveChildren = true;
                    }

                    if( child->m_IsSoloFade )
                    {
                        // m_IsSoloFade が立っているのを検知したら、通知フラグを立てる
                        enableIsAnyEmitterFade = true;
                    }

                    child = childNext;
                }
            }

            // 子エミッタも全死亡であれば親エミッタも削除
            if ( ret == EmitterCalculationResult_Kill && !emitter->aliveChildren )
            {
                KillEmitter( emitter );
            }
        }

        if( emitter->m_IsSoloFade )
        {
            // m_IsSoloFade が立っているのを検知したら、通知フラグを立てる
            enableIsAnyEmitterFade = true;
        }

        emitter = next;
    }

    if( enableIsAnyEmitterFade )
    {
        // エミッタのどれかが単独で Fade されている
        m_IsAnyEmitterFade = true;
    }

    // Dirtyフラグ解除
    if ( frameRate > 0.0f )
    {
        m_IsEmitterSRTDirty = 0;
    }
    // フリップ済みフラグを元に戻す
    m_IsBufferFliped = false;
}


//---------------------------------------------------------------------------
//  エミッタ計算処理
//---------------------------------------------------------------------------
EmitterCalculationResult EmitterSet::CalcEmitter( Emitter* emitter, bool swapBuffer, EmitterCalcLodCallback emitterCalcLod )
{
    bool ret                    = true;
    bool isBatchProcStreamOut   = m_System->IsBatchProcessStreamOutEmitter();

    if ( m_FrameRate > 0.0f )
    {
        // エミッタカリングコールバック
        // コールバック内で emitter->calcSkipFlag/emitter->drawViewFlag のフラグ操作が行われる
        if ( emitterCalcLod )
        {
            EmitterCalcLodArg arg;
            arg.emitter   = emitter;
            const EmitterCalculationResult lodResult = emitterCalcLod( arg );
            if( lodResult == EmitterCalculationResult_Kill )
            {
                // Kill の場合はもう処理を回さない
                return EmitterCalculationResult_Kill;
            }
            else if( lodResult == EmitterCalculationResult_Fade )
            {
                // Fade の場合は個別フェードのスイッチを入れる
                if( !emitter->m_IsSoloFade )
                {
                    emitter->Fade();
                }
            }
        }

        if ( !emitter->calcSkipFlag )
        {
            const bool swapBufferEnable = m_IsBufferFliped ? false : swapBuffer;
            // 通常計算処理
            {
#ifndef NW_RELEASE
                nw::ut::Tick begin = nw::ut::Tick::GetSystemCurrent();
#endif
                ret = emitter->emitterCalc->Calc( emitter, m_FrameRate, swapBufferEnable, ( m_IsFade != 0 ), !m_IsManualEmission );
#ifndef NW_RELEASE
                nw::ut::Tick end = nw::ut::Tick::GetSystemCurrent();
                emitter->m_CpuCost = end - begin;
#endif
            }
            // エミッタ計算処理内での生死判定
            if ( ret )
            {
                // ストリームアウトエミッタ一括処理モードの場合はリストに追加
                if ( isBatchProcStreamOut && emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_GPU_SO && ( emitter->ptclNum > 0 || emitter->emitterRes->emitterPluginData ) )
                {
                    m_System->AddStreamOutEmitterList( emitter );
                }

                UpdateProcessingInfo( emitter );
            }
        }
        else
        {
            // calcSkipFlagで処理をスキップ
            emitter->frameRate = m_FrameRate;
            if ( swapBuffer )
            {
                emitter->Swap( EFT_BUFFER_MODE_DOUBLE );
                emitter->emitterCalc->MakeDynamicUniformBlock( emitter, m_FrameRate );
            }

            //
            // TODO:エミッタマトリクスのみ更新する
            //

            UpdateProcessingInfo( emitter );
            m_CalcSkipEmitterNum++;
        }
    }
    else
    {
        // フレームを進めない
        emitter->frameRate = m_FrameRate;
        if ( swapBuffer )
        {
            emitter->Swap( EFT_BUFFER_MODE_DOUBLE );
            emitter->emitterCalc->MakeDynamicUniformBlock( emitter, m_FrameRate );
        }

        UpdateProcessingInfo( emitter );
    }

    // 描画パスフラグを立てる
    m_DrawPath |= ( 0x1 << emitter->drawPath );

    // フレームバッファ要求フラグを立てる
    if ( emitter->IsRequestFrameBufferTexture() )
    {
        m_ReqFrameBufferTexturePath |= ( 0x1 << emitter->drawPath );
    }

    // デプスバッファ要求フラグを立てる
    if ( emitter->IsRequestDepthTexture() )
    {
        m_ReqDepthBufferTexturePath |= ( 0x1 << emitter->drawPath );
    }

    // ここまで処理が来た場合は、 true なら生存状態、 false なら Kill して終了する
    return ret ? EmitterCalculationResult_Alive : EmitterCalculationResult_Kill;
}


//---------------------------------------------------------------------------
// エミッタ描画GPU処理コスト計測コールバック
//---------------------------------------------------------------------------
void _drawEmitterProfilerCallback( DrawEmitterProfilerCallback profiler,
                                   System*     system,
                                   EmitterSet* emitterSet,
                                   Emitter*    emitter,
                                   bool        beforeRender,
                                   ShaderType  shaderType,
                                   bool        calcStreamOut,
                                   void*       userParam,
                                   u32         resourceID )
{
    DrawEmitterProfilerArg arg;
    arg.system              = system;
    arg.emitterSet          = emitterSet;
    arg.emitter             = emitter;
    arg.beforeRenderEmitter = beforeRender;
    arg.shaderType          = shaderType;
    arg.calcStreamOut       = calcStreamOut;
    arg.userParam           = userParam;
    arg.resourceID          = resourceID;

    profiler( arg );
}

//---------------------------------------------------------------------------
//  チャイルドエミッタセットの描画処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::DrawChildEmitter( bool                         beforeThanParent,
                                   Emitter*                     parentEmitter,
                                   ShaderType                   shaderType,
                                   u32                          drawPath,
                                   u32                          drawViewFlag,
                                   bool                         calcStreamOut,
                                   void*                        userParam,
                                   EmitterDrawCullingCallback   emitterDrawCull,
                                   DrawEmitterProfilerCallback  profiler )
{
    // 子エミッタの処理
    for ( u32 i = 0; i < parentEmitter->emitterRes->childEmitterResCount; i++ )
    {
        if ( parentEmitter->emitterRes->childEmitterResSet[i]->emitterData->inherit.preDraw == (u8)beforeThanParent )
        {
            Emitter* child = parentEmitter->childHead[i];
            while( child )
            {
                Emitter* childNext = child->next;
                DrawEmitter( child, shaderType, drawPath, drawViewFlag, calcStreamOut, userParam, emitterDrawCull, profiler );
                child = childNext;
            }
        }
    }
}


//---------------------------------------------------------------------------
//  エミッタセットの計算処理を行います。
//---------------------------------------------------------------------------
void EmitterSet::Draw( ShaderType shaderType, u32 drawPath, u32 drawViewFlag, bool calcStreamOut, void* userParam,
                       EmitterDrawCullingCallback emitterDrawCull, DrawEmitterProfilerCallback profiler )
{
    // TODO:EffectMakerによるVisibility操作
    if ( !m_EmitterSetRes->visibility && m_GroupID == (EFT_GROUP_MAX-1) )
    {
        return;
    }
#ifndef NW_RELEASE
    if ( profiler )
    {
        _drawEmitterProfilerCallback( profiler, this->GetSystem(), this, NULL, true, shaderType, calcStreamOut, userParam, m_ResourceID );
    }
#endif
    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        // 子エミッタの描画処理(親より前に描画)
        if ( emitter->aliveChildren )
        {
            DrawChildEmitter( true, emitter, shaderType, drawPath, drawViewFlag, calcStreamOut, userParam, emitterDrawCull, profiler );
        }

        // 親エミッタの描画処理
        DrawEmitter( emitter, shaderType, drawPath, drawViewFlag, calcStreamOut, userParam, emitterDrawCull, profiler );

        // 子エミッタの描画処理(親より後に描画)
        if ( emitter->aliveChildren )
        {
            DrawChildEmitter( false, emitter, shaderType, drawPath, drawViewFlag, calcStreamOut, userParam, emitterDrawCull, profiler );
        }

        emitter = emitter->next;
    }
#ifndef NW_RELEASE
    if ( profiler )
    {
        _drawEmitterProfilerCallback( profiler, this->GetSystem(), this, NULL, false, shaderType, calcStreamOut, userParam, m_ResourceID );
    }
#endif
}

//---------------------------------------------------------------------------
//  エミッタ描画処理
//---------------------------------------------------------------------------
void EmitterSet::DrawEmitter( Emitter*                    emitter,
                              ShaderType                  shaderType,
                              u32                         drawPath,
                              u32                         drawViewFlag,
                              bool                        calcStreamOut,
                              void*                       userParam,
                              EmitterDrawCullingCallback  emitterDrawCull,
                              DrawEmitterProfilerCallback profiler )
{
    bool    drawCullCheck   = true;
    bool    alive           = emitter->IsAlive();
    bool    renderEnable    = emitter->IsRenderAvailable();
    bool    drawViewEnable  = ( emitter->drawViewFlag & drawViewFlag ) != 0;
    bool    drawPathEnable  = ( drawPath & ( 0x1 << emitter->drawPath ) ) != 0;

    if( alive && renderEnable && drawViewEnable && drawPathEnable )
    {
        if ( emitterDrawCull )
        {
            EmitterDrawCullArg arg;
            arg.emitter     = emitter;
            arg.userParam   = userParam;
            drawCullCheck   = emitterDrawCull( arg );
        }

        if( drawCullCheck )
        {
            if ( profiler )
            {
                _drawEmitterProfilerCallback( profiler, m_System, NULL, emitter, false, shaderType, calcStreamOut, userParam, m_ResourceID );
            }
            else
            {
                m_System->DrawEmitter( emitter, shaderType, calcStreamOut, userParam );
            }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタを生成します。
//---------------------------------------------------------------------------
Emitter* EmitterSet::CreateEmitter( const EmitterResource* emitterRes, u32 particleMax, Emitter* parent, s32 childIndex )
{
    Emitter* emitter = m_System->AllocEmitter();
    if ( !emitter )
    {
        return NULL;
    }

    // エミッタセット登録
    emitter->emitterSet = this;

    // エミッタプラグインコールバックの取得
    if ( emitterRes->emitterPluginIndex > 0 )
    {
        emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] =
            m_System->GetEmitterPluginCallbackSet( static_cast<EmitterPluginID>( emitterRes->emitterPluginIndex-1 ) );
    }

    // カスタムシェーダコールバックの取得
    if ( emitterRes->emitterData->shader.customShaderIndex > 0 )
    {
        u32 id = emitterRes->emitterData->shader.customShaderIndex + EFT_CUSTOM_SHADER_CALLBACK_ID_NONE;
        if ( m_System->IsEnabelCallbackSet( static_cast<CallBackID>(id) ) )
        {
            CallbackSet* cbSet = m_System->GetCallbackSet( static_cast<CallBackID>( id ) );
            emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] = cbSet;
        }
        else
        {
            OutputWarning( "CustomShader Callback not Set. CustomShaderID:%d\n", emitterRes->emitterData->shader.customShaderIndex );
        }
    }
    else
    {
        // カスタムシェーダ 設定ナシの場合
        if ( m_System->IsEnabelCallbackSet( static_cast<CallBackID>( EFT_CUSTOM_SHADER_CALLBACK_ID_NONE ) ) )
        {
            CallbackSet* cbSet = m_System->GetCallbackSet( EFT_CUSTOM_SHADER_CALLBACK_ID_NONE );
            emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] = cbSet;
        }
    }

    // カスタムアクションコールバックの取得
    if ( emitterRes->emitterData->action.customActionIndex > 0 )
    {
        if ( emitterRes->customActionParam )
        {
            const u32 id = emitterRes->emitterData->action.customActionIndex - 1;
            if ( m_System->IsEnabelCallbackSet( static_cast<CallBackID>(id) ) )
            {
                CallbackSet* cbSet = m_System->GetCallbackSet( static_cast<CallBackID>( id ) );
                emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] = cbSet;
            }
            else
            {
                OutputWarning( "CustomAction Callback not Set. CustomActionID:%d\n", emitterRes->emitterData->action.customActionIndex );
            }
        }
        else
        {
            Warning( (void *)emitter, EFT_WARNING_CUSTOMACTION_PARAM_IS_NOT_EXIST );
        }
    }

    // 描画パスコールバック
    emitter->SetDrawPath( emitterRes->emitterData->emitter.drawPath );

    // システム依存のエミッタ初期化
    m_System->InitializeEmitter( emitter );

    // エミッタの初期化
    bool ret = emitter->Initialize( emitterRes, particleMax, m_Heap );
    if ( !ret )
    {
        // eft2ランタイム側でエミッタ初期化失敗時は、初期化コールバックを呼び出していないので、
        // pEmitter->Finalize では破棄コールバックを呼び出さない
        bool invokeCallback = emitter->isEmitterInitializeFailed ? false : true;
        emitter->Finalize( invokeCallback );
        m_System->FinalizeEmitter( emitter );
        if ( emitter->isEmitterInitializeFailed )
        {
            OutputWarning( "Emitter Initialize Failed.\n" );
        }
        return NULL;
    }

    // 親か子か
    if ( childIndex != -1 )
    {
        emitter->isChild            = true;
        emitter->childResIndex      = childIndex;
    }
    else
    {
        emitter->isChild            = false;
        emitter->childResIndex      = -1;
    }

    // emitterをセットツリーに追加
    AddEmitterToList( emitter, parent );

    emitter->emitterCreateID = m_EmitterCreateID;
    m_EmitterCreateID++;

    return emitter;
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::KillEmitter( Emitter* emitter )
{
    _KillEmitter( emitter );
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::_KillEmitter( Emitter* emitter )
{
    // エミッタをリストから削除
    RemoveEmitterFromList( emitter );

    // エミッタの終了処理
    emitter->Finalize();

    // システム依存のエミッタ終了処理
    m_System->FinalizeEmitter( emitter );
}

//---------------------------------------------------------------------------
//  エミッタを追加します。
//---------------------------------------------------------------------------
void EmitterSet::AddEmitterToList( Emitter* emitter, Emitter* parent )
{
    // 親の場合
    if ( !emitter->isChild )
    {
        if ( m_EmitterTail == NULL )
        {
            m_EmitterTail        = emitter;
            emitter->prev       = NULL;
            emitter->next       = NULL;
        }
        else
        {
            m_EmitterTail->next  = emitter;
            emitter->prev       = m_EmitterTail;
            m_EmitterTail        = emitter;
            emitter->next       = NULL;
        }

        if ( m_EmitterHead == NULL )
        {
            m_EmitterHead = emitter;
        }
    }
    else
    // 子の場合
    {
        EFT_NULL_ASSERT( parent );
        emitter->parentEmitter  = parent;
        s32      index          = emitter->childResIndex;

        if ( parent->childTail[index] == NULL )
        {
            parent->childTail[index]        = emitter;
            emitter->prev                   = NULL;
            emitter->next                   = NULL;
        }
        else
        {
            parent->childTail[index]->next  = emitter;
            emitter->prev                   = parent->childTail[index];
            parent->childTail[index]        = emitter;
            emitter->next                   = NULL;
        }

        if ( parent->childHead[index] == NULL )
        {
            parent->childHead[index] = emitter;
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::RemoveEmitterFromList( Emitter* emitter )
{
    // 親の場合
    if ( !emitter->isChild )
    {
        if ( emitter->prev && emitter->next )
        {
            emitter->prev->next = emitter->next;
            emitter->next->prev = emitter->prev;
        }
        else if ( m_EmitterHead == emitter )
        {
            m_EmitterHead = emitter->next;
            if ( emitter->next ) { emitter->next->prev = emitter->prev; }
            else                 { m_EmitterTail = NULL; }
        }
        else if ( m_EmitterTail == emitter )
        {
            m_EmitterTail = emitter->prev;
            if ( emitter->prev ) { emitter->prev->next = emitter->next; }
            else                 { m_EmitterHead = NULL; }
        }
    }
    else
    // 子の場合
    {
        EFT_NULL_ASSERT( emitter->parentEmitter );
        Emitter* parent = emitter->parentEmitter;
        s32      index  = emitter->childResIndex;

        if ( emitter->prev && emitter->next )
        {
            emitter->prev->next = emitter->next;
            emitter->next->prev = emitter->prev;
        }
        else if ( parent->childHead[index] == emitter )
        {
            parent->childHead[index] = emitter->next;
            if ( emitter->next ) { emitter->next->prev = emitter->prev; }
            else                 { parent->childTail[index] = NULL; }
        }
        else if ( parent->childTail[index] == emitter )
        {
            parent->childTail[index] = emitter->prev;
            if ( emitter->prev ) { emitter->prev->next = emitter->next; }
            else                 { parent->childHead[index] = NULL; }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタの放出を停止します。
//---------------------------------------------------------------------------
void EmitterSet::Fade()
{
    m_IsFade = true;
}

//---------------------------------------------------------------------------
//  リソース更新に伴うアップデートを行います。
//---------------------------------------------------------------------------
bool EmitterSet::UpdateFromResource( EmitterResource* emitterResSet )
{
    Emitter* emitter = m_EmitterHead;

    while( emitter )
    {
        if( emitter->IsAlive() && emitter->emitterRes == emitterResSet )
        {
            bool result = emitter->ResourceUpdate();
            if( !result )
            {
                return false;
            }
        }
        emitter = emitter->next;
    }
    return true;
}

//---------------------------------------------------------------------------
//  放出レートのスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmissionRatioScale( f32 ratio )
{
    m_EmissionRatioScale = ratio;
    if ( m_EmissionRatioScale > 1.0f )
    {
        m_EmissionRatioScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  放出間隔のスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmissionIntervalScale( f32 ratio )
{
    m_EmissionIntervalScale = ratio;
    if ( m_EmissionIntervalScale < 1.0f )
    {
        m_EmissionIntervalScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  パーティクル寿命のスケール値を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetParticleLifeScale( f32 ratio )
{
    m_ParticleLifeScale = ratio;
    if ( m_ParticleLifeScale > 1.0f )
    {
        m_ParticleLifeScale = 1.0f;
    }
}

//---------------------------------------------------------------------------
//  マトリクスを設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetMatrix( const nw::math::MTX34& matrixSRT )
{
    m_MatrixSRT = matrixSRT;

    // matrixRT
    nw::math::VEC3 basisX( matrixSRT.m[0][0], matrixSRT.m[1][0], matrixSRT.m[2][0] );
    nw::math::VEC3 basisY( matrixSRT.m[0][1], matrixSRT.m[1][1], matrixSRT.m[2][1] );
    nw::math::VEC3 basisZ( matrixSRT.m[0][2], matrixSRT.m[1][2], matrixSRT.m[2][2] );

    m_AutoCalcScale.x = basisX.Length();
    m_AutoCalcScale.y = basisY.Length();
    m_AutoCalcScale.z = basisZ.Length();

    if( m_AutoCalcScale.x > 0.0f )
    {
        f32 inv = 1.0f / m_AutoCalcScale.x;
        m_MatrixRT.m[0][0] = matrixSRT.m[0][0] * inv;
        m_MatrixRT.m[1][0] = matrixSRT.m[1][0] * inv;
        m_MatrixRT.m[2][0] = matrixSRT.m[2][0] * inv;
    }
    else
    {
        m_MatrixRT.m[0][0] = m_MatrixRT.m[1][0] = m_MatrixRT.m[2][0] = 0.0f;
    }

    if( m_AutoCalcScale.y > 0.0f )
    {
        f32 inv = 1.0f / m_AutoCalcScale.y;
        m_MatrixRT.m[0][1] = matrixSRT.m[0][1] * inv;
        m_MatrixRT.m[1][1] = matrixSRT.m[1][1] * inv;
        m_MatrixRT.m[2][1] = matrixSRT.m[2][1] * inv;
    }
    else
    {
        m_MatrixRT.m[0][1] = m_MatrixRT.m[1][1] = m_MatrixRT.m[2][1] = 0.0f;
    }

    if( m_AutoCalcScale.z > 0.0f )
    {
        f32 inv = 1.0f / m_AutoCalcScale.z;
        m_MatrixRT.m[0][2] = matrixSRT.m[0][2] * inv;
        m_MatrixRT.m[1][2] = matrixSRT.m[1][2] * inv;
        m_MatrixRT.m[2][2] = matrixSRT.m[2][2] * inv;
    }
    else
    {
        m_MatrixRT.m[0][2] = m_MatrixRT.m[1][2] = m_MatrixRT.m[2][2] = 0.0f;
    }

    m_MatrixRT.m[0][3] = matrixSRT.m[0][3];
    m_MatrixRT.m[1][3] = matrixSRT.m[1][3];
    m_MatrixRT.m[2][3] = matrixSRT.m[2][3];

    updateParticleScale_();
    m_IsEmitterSRTDirty = 1;
}

//---------------------------------------------------------------------------
//  マトリクスを設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetPos( const nw::math::VEC3& pos )
{
    m_MatrixSRT.SetTranslate( pos );
    m_MatrixRT.SetTranslate( pos );
    m_IsEmitterSRTDirty = 1;
}

//---------------------------------------------------------------------------
//  エミッタセットを削除します。
//---------------------------------------------------------------------------
void EmitterSet::Kill( bool immediate )
{
    m_System->KillEmitterSet( this, immediate );
}

//---------------------------------------------------------------------------
//  無限寿命のエミッタを削除します。
//---------------------------------------------------------------------------
void EmitterSet::KillInfinityEmitter()
{
    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        Emitter* next = emitter->next;
        if ( emitter->emitterData->ptcl.isLifeInfinity )
        {
            KillEmitter( emitter );
        }
        emitter = next;
    }
}

//---------------------------------------------------------------------------
//  無限寿命のエミッタ( パーティクル )を含むかどうかチェックします。
//---------------------------------------------------------------------------
bool EmitterSet::IsHaveInfinityEmitter() const
{
    return m_EmitterSetRes->isLifeInfinity;
}

//---------------------------------------------------------------------------
//  所属エミッタインスタンスを取得します。
//---------------------------------------------------------------------------
const Emitter* EmitterSet::GetAliveEmitter( u32 idx )
{
    u32 cnt = 0;
    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        if ( cnt == idx ) return emitter;
        cnt++;
        emitter = emitter->next;
    }

    return NULL;
}

//---------------------------------------------------------------------------
//  エミッタ描画処理の有効/無効を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterVisible( const char* emitterName, bool flag )
{
    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        if ( strcmp( emitter->emitterData->name, emitterName ) == 0 )
        {
            emitter->isVisible = flag;
        }
        emitter = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  パーティクル放出ポイントリストを設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetParticleEmissionPoints( s32 numPoint, nw::math::VEC3* points )
{
    if ( !m_IsManualEmission )
    {
        Warning( this, EFT_WARNING_NOT_SET_MANUAL_EMIT_MODE );
        return;
    }

    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        emitter->manualEmitPoints  = points;
        emitter->manualEmissionNum = numPoint;

        emitter = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  パーティクルを手動で放出します。
//---------------------------------------------------------------------------
void EmitterSet::EmitParticle( const nw::math::VEC3 &pos )
{
    if ( !m_IsManualEmission )
    {
        Warning( this, EFT_WARNING_NOT_SET_MANUAL_EMIT_MODE );
        return;
    }

    Emitter* emitter    = m_EmitterHead;

    while( emitter )
    {
        if( !m_IsBufferFliped )
        {
            // まだバッファをフリップしていないなら、この時点でフリップする
            emitter->Swap( EFT_BUFFER_MODE_DOUBLE );
        }

        // エミッタマトリクスを更新
        emitter->matrixSRT.SetMult( m_MatrixSRT, emitter->resMatrixSRT );
        emitter->matrixRT.SetMult(  m_MatrixRT,  emitter->resMatrixRT );

        const f32 emitRateRandom = emitter->emitterData->emission.rateRandom / 100.0f * emitter->emitterData->emission.rate;
        const f32 emitReduction  = emitRateRandom * emitter->random.GetF32();
        const u32 emitNum        = static_cast<u32>(
                    ( emitter->emitterData->emission.rate - emitReduction ) * emitter->emitRatio * GetEmissionRatioScale() );

        // マニュアル放出用オフセットを設定
        emitter->manualEmitCommonOffset = pos;

        // 手動で放出
        emitter->emitterCalc->Emit( emitter, emitNum, true );

        // マニュアル放出用オフセットを解除
        emitter->manualEmitCommonOffset = nw::math::VEC3( 0, 0, 0 );

        emitter   = emitter->next;
    }

    // フリップ済みフラグを必要に応じて立てる
    if( !m_IsBufferFliped )
    {
        m_IsBufferFliped = true;
    }
}

//---------------------------------------------------------------------------
//! @brief        パーティクルを手動で放出します。
//---------------------------------------------------------------------------
void EmitterSet::EmitParticle( const nw::math::VEC3 &pos,
                               const nw::math::VEC4 &color,
                               const nw::math::VEC2 &scale,
                               const nw::math::MTX34 *matrixSRT )
{
    if ( !m_IsManualEmission )
    {
        Warning( this, EFT_WARNING_NOT_SET_MANUAL_EMIT_MODE );
        return;
    }

    Emitter* emitter    = m_EmitterHead;
    s32      emitIndex  = 0;

    // エミッタセットマトリクスを上書きする。
    if ( matrixSRT )
    {
        this->SetMatrix( *matrixSRT );
    }

    while( emitter )
    {
        if( !m_IsBufferFliped )
        {
            // まだバッファをフリップしていないなら、この時点でフリップする
            emitter->Swap( EFT_BUFFER_MODE_DOUBLE );
        }

        // エミッタマトリクスを更新
        emitter->matrixSRT.SetMult( m_MatrixSRT, emitter->resMatrixSRT );
        emitter->matrixRT.SetMult(  m_MatrixRT,  emitter->resMatrixRT );

        const f32 emitRateRandom = emitter->emitterData->emission.rateRandom / 100.0f * emitter->emitterData->emission.rate;
        const f32 emitReduction  = emitRateRandom * emitter->random.GetF32();
        const u32 emitNum        = static_cast<u32>(
            ( emitter->emitterData->emission.rate - emitReduction ) * emitter->emitRatio * GetEmissionRatioScale() );

        for ( u32 i = 0; i < emitNum; i++ )
        {
            // マニュアル放出用オフセットを設定
            emitter->manualEmitCommonOffset = pos;

            emitIndex = emitter->emitterCalc->Emit( emitter, 1, true );

            // マニュアル放出用オフセットを解除
            emitter->manualEmitCommonOffset = nw::math::VEC3( 0, 0, 0 );

            if ( emitIndex != -1 )
            {
                emitter->particleAttr[emitIndex].initColor0.x  *= color.x;
                emitter->particleAttr[emitIndex].initColor0.y  *= color.y;
                emitter->particleAttr[emitIndex].initColor0.z  *= color.z;
                emitter->particleAttr[emitIndex].initColor0.w  *= color.w;

                emitter->particleAttr[emitIndex].scale.x       *= scale.x;
                emitter->particleAttr[emitIndex].scale.y       *= scale.y;

                // ParticlePosAttribute は毎フレームの Calc() で FlushCache されるのでここでは Flush しない。
                // ParticleAttribute は更新が起きた時以外 Flush されないので、このタイミングで Flush しておく。
                MemUtil::FlushCache( &emitter->particleAttr[emitIndex].initColor0, sizeof(nw::math::VEC4) );
                MemUtil::FlushCache( &emitter->particleAttr[emitIndex].scale, sizeof(nw::math::VEC4) );
            }
        }

        emitter   = emitter->next;
    }

    // フリップ済みフラグを必要に応じて立てる
    if( !m_IsBufferFliped )
    {
        m_IsBufferFliped = true;
    }
}


//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor( const nw::math::VEC4 &color0, const nw::math::VEC4 &color1 )
{
    Emitter* emitter    = m_EmitterHead;

    while( emitter )
    {
        emitter->color0.x = color0.x;
        emitter->color0.y = color0.y;
        emitter->color0.z = color0.z;
        emitter->color0.w = color0.w;

        emitter->color1.x = color1.x;
        emitter->color1.y = color1.y;
        emitter->color1.z = color1.z;
        emitter->color1.w = color1.w;

        emitter   = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor0( const nw::math::VEC4 &color0 )
{
    Emitter* emitter    = m_EmitterHead;

    while( emitter )
    {
        emitter->color0.x = color0.x;
        emitter->color0.y = color0.y;
        emitter->color0.z = color0.z;
        emitter->color0.w = color0.w;
        emitter   = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  カラー RGBA値(乗算値)を設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetEmitterColor1( const nw::math::VEC4 &color1 )
{
    Emitter* emitter    = m_EmitterHead;

    while( emitter )
    {
        emitter->color1.x = color1.x;
        emitter->color1.y = color1.y;
        emitter->color1.z = color1.z;
        emitter->color1.w = color1.w;

        emitter   = emitter->next;
    }
}

//---------------------------------------------------------------------------
//  個数を指定してパーティクルを手動で放出するモードに設定します。
//---------------------------------------------------------------------------
void EmitterSet::SetManualParticleEmissionWithParticleCount( u32 particleCount )
{
    m_IsManualEmission          = true;
    m_ManualEmissionAssignNum   = particleCount;

    Emitter* emitter    = m_EmitterHead;
    while( emitter )
    {
        if( emitter->IsAlive() )
        {
            emitter->ptclMaxAssignmentNum = m_ManualEmissionAssignNum;
            emitter->ResourceUpdate();
        }
        emitter   = emitter->next;
    }
}
//---------------------------------------------------------------------------
//  マニュアルエミッタセットが消えても大丈夫な状態かを取得します。
//---------------------------------------------------------------------------
bool EmitterSet::IsManualEmitterSetReadyToExit() const
{
    if( !IsManualParticleEmission() )
    {
        return false;
    }

    Emitter* emitter = m_EmitterHead;
    while( emitter )
    {
        if( emitter->IsManualEmitterReadyToExit() == false )
        {
            // 一つでもまだ終われないエミッタがあるなら false
            return false;
        }
        emitter = emitter->next;
    }

    // ここまで処理が来たなら配下のエミッタは全部消えて大丈夫な状態。
    return true;
}

//---------------------------------------------------------------------------
//! @briefprivate 情報更新
//! @param[in] emitter TBD
//---------------------------------------------------------------------------
void EmitterSet::UpdateProcessingInfo( Emitter* emitter )
{
    if ( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_CPU )
    {
        m_ProcessingCpuParticleNum   += emitter->ptclProcessingNum;
        m_ProcessingCpuEmitterNum++;
    }
    if ( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_GPU )
    {
        m_ProcessingGpuParticleNum   += emitter->ptclNum;
        m_ProcessingGpuEmitterNum++;
    }
    if ( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_GPU_SO )
    {
        m_ProcessingGpuSoParticleNum += emitter->ptclNum;
        m_ProcessingGpuSoEmitterNum++;
    }

    m_ProcessingEmitterNum++;
    switch( emitter->emitterRes->emitterPluginIndex )
    {
    case ConnectionStripeSystem::PluginId:
    case StripeSystem::PluginId:
        // MEMO: ストライプは強制CPUエミッタなので、CPU基準でカウントする
        m_ProcessingStripeNum += emitter->ptclProcessingNum;
        break;
    case SuperStripeSystem::PluginId:
        // MEMO: ストライプは強制CPUエミッタなので、CPU基準でカウントする
        m_ProcessingSuperStripeNum += emitter->ptclProcessingNum;
        break;
    default:
        break;
    }

    m_ProcessingEmitterAnimNum   += emitter->calcedEmitterAnimCount;
    m_AllocedFromDynamicHeapSize += emitter->allocedFromDynamicHeapSize;
}

} // namespace eft2
} // namespace nw
