﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_TransitionCaptureModule.h"

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

#define NN_CAPSRV_SCOPED_LOCK_MUTEX()   \
            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

namespace nn{ namespace capsrv{ namespace server{

    void TransitionCaptureModule::Initialize() NN_NOEXCEPT
    {
        m_Mutex.Initialize();
        m_CaptureModuleSharedTexture.Initialize(nn::capsrv::capture::GetCaptureModule(), "lastframe");
    }

    void TransitionCaptureModule::Finalize() NN_NOEXCEPT
    {
        m_CaptureModuleSharedTexture.Finalize();
    }

    nn::Result TransitionCaptureModule::AttachSharedMemoryPool(
        const nn::capsrv::capture::MemoryPoolImportData& poolInfo,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_SCOPED_LOCK_MUTEX();

        m_CaptureModuleSharedTexture.AttachSharedMemoryPool(poolInfo, layout);
        NN_RESULT_SUCCESS;
    }

    nn::Result TransitionCaptureModule::CaptureDisplayToSharedTexture(int dstTextureIndex, nn::vi::LayerStack srcLayerStack, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_CAPSRV_SCOPED_LOCK_MUTEX();

        // 必要なら復旧処理
        if(m_CaptureModuleSharedTexture.IsRecoveryRequired())
        {
            NN_RESULT_THROW_UNLESS(m_CaptureModuleSharedTexture.RecoverCapture(timeout), ResultScreenShotCaptureFailed());
        }

        // 撮影を実行
        {
            auto pCapturedBuffer = m_CaptureModuleSharedTexture.AcquireCaptureBuffer(dstTextureIndex, timeout, srcLayerStack);
            NN_RESULT_THROW_UNLESS(pCapturedBuffer != nullptr, ResultScreenShotCaptureFailed());
            NN_UTIL_SCOPE_EXIT{
                m_CaptureModuleSharedTexture.ReleaseCaptureBuffer(pCapturedBuffer);
            };

            // 撮影対象が居なければ失敗
            if(pCapturedBuffer->IsContentTrivial())
            {
                //NN_CAPSRV_LOG_WARN("Acquired display capture id trivial\n");
                NN_RESULT_THROW(ResultScreenShotNoTarget());
            }
        }

        NN_RESULT_SUCCESS;
    }



}}}

