﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotWorker.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "visrv_ScreenShotServerObject.h"

namespace nn{ namespace capsrv{ namespace server{

    ScreenShotWorker::ScreenShotWorker() NN_NOEXCEPT
        : m_IsIdleEvent()
        , m_IsStopRequestedEvent()
        , m_MultiWait()
        , m_StopRequestedEventWaitHolder()
        , m_pWorkerThread()
    {
    }


    nn::Result ScreenShotWorker::Initialize(nn::os::ThreadType* pScreenShotWorkerThread) NN_NOEXCEPT
    {
        nn::os::InitializeEvent(&m_IsIdleEvent, true, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&m_IsStopRequestedEvent, false, nn::os::EventClearMode_ManualClear);

        nn::os::InitializeMultiWait(&m_MultiWait);
        nn::os::InitializeMultiWaitHolder(&m_StopRequestedEventWaitHolder, &m_IsStopRequestedEvent);
        m_StopRequestedEventWaitHolder.userData = StopRequestedWaitHolderTag;

        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_StopRequestedEventWaitHolder);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, g_ScreenShotControlRequestQueue.GetDequeueMultiWaitHolder());
        nn::os::LinkMultiWaitHolder(&m_MultiWait, g_ScreenShotRequestQueue.GetDequeueMultiWaitHolder());

        m_pWorkerThread = pScreenShotWorkerThread;

        NN_RESULT_SUCCESS;
    }

    void ScreenShotWorker::Finalize() NN_NOEXCEPT
    {
        nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);

        nn::os::FinalizeMultiWaitHolder(&m_StopRequestedEventWaitHolder);
        nn::os::FinalizeMultiWait(&m_MultiWait);

        nn::os::FinalizeEvent(&m_IsStopRequestedEvent);
        nn::os::FinalizeEvent(&m_IsIdleEvent);
    }

    void ScreenShotWorker::RunWorker() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_pWorkerThread == nn::os::GetCurrentThread(), "ScreenShotWorker::RunWorker() must be called on pre registerd worker thread");

        // TORIAEZU:
        // 排他としては不十分だが実用上問題ない。
        NN_SDK_REQUIRES(nn::os::TryWaitEvent(&m_IsIdleEvent));
        nn::os::ClearEvent(&m_IsIdleEvent);

        for(;;)
        {
            auto pHolder = nn::os::WaitAny(&m_MultiWait);
            auto tag = pHolder->userData;
            if(tag == StopRequestedWaitHolderTag)
            {
                break;
            }

            switch(tag)
            {
            case ScreenShotRequestWaitHolderTag_ScreenShotControl:
                {
                    void (*pFunction)(void*) = nullptr;
                    void* pParameter = nullptr;
                    g_ScreenShotControlRequestQueue.BeginProcessingRequest(&pFunction, &pParameter);
                    pFunction(pParameter);
                    g_ScreenShotControlRequestQueue.EndProcessingRequest();
                    break;
                }
            case ScreenShotRequestWaitHolderTag_ScreenShot:
                {
                    void (*pFunction)(void*) = nullptr;
                    void* pParameter = nullptr;
                    g_ScreenShotRequestQueue.BeginProcessingRequest(&pFunction, &pParameter);
                    pFunction(pParameter);
                    g_ScreenShotRequestQueue.EndProcessingRequest();
                    break;
                }
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        nn::os::SignalEvent(&m_IsIdleEvent);
    }

    void ScreenShotWorker::StopWorker() NN_NOEXCEPT
    {
        nn::os::SignalEvent(&m_IsStopRequestedEvent);
        nn::os::WaitEvent(&m_IsIdleEvent);
    }

    void ScreenShotWorker::BeginBoostingWorkerThreadPriority() NN_NOEXCEPT
    {
        auto pThread = m_pWorkerThread.load();
        NN_SDK_REQUIRES_NOT_NULL(pThread);
        if(pThread == nullptr)
        {
            return;
        }

        nn::os::ChangeThreadPriority(pThread, NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotWorkerBoosted));
    }

    void ScreenShotWorker::EndBoostingWorkerThreadPriority() NN_NOEXCEPT
    {
        auto pThread = m_pWorkerThread.load();
        NN_SDK_REQUIRES_NOT_NULL(pThread);
        if(pThread == nullptr)
        {
            return;
        }

        nn::os::ChangeThreadPriority(pThread, NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotWorker));
    }

}}}
