﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/capsrv/capsrv_ScreenShotAttribute.h>
#include "../screenshot/visrv_ScreenShotContext.h"

namespace nn{ namespace capsrv{ namespace server{

    class ScreenShotContextManager
    {
    public:
        static const int ContextCount = 2;

        enum ContextStatus
        {
            ContextStatus_Free = 0,
            ContextStatus_Used,
            ContextStatus_Suspended,
            ContextStatus_SuspendedNeedRecovery,
        };
    public:
        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        ScreenShotContext* AcquireContext() NN_NOEXCEPT;
        void ReleaseContext(ScreenShotContext* pContext) NN_NOEXCEPT;
        void SuspendContext(ScreenShotContext* pContext) NN_NOEXCEPT;
        ScreenShotContext* ResumeContext() NN_NOEXCEPT;
        ScreenShotContext* ResumeContext(ScreenShotContextSuspendedTask task) NN_NOEXCEPT;

        // Suspend された Context が存在する場合 true を返す。
        // true を返す場合、 ResumeContext() は非 null を返す。
        bool HasSuspendedContext() NN_NOEXCEPT;

        // 指定された task で Suspend された Context が存在する場合 true を返す。
        // true を返す場合、 ResumeContext() は非 null を返す。
        bool HasSuspendedContext(ScreenShotContextSuspendedTask task) NN_NOEXCEPT;

        // Suspend された Context のうち復旧が必要なものを順に復旧させる。
        // 復旧に失敗したものがあった場合、以降の Context は復旧させない。
        // 失敗した Context がない場合、 true を返す。
        bool TryRecoveryAll() NN_NOEXCEPT;


        static void SetupContextForTakingScreenShot(
            ScreenShotContext* pContext,
            uint64_t seqNo,
            nn::ncm::ApplicationId applicationId,
            nn::vi::LayerStack layerStack,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::UserIdList& userIdList,
            nn::TimeSpan timeout
        ) NN_NOEXCEPT;

        // pAppletData == nullptr の場合デフォルト値になる。
        static void SetupContextForSavingEditedScreenShot(
            ScreenShotContext* pContext,
            const void* pFullData,
            size_t fullDataSize,
            const void* pThumbData,
            size_t thumbDataSize,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::AppletData* pAppletData,
            nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT;

        // pAppletData == nullptr の場合デフォルト値になる。
        static void SetupContextForSavingScreenShotOfMovie(
            ScreenShotContext* pContext,
            const void* pFullData,
            size_t fullDataSize,
            const void* pThumbData,
            size_t thumbDataSize,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::AppletData* pAppletData,
            nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT;

        static void SetupContextForSavingApplicationScreenShotAruid(
            ScreenShotContext* pContext,
            const void* pFullData,
            size_t fullDataSize,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::UserIdList& userIdList,
            nn::applet::AppletResourceUserId aruid,
            OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT;

        static void SetupContextForOverlayMovieThumbnail(
            ScreenShotContext* pContext,
            capture::ImageBuffer* pImageBufferY,
            capture::ImageBuffer* pImageBufferUv,
            const nn::capsrv::AlbumFileId& fileId
        ) NN_NOEXCEPT;

        // pAppletData == nullptr の場合デフォルト値になる。
        // pApplicationData == nullptr の場合デフォルト値になる。
        static void SetupContextForProtoMovieMetaDataNv12(
            ScreenShotContext* pContext,
            void* pOutBuffer,
            size_t outBufferSize,
            capture::ImageBuffer* pImageBufferY,
            capture::ImageBuffer* pImageBufferUv,
            const nn::capsrv::AlbumFileId& fileId,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::AppletData* pAppletData,
            const nn::capsrv::ApplicationData* pApplicationData
        ) NN_NOEXCEPT;

        // pAppletData == nullptr の場合デフォルト値になる。
        // pApplicationData == nullptr の場合デフォルト値になる。
        static void SetupContextForProtoMovieMetaDataRgba(
            ScreenShotContext* pContext,
            void* pOutBuffer,
            size_t outBufferSize,
            const void* pFullData,
            size_t fullDataSize,
            const nn::capsrv::AlbumFileId& fileId,
            const nn::capsrv::ScreenShotAttribute& attribute,
            const nn::capsrv::AppletData* pAppletData,
            const nn::capsrv::ApplicationData* pApplicationData
        ) NN_NOEXCEPT;

        static void SetupContextForRawScreenShotReadStream(
            ScreenShotContext* pContext,
            nn::vi::LayerStack layerStack,
            nn::TimeSpan timeout
        ) NN_NOEXCEPT;

    private:
        int GetIndex(ScreenShotContext* pContext) const NN_NOEXCEPT;
        void ReleaseContextImpl(int index) NN_NOEXCEPT;

    private:
        nn::os::SdkMutexType m_Mutex;
        ContextStatus m_ContextStatusList[ContextCount];
        ScreenShotContext m_ContextList[ContextCount];
    };

}}}
