﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotCaptureModule.h"

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/capsrv/capsrv_ThumbnailFormat.h>
#include "../detail/capsrvServer_ParseLayerStackName.h"

#include "visrv_ScreenShotServerObject.h"

namespace nn{ namespace capsrv{ namespace server{

    namespace {

        ptrdiff_t GetOffset(ScreenShotWorkMemory* pWorkMemory, char* pMemory) NN_NOEXCEPT
        {
            char* pBase = pWorkMemory->deviceShared.all.memory;
            return pMemory - pBase;
        }

    }

    void ScreenShotCaptureModule::Initialize() NN_NOEXCEPT
    {
        auto pWorkMemory = &g_ScreenShotWorkMemory;
        capture::InitializeCaptureModule();
        m_pModule = capture::GetCaptureModule();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MemoryPool.Initialize(m_pModule, pWorkMemory->deviceShared.all.memory, sizeof(pWorkMemory->deviceShared.all)));

        // ScreenShot
        {
            // レイヤースタックの設定を取得
            nn::vi::LayerStack layerStack = DefaultScreenShotLayerStack;
            {
                char name[LayerStackNameSize + 1] = {};
                nn::settings::fwdbg::GetSettingsItemValue(name, LayerStackNameSize, "capsrv", "screenshot_layerstack");
                layerStack = detail::ParseLayerStackName(name, detail::CaptureUsage_ScreenShot);
                m_ScreenShot.defaultLayerStack = layerStack;
            }

            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight;
                info.format = capture::ImageFormat_Y_Bt601;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_ScreenShot.fullSizeY.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.screenShot.fullSizeRawImageY), ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageY
                ));
            }
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight / 2;
                info.format = capture::ImageFormat_U_Bt601;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_ScreenShot.fullSizeU.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.screenShot.fullSizeRawImageU), ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageU
                ));
            }
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight / 2;
                info.format = capture::ImageFormat_V_Bt601;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_ScreenShot.fullSizeV.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.screenShot.fullSizeRawImageV), ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageV
                ));
            }
            m_ScreenShot.fullSizeBuffer.InitializeYuvPlanar(m_pModule, capture::ImageFormat_Yuv440_Bt601_Planar, &m_ScreenShot.fullSizeY, &m_ScreenShot.fullSizeU, &m_ScreenShot.fullSizeV);
            {
                m_ScreenShot.fullSizeCapture.Initialize(m_pModule, "screenshot", layerStack, ScreenShotWidth, ScreenShotHeight, capture::ImageFormat_Yuv440_Bt601_Planar);
                m_ScreenShot.fullSizeCapture.SetCaptureBuffer(0, &m_ScreenShot.fullSizeBuffer);
            }
            {
                capture::ImageBufferInfo info;
                info.width  = ThumbnailImageSize_Width;
                info.height = ThumbnailImageSize_Height;
                info.format = capture::ImageFormat_Rgba8;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_ScreenShot.overlayThumbnail.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.screenShot.overlayThumbnailRawImage), ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawImage
                ));
            }
            {
                capture::ImageBufferInfo info;
                info.width = ViewerThumbnailImageSize_Width;
                info.height = ViewerThumbnailImageSize_Height;
                info.format = capture::ImageFormat_Rgba8;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_ScreenShot.viewerThumbnail.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.screenShot.viewerThumbnailRawImage), ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawImage
                ));
            }
        }

        // DebugScreenShot
        {
            nn::vi::LayerStack debugLayerStack = nn::vi::LayerStack_ApplicationForDebug;
            // imageBuffer
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight;
                info.format = capture::ImageFormat_Rgba8;
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_DebugScreenShot.imageBuffer.Initialize(
                    m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->deviceShared.keepPresentation.fullSizeRawImageRgba), ScreenShotWorkMemorySize_KeepPresentationFullSizeRawImage
                ));
            }
            // displayBuffer
            {
                m_DebugScreenShot.displayBuffer.InitializeRgba(m_pModule, capture::ImageFormat_Rgba8, &m_DebugScreenShot.imageBuffer);
            }
            // capture
            {
                m_DebugScreenShot.capture.Initialize(m_pModule, "raw", debugLayerStack, ScreenShotWidth, ScreenShotHeight, capture::ImageFormat_Rgba8);
                m_DebugScreenShot.capture.SetCaptureBuffer(0, &m_DebugScreenShot.displayBuffer);
            }
        }
    };

    void ScreenShotCaptureModule::Finalize() NN_NOEXCEPT
    {
        m_DebugScreenShot.capture.Finalize();
        m_DebugScreenShot.displayBuffer.Finalize();
        m_DebugScreenShot.imageBuffer.Finalize();

        m_ScreenShot.fullSizeCapture.Finalize();
        m_ScreenShot.fullSizeBuffer.Finalize();
        m_ScreenShot.fullSizeY.Finalize();
        m_ScreenShot.fullSizeU.Finalize();
        m_ScreenShot.fullSizeV.Finalize();
        m_ScreenShot.overlayThumbnail.Finalize();
        m_ScreenShot.viewerThumbnail.Finalize();

        m_MemoryPool.Finalize();
        capture::FinalizeCaptureModule();
    }

}}}
