﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

#include "../../capsrvServer_Config.h"

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    // DisplayCapture からメモリにコピーする
    // @param[in]  context.DisplayBufferRgba
    static nn::Result CopyDisplayCaptureRgbaToMemoryPartial(size_t* pOutReadSize, void* pOutBuffer, size_t bufferSize, ptrdiff_t offset, ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, DisplayBufferRgba);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("copy part ");

        const int pixelSize = 4;
        const int width = ScreenShotWidth;
        const int height = ScreenShotHeight;

        // 最小 1 px 単位での読取
        NN_RESULT_THROW_UNLESS(bufferSize % pixelSize == 0, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(offset % pixelSize == 0, ResultScreenShotInvalidSize());

        *pOutReadSize = 0;
        std::memset(pOutBuffer, 0, bufferSize);

        size_t stride = pixelSize * width;
        size_t imageSize = stride * height;

        ptrdiff_t srcStart = offset;
        ptrdiff_t dstStart = 0;
        ptrdiff_t copySize = bufferSize;

        if(srcStart < 0)
        {
            auto delta = -srcStart;
            srcStart = 0;
            dstStart += delta;
            copySize -= delta;
        }
        NN_RESULT_THROW_UNLESS(copySize > 0, nn::ResultSuccess());

        if(srcStart + copySize > static_cast<ptrdiff_t>(imageSize))
        {
            auto delta = srcStart + copySize - imageSize;
            copySize -= delta;
        }
        NN_RESULT_THROW_UNLESS(copySize > 0, nn::ResultSuccess());

        if(dstStart + copySize > static_cast<ptrdiff_t>(bufferSize))
        {
            auto delta = dstStart + copySize - bufferSize;
            copySize -= delta;
        }
        NN_RESULT_THROW_UNLESS(copySize > 0, nn::ResultSuccess());

        NN_CAPSRV_LOG_DEV("srcStart = %lld, dstStart = %lld, size = %lld\n", srcStart, dstStart, copySize);

        int srcYMin = static_cast<int>(srcStart / stride);
        int srcYMax = static_cast<int>((srcStart + copySize - pixelSize) / stride);

        int srcXMin = static_cast<int>((srcStart - srcYMin * stride) / pixelSize);
        int srcXMax = static_cast<int>(((srcStart + copySize - pixelSize) - srcYMax * stride) / pixelSize);

        NN_SDK_ASSERT(static_cast<ptrdiff_t>(srcXMin * pixelSize + srcYMin * stride) == static_cast<ptrdiff_t>(srcStart));
        NN_SDK_ASSERT(static_cast<ptrdiff_t>(srcXMax * pixelSize + srcYMax * stride) == static_cast<ptrdiff_t>(srcStart + copySize - pixelSize));

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pDisplayBuffer, context, DisplayBufferRgba);

        auto pImageBuffer = pDisplayBuffer->GetImageBuffer();
        NN_SDK_REQUIRES_NOT_NULL(pImageBuffer);

        size_t writtenSizeSum = 0;
        char* pDest = reinterpret_cast<char*>(pOutBuffer) + dstStart;
        for(int y = srcYMin; y <= srcYMax; y++)
        {
            int xMin = 0;
            int xMax = width - 1;
            if(y == srcYMin)
            {
                xMin = srcXMin;
            }
            if(y == srcYMax)
            {
                xMax = srcXMax;
            }
            int h = 1;
            int w = xMax - xMin + 1;
            auto rect = capture::MakeRectangle(xMin, y, w, h);

            NN_CAPSRV_LOG_DEV("copy y=[%d] x=[%d-%d]\n", y, xMin, xMax);

            size_t writtenSize = 0;
            NN_RESULT_DO(pImageBuffer->WriteToMemory(&writtenSize, pDest, 4 * w, capture::ImageFormat_Rgba8, &rect));
            NN_SDK_ASSERT(writtenSize == static_cast<size_t>(4 * w));
            pDest += writtenSize;
            writtenSizeSum += writtenSize;
        }
        NN_SDK_ASSERT(writtenSizeSum == static_cast<size_t>(copySize));

        *pOutReadSize = writtenSizeSum;
        NN_RESULT_SUCCESS;
    }


}}}}}
