﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    static ScreenShotContextRecoveryResult TryRecoveryAcquiringDisplayCaptureRgba(void* pArg) NN_NOEXCEPT
    {
        ScreenShotContext& context = *reinterpret_cast<ScreenShotContext*>(pArg);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, Timeout);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("recoverRGB");

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pDisplayCapture, context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(timeout, context, Timeout);

        if(pDisplayCapture->IsRecoveryRequired())
        {
            if(!pDisplayCapture->RecoverCapture(timeout))
            {
                return ScreenShotContextRecoveryResult_Failure;
            }
        }

        return ScreenShotContextRecoveryResult_SuccessToRelease;
    }

    // DisplayCapture を取得する
    // @param[in] context.CaptureModule
    // @param[in] context.DisplayCaptureRgba
    // @param[in] context.CapturingLayerStack
    // @param[in] context.Timeout
    // @param[out] context.DisplayBufferRgba
    static nn::Result AcquireDisplayCaptureRgba(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, Timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, CapturingLayerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, DisplayBufferRgba);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("captureRGB");
        bool isSuccess = false;

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pDisplayCapture, context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(timeout, context, Timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(layerStack, context, CapturingLayerStack);

        auto pCapturedBuffer = pDisplayCapture->AcquireCaptureBuffer(timeout, layerStack);

        // キャプチャ失敗
        if(pCapturedBuffer == nullptr)
        {
            //NN_CAPSRV_LOG_ERR("Acquiring display capture failed\n");
            if(pDisplayCapture->IsRecoveryRequired())
            {
                context.MarkRecoveryRequired(TryRecoveryAcquiringDisplayCaptureRgba, &context);
            }
            NN_RESULT_THROW(ResultScreenShotCaptureFailed());
        }
        NN_UTIL_SCOPE_EXIT{
            if(!isSuccess)
            {
                pDisplayCapture->ReleaseCaptureBuffer(pCapturedBuffer);
            }
        };

        // 撮影対象が居なければ失敗
        if(pCapturedBuffer->IsContentTrivial())
        {
            //NN_CAPSRV_LOG_WARN("Acquired display capture id trivial\n");
            NN_RESULT_THROW(ResultScreenShotNoTarget());
        }

        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, DisplayBufferRgba, pCapturedBuffer);
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    // DisplayCapture を解放する
    static void ReleaseDisplayCaptureRgba(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_INOUT(context, DisplayBufferRgba);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pCaputre, context, DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pBuffer, context, DisplayBufferRgba);

        pCaputre->ReleaseCaptureBuffer(pBuffer);

        NN_CAPSRV_SCREENSHOT_CONTEXT_RESET(context, DisplayBufferRgba);
    }

}}}}}
