﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SoftwareJpegEncoderYuv.h"

#include <csetjmp>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"
#include "capsrvServer_JpegErrorHandler.h"
#include "visrv_JpegDestinationMgrBase.h"

#ifdef NN_CAPSRV_USE_LIBJPEG_TURBO
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpegTurbo.h"
#else
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpeg.h"
#endif

namespace nn{ namespace capsrv{ namespace server{ namespace jpeg{


    namespace {
        static const uint32_t ComponentCountYuv = 3;

        void SetupEncodingParameterYuv(
            JpegEncoderType::jpeg_compress_struct* pCinfo,
            const SoftwareJpegEncoderStreamInputInfoYuv& inputInfo,
            uint32_t horizontalSubsamplingShift,
            uint32_t verticalSubsamplingShift,
            int quality,
            bool hasExif
        ) NN_NOEXCEPT
        {
            pCinfo->image_width  = static_cast<JpegEncoderType::JDIMENSION>(inputInfo.width);
            pCinfo->image_height = static_cast<JpegEncoderType::JDIMENSION>(inputInfo.height);
            pCinfo->input_components = ComponentCountYuv;
            pCinfo->in_color_space   = JpegEncoderType::J_COLOR_SPACE::JCS_YCbCr; // ignored if use raw data

            jpeg_set_defaults(pCinfo);
            jpeg_set_colorspace(pCinfo, JpegEncoderType::J_COLOR_SPACE::JCS_YCbCr);

            // enable raw data input
            pCinfo->raw_data_in = true;
            pCinfo->do_fancy_downsampling = false;
            pCinfo->comp_info[0].h_samp_factor = 1 << horizontalSubsamplingShift;
            pCinfo->comp_info[0].v_samp_factor = 1 << verticalSubsamplingShift;
            pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
            pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;

            jpeg_set_quality(pCinfo, quality, true);

            pCinfo->dct_method            = JpegEncoderType::J_DCT_METHOD::JDCT_ISLOW;
            pCinfo->optimize_coding       = false;

            pCinfo->write_JFIF_header = !hasExif;
        }

        void ProcessEncodeYuv(
            JpegEncoderType::jpeg_compress_struct* pCinfo,
            const SoftwareJpegEncoderYuvBuffer& imageWorkBuffers,
            uint32_t width,
            uint32_t horizontalSubsamplingShift,
            uint32_t verticalSubsamplingShift
        ) NN_NOEXCEPT
        {
            const uint32_t MaxVerticalSubsampling = 2; // Maximum supported vertical subsampling
            const uint32_t verticalSubsampling    = 1 << verticalSubsamplingShift;
            NN_SDK_REQUIRES_LESS_EQUAL(verticalSubsampling, MaxVerticalSubsampling);

            const size_t lumaStride   = width;
            const size_t chromaStride = width >> horizontalSubsamplingShift;

            JpegEncoderType::JSAMPROW   rawDataRow[ComponentCountYuv][DCTSIZE * MaxVerticalSubsampling] = {};
            JpegEncoderType::JSAMPARRAY rawData[ComponentCountYuv] = {rawDataRow[0], rawDataRow[1], rawDataRow[2]};

            // Fill array of rows to encode
            JpegEncoderType::JSAMPLE* yRow = static_cast<JpegEncoderType::JSAMPLE*>(imageWorkBuffers.pBufferY);
            JpegEncoderType::JSAMPLE* uRow = static_cast<JpegEncoderType::JSAMPLE*>(imageWorkBuffers.pBufferU);
            JpegEncoderType::JSAMPLE* vRow = static_cast<JpegEncoderType::JSAMPLE*>(imageWorkBuffers.pBufferV);
            size_t yIndex = 0;
            for (int i=0; i < DCTSIZE; ++i)
            {
                for (uint32_t j=0; j < verticalSubsampling; ++j)
                {
                    rawData[0][yIndex++] = yRow;
                    yRow += lumaStride;
                }
                rawData[1][i] = uRow;
                rawData[2][i] = vRow;
                uRow += chromaStride;
                vRow += chromaStride;
            }

            jpeg_write_raw_data(pCinfo, rawData, DCTSIZE * verticalSubsampling);
        }
    }

    nn::Result SoftwareJpegEncoderYuv::EncodeYuvStreamImpl(
        JpegDestinationMgrBase& destMgr,
        const SoftwareJpegEncoderStreamInputInfoYuv& inputInfo,
        int quality,
        bool hasExif,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        char* pWork = reinterpret_cast<char*>(pWorkBuffer);
        char* pWorkEnd = pWork + workBufferSize;

        const uint32_t horizontalSubsamplingShift = GetHorizontalSubsamplingShift(inputInfo.format);
        const uint32_t verticalSubsamplingShift   = GetVerticalSubsamplingShift(inputInfo.format);

        SoftwareJpegEncoderYuvBuffer partialImage;
        {
            size_t chromaBufferSize  = (inputInfo.width >> horizontalSubsamplingShift) * DCTSIZE;
            partialImage.bufferSizeY = (inputInfo.width * DCTSIZE) << verticalSubsamplingShift;
            partialImage.bufferSizeU = chromaBufferSize;
            partialImage.bufferSizeV = chromaBufferSize;
            NN_RESULT_THROW_UNLESS(
                workBufferSize >= partialImage.bufferSizeY + partialImage.bufferSizeU + partialImage.bufferSizeV,
                ResultInternalJpegWorkMemoryShortage()
            );
            partialImage.pBufferY = pWork;
            pWork += partialImage.bufferSizeY;
            partialImage.pBufferU = pWork;
            pWork += partialImage.bufferSizeU;
            partialImage.pBufferV = pWork;
            pWork += partialImage.bufferSizeV;
        }

        JpegEncoderType::jpeg_compress_struct cinfo = {};
        JpegEncoderType::jpeg_workbuf workbuf = {pWork, static_cast<size_t>(pWorkEnd - pWork)};
        cinfo.workbuf = &workbuf;

        JpegErrorHandler errorHandler;
        std::memset(&errorHandler, 0, sizeof(errorHandler));
        cinfo.err = jpeg_std_error(&errorHandler);
        errorHandler.error_exit = JpegErrorHandler::HandleError;
        errorHandler.encoderResult = nn::ResultSuccess();

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
        if (setjmp(errorHandler.jmpContext) == 0)
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        {
            jpeg_create_compress(&cinfo);
            NN_UTIL_SCOPE_EXIT{
                jpeg_destroy_compress(&cinfo);
            };

            cinfo.dest = destMgr.Get();
            SetupEncodingParameterYuv(&cinfo, inputInfo, horizontalSubsamplingShift, verticalSubsamplingShift, quality, hasExif);

            jpeg_start_compress(&cinfo, true);
            uint32_t yStep = DCTSIZE << verticalSubsamplingShift;
            for(uint32_t y = 0; y < inputInfo.height; y += yStep)
            {
                NN_RESULT_DO(
                    inputInfo.pGetSubRegionFunction(
                        partialImage,
                        inputInfo.format,
                        0,                     // x
                        y,                     // y
                        inputInfo.width,       // width
                        yStep,                 // height
                        inputInfo.pGetSubRegionUserPtr
                    )
                );


                ProcessEncodeYuv(&cinfo, partialImage, inputInfo.width, horizontalSubsamplingShift, verticalSubsamplingShift);
            }
            jpeg_finish_compress(&cinfo);
        }
        else
        {
            NN_RESULT_THROW(errorHandler.encoderResult);
        }

        NN_RESULT_SUCCESS;
    }
}}}}
