﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SdkConditionVariable.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include "../capsrvServer_Config.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    enum OverlayThumbnailIndex
    {
        OverlayThumbnailIndex_ScreenShot = 0,
        OverlayThumbnailIndex_Movie = 1,
    };

    class OverlayThumbnailHolder
    {
    public:
        static const int ThumbnailCount = OverlayThumbnailImageCount;

    private:
        enum ThumbnailLockState
        {
            ThumbnailLockState_Unlocked,
            ThumbnailLockState_Locked,
        };

        struct Thumbnail
        {
            // 静止画サムネイル画像
            void*       m_ImageBuffer;
            // 静止画サムネイル画像の元画像のファイル ID
            AlbumFileId m_FileId;
            // ロック状態
            ThumbnailLockState m_LockState;
        };

    public:
        OverlayThumbnailHolder() NN_NOEXCEPT;

        static size_t GetThumbnailDataSize() NN_NOEXCEPT;
        static size_t GetRequiredMemorySize() NN_NOEXCEPT;
        static size_t GetRequiredMemoryAlignment() NN_NOEXCEPT;
        void SetMemory(void* memory, size_t size) NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        // @pre index >= 0 && index < ThumbnailCount
        // @details
        //   ロック中だった場合、ロック解除されるまで待つ
        nn::Result GetThumbnailImage(
            AlbumFileId* pOutAlbumFileId,
            size_t* pOutDataSize,
            void* pOutDataBuffer,
            size_t bufferSize,
            int index
            ) NN_NOEXCEPT;

        // @pre index >= 0 && index < ThumbnailCount
        // @post index 番目のサムネイルがロックされている
        // @details
        //   ロック中だった場合、ロック解除されるまで待つ
        void* BeginUpdateThumbnailImage(const AlbumFileId& fileId, int imdex) NN_NOEXCEPT;
        // @pre index >= 0 && index < ThumbnailCount
        // @pre index 番目のサムネイルがロックされている
        // @post index 番目のサムネイルがロックされていない
        void EndUpdateThumbnailImage(int index) NN_NOEXCEPT;

    private:
        // @pre 呼び出し元スレッドが LockStateMutex をロック中
        // @post e.m_LockState == Unlocked
        void WaitThumbnailUnlockedImpl(Thumbnail& e) NN_NOEXCEPT;

        // @pre 呼び出し元スレッドが LockStateMutex をロック中
        // @pre e.m_LockState == Unlocked
        // @post e.m_LockState == Locked
        void LockThumbnailImpl(Thumbnail& e) NN_NOEXCEPT;

        // @pre 呼び出し元スレッドが LockStateMutex をロック中
        // @pre e.m_LockState == Locked
        // @post e.m_LockState == Unlocked
        void UnlockThumbnailImpl(Thumbnail& e) NN_NOEXCEPT;


    private:
        bool m_IsInitialized;

        void*  m_Memory;
        size_t m_MemorySize;

        Thumbnail m_ThumbnailList[ThumbnailCount];

        nn::os::SdkMutexType m_LockStateMutex;
        nn::os::SdkConditionVariableType m_LockStateConditionVariable;
    };

}}}}
