﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "capsrvServer_MakerNoteEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename TInfoType, typename Tag>
    class MakerNoteParser
    {
    public:
        typedef TInfoType InfoType;

    private:
        static nn::Result VisitImpl(MakerNoteVersionType* pOutVersion, char* buf, size_t size, DecodeEntryParameter<InfoType, Tag>& param) NN_NOEXCEPT
        {
            return MakerNoteEntryVisitor<MakerNoteTag>::VisitEntries<
                MakerNoteEntryDecoder,
                InfoType,
                DecodeEntryParameter<InfoType, MakerNoteTag>,
                MakerNoteEntryParser<MakerNoteTag>
            >(pOutVersion, buf, size, param);
        }

        static nn::Result PostprocessImpl(InfoType& info, const DecodedCount<Tag>& counter, MakerNoteVersionType version) NN_NOEXCEPT
        {
            return MakerNoteEntryDecoderPostprocessor<InfoType, MakerNoteTag>::Invoke(info, counter, info.version);
        }

    public:

        static nn::Result Parse(
            InfoType* pOutValue,
            const void* pMakerNoteData,
            size_t makerNoteDataSize
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            NN_RESULT_THROW_UNLESS(pMakerNoteData != nullptr, ResultAlbumInvalidFileData());
            NN_RESULT_THROW_UNLESS(makerNoteDataSize <= MakerNoteSizeLimit, ResultAlbumInvalidFileData());

            char buf[MakerNoteSizeLimit] = {};
            std::memcpy(buf, pMakerNoteData, makerNoteDataSize);

            InfoType info = {};
            DecodedCount<Tag> counter = {};

            DecodeEntryParameter<InfoType, Tag> param;
            param.pInfo      = &info;
            param.pCounter   = &counter;

            NN_CAPSRV_LOG_MAKERNOTE("parsing MakerNote(size=%llu) ...\n", makerNoteDataSize);
            NN_RESULT_DO(VisitImpl(&info.version, buf, makerNoteDataSize, param));
            NN_CAPSRV_LOG_MAKERNOTE("-> success: version %d\n", static_cast<int>(info.version));

            NN_CAPSRV_LOG_MAKERNOTE("postprocessing\n");
            NN_RESULT_DO(PostprocessImpl(info, counter, info.version));
            NN_CAPSRV_LOG_MAKERNOTE("-> success\n");

            *pOutValue = info;
            NN_RESULT_SUCCESS;
        }

    };

}}}}
