﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/capsrv/capsrv_Result.h>

#include "../capsrvServer_Config.h"
#include "capsrvServer_MakerNoteValuePropertyUtility.h"
#include "capsrvServer_MakerNoteEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename Tag>
    struct DecodedCount
    {
        int list[MakerNoteEntryTag_Max];
    };

    template<typename TInfoType, typename Tag>
    struct DecodeEntryParameter
    {
        DecodedCount<Tag>* pCounter;
        TInfoType* pInfo;
    };

    template<typename TInfoType, typename VersionPolicy, typename Tag>
    class MakerNoteEntryDecoder
    {
    public:
        typedef TInfoType InfoType;
        typedef DecodedCount<Tag> DecodedCount;
        typedef DecodeEntryParameter<TInfoType, Tag> DecodeEntryParameter;

    private:
        // TryDecode 系関数共通
        // @param[out] pOutValue 解析結果を受け取る変数のポインタ
        // @param[in,out] count  解析したエントリーのカウンタ
        // @param[in] entry Parse 済のエントリー
        // @retval nn::ResultSuccess デコードに成功
        // @retval nn::capsrv::ResultAlbumInvalidFileData デコードに失敗
        // @details
        //   デコードに失敗した場合、出力バッファ／出力変数の値は変更されません。

        template<typename TProperty, MakerNoteValueClass TValueClass>
        struct EntryValueDecoder;

        template<typename TProperty>
        struct EntryValueDecoder<TProperty, MakerNoteValueClass_Bytes>
        {
            typedef typename TProperty::ValueType ValueType;

            static nn::Result Decode(ValueType* pOutValue, DecodedCount& counter, const MakerNoteEntry& entry) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TProperty::EntryTag < MakerNoteEntryTag_Max);
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);

                if (entry.size != sizeof(ValueType))
                {
                    NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) : bytes <= Invalid size\n", static_cast<int>(entry.tag), static_cast<int>(entry.size));
                    NN_RESULT_THROW( ResultAlbumInvalidFileData() );
                }

                std::memcpy(pOutValue, entry.body, sizeof(ValueType));
                counter.list[TProperty::EntryTag]++;
                if (entry.size == 0)
                {
                    NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) : bytes\n", static_cast<int>(entry.tag), static_cast<int>(entry.size));
                }
                else if (entry.size <= 16)
                {
                    const char* hex = "0123456789abcdef";
                    char buf[64];
                    auto* p = reinterpret_cast<const char*>(pOutValue);
                    NN_UNUSED(hex);
                    NN_UNUSED(buf);
                    NN_UNUSED(p);
                    for (uint16_t i=0; i<entry.size; ++i)
                    {
                        buf[i * 3 + 0] = (i == 0) ? '[' : ',';
                        buf[i * 3 + 1] = hex[(p[i] >> 4) & 0xf];
                        buf[i * 3 + 2] = hex[(p[i] >> 0) & 0xf];
                    }
                    buf[entry.size * 3] = ']';
                    buf[entry.size * 3 + 1] = '\0';
                    NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) : bytes %s\n", static_cast<int>(entry.tag), static_cast<int>(entry.size),buf);
                }
                else
                {
                    auto* p1 = reinterpret_cast<const char*>(pOutValue);
                    auto* p2 = reinterpret_cast<const char*>(pOutValue) + entry.size - 8;
                    NN_UNUSED(p1);
                    NN_UNUSED(p2);
                    NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) : bytes [%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,...,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x]\n", static_cast<int>(entry.tag), static_cast<int>(entry.size),
                    p1[0], p1[1], p1[2], p1[3], p1[4], p1[5], p1[6], p1[7],
                    p2[0], p2[1], p2[2], p2[3], p2[4], p2[5], p2[6], p2[7]);
                }
                NN_RESULT_SUCCESS;
            }
        };

        template<typename TProperty>
        struct EntryValueDecoder<TProperty, MakerNoteValueClass_Integer>
        {
            typedef typename TProperty::ValueType ValueType;

            static nn::Result Decode(ValueType* pOutValue, DecodedCount& counter, const MakerNoteEntry& entry) NN_NOEXCEPT
            {
                NN_STATIC_ASSERT(TProperty::EntryTag < MakerNoteEntryTag_Max);
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);

                NN_RESULT_THROW_UNLESS(
                    entry.size == sizeof(ValueType),
                    ResultAlbumInvalidFileData()
                );

                ValueType tmp;
                std::memcpy(&tmp, entry.body, sizeof(ValueType));
                *pOutValue = nn::util::LoadLittleEndian(&tmp);
                counter.list[TProperty::EntryTag]++;
                NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes): integer %lld\n", static_cast<int>(entry.tag), static_cast<int>(entry.size), static_cast<int64_t>(*pOutValue));
                NN_RESULT_SUCCESS;
            }
        };

        template<typename TProperty>
        static nn::Result DecodeEntryValueImpl(InfoType& outInfo, DecodedCount& count, const MakerNoteEntry& entry) NN_NOEXCEPT
        {
            // 今の実装では同一 tag は高々 1 つまでしかサポートしていない
            // 重複が必要なら VersionedMakerNoteEntryVisitor::VisitEntries() の修正も必要。
            NN_SDK_ASSERT(TProperty::GetEntryCountMax(VersionPolicy::Version) <= 1);

            if(TProperty::GetEntryCountMax(VersionPolicy::Version) <= 0)
            {
                if(VersionPolicy::IsErrorOnUnknownEntryEnabled)
                {
                    NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) not supported\n", static_cast<int>(entry.tag), static_cast<int>(entry.size));
                    NN_RESULT_THROW(ResultAlbumInvalidFileData());
                }
                else
                {
                    NN_CAPSRV_LOG_MAKERNOTE("    skipping unsupported tag %d (%d bytes)\n", static_cast<int>(entry.tag), static_cast<int>(entry.size));
                    NN_RESULT_SUCCESS;
                }
            }

            // 値の種類に応じて解釈
            return EntryValueDecoder<TProperty, TProperty::ValueClass>::Decode(TProperty::GetPointer(outInfo), count, entry);
        }

    public:
        static nn::Result Invoke(const MakerNoteEntry* pEntry, DecodeEntryParameter& param) NN_NOEXCEPT
        {
            InfoType& info   = *param.pInfo;
            DecodedCount& counter = *param.pCounter;

            switch(pEntry->tag)
            {
            #define NN_CAPSRV_DETAIL_CASE_ENTRYTAG(PropertyType)    \
            case PropertyType::EntryTag: \
                NN_RESULT_DO(DecodeEntryValueImpl<PropertyType>(info, counter, *pEntry)); \
                break;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEPROPERTY( NN_CAPSRV_DETAIL_CASE_ENTRYTAG )

            #undef NN_CAPSRV_DETAIL_CASE_ENTRYTAG
            default:
                {
                    if(VersionPolicy::IsErrorOnUnknownEntryEnabled)
                    {
                        NN_CAPSRV_LOG_MAKERNOTE("    tag %d (%d bytes) is unknown\n", static_cast<int>(pEntry->tag), static_cast<int>(pEntry->size));
                        NN_RESULT_THROW(ResultAlbumInvalidFileData());
                    }
                    else
                    {
                        NN_CAPSRV_LOG_MAKERNOTE("    skipping unknown tag %d (%d bytes)\n", static_cast<int>(pEntry->tag), static_cast<int>(pEntry->size));
                    }
                }
            }

            NN_RESULT_SUCCESS;
        }
    };

    template<typename TInfoType, typename Tag>
    class MakerNoteEntryDecoderPostprocessor
    {
    public:
        typedef TInfoType InfoType;
        typedef DecodedCount<Tag> DecodedCount;

    private:
        // 出現回数が正しいかの検証
        template<typename VersionPolicy, typename TProperty>
        static nn::Result CheckCountVersionPropertyImpl(const DecodedCount& counter) NN_NOEXCEPT
        {
            int count = counter.list[TProperty::EntryTag];
            NN_RESULT_THROW_UNLESS(
                count >= TProperty::GetEntryCountMin(VersionPolicy::Version) && count <= TProperty::GetEntryCountMax(VersionPolicy::Version),
                ResultAlbumInvalidFileData()
            );
            NN_RESULT_SUCCESS;
        }

        template<typename VersionPolicy>
        static nn::Result CheckCountVersionImpl(const DecodedCount& counts) NN_NOEXCEPT
        {
            // 全プロパティをチェック
        #define NN_CAPSRV_DETAIL_CHECK_COUNTIMPL(PropertyType)  \
            {                                                   \
                auto func = CheckCountVersionPropertyImpl<VersionPolicy, PropertyType>; \
                NN_RESULT_DO(func(counts));                     \
            }

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEPROPERTY( NN_CAPSRV_DETAIL_CHECK_COUNTIMPL );

        #undef NN_CAPSRV_DETAIL_CHECKCOUNT_PROPERTY

            NN_RESULT_SUCCESS;
        }

        static nn::Result CheckCountImpl(const DecodedCount& counts, MakerNoteVersionType version) NN_NOEXCEPT
        {
            // バージョンで分岐
            switch(version)
            {
        #define NN_CAPSRV_DETAIL_CASE_VERSION(policy)    \
            case policy::Version:                                 \
                return CheckCountVersionImpl<policy>(counts);

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_DETAIL_CASE_VERSION)

        #undef NN_CAPSRV_DETAIL_CASE_VERSION
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        // 既定値で埋める
        template<typename VersionPolicy, typename TProperty>
        static void FillWithDefaultValueVersionPropertyImpl(InfoType& info, const DecodedCount& counter) NN_NOEXCEPT
        {
            // カウントが 0 ならデフォルト値で埋める
            if(counter.list[TProperty::EntryTag] == 0)
            {
                TProperty::SetDefault(TProperty::GetPointer(info), VersionPolicy::Version);
                NN_CAPSRV_LOG_MAKERNOTE("  tag %d (%d bytes): default\n", TProperty::EntryTag, sizeof(typename TProperty::ValueType));
            }
        }

        template<typename VersionPolicy>
        static void FillWithDefaultValueVersionImpl(InfoType& info, const DecodedCount& counter) NN_NOEXCEPT
        {
            // 全プロパティを処理
        #define NN_CAPSRV_DETAIL_FILL_DEFAULTVALUEIMPL(PropertyType)    \
            FillWithDefaultValueVersionPropertyImpl<VersionPolicy, PropertyType>(info, counter);

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEPROPERTY( NN_CAPSRV_DETAIL_FILL_DEFAULTVALUEIMPL );

        #undef NN_CAPSRV_DETAIL_FILL_DEFAULTVALUEIMPL
        }

        static void FillWithDefaultValueImpl(InfoType& info, const DecodedCount& counter, MakerNoteVersionType version) NN_NOEXCEPT
        {
            // バージョンで分岐
            switch(version)
            {
        #define NN_CAPSRV_DETAIL_CASE_VERSION(policy)    \
            case policy::Version:                                 \
                return FillWithDefaultValueVersionImpl<policy>(info, counter);

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_DETAIL_CASE_VERSION)

        #undef NN_CAPSRV_DETAIL_CASE_VERSION
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    public:

        static nn::Result Invoke(InfoType& info, const DecodedCount& counter, MakerNoteVersionType version) NN_NOEXCEPT
        {

            NN_CAPSRV_LOG_MAKERNOTE("  checking entry counts\n");
            NN_RESULT_DO(CheckCountImpl(counter, version));

            NN_CAPSRV_LOG_MAKERNOTE("  filling unspecified entries by default\n");
            FillWithDefaultValueImpl(info, counter, version);

            NN_RESULT_SUCCESS;
        }

    };


}}}}
