﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename BaseQueue>
    class LambdaFunctionInvokationRequestQueue
        : public BaseQueue
    {
    public:
        template<typename F>
        struct SyncFunctionParameter
        {
            F* pFunction;
            nn::Result result;
        };

    public:
        // @brief 関数をキューに入れます。実行完了を待ちます。
        // @param[in] function 0 個の引数を取り nn::Result を返す関数オブジェクト
        template<typename F>
        nn::Result EnqueueSync(F function) NN_NOEXCEPT
        {
            SyncFunctionParameter<F> param = { &function };
            nn::os::EventType completeEvent = {};
            nn::os::InitializeEvent(&completeEvent, false, nn::os::EventClearMode_ManualClear);
            NN_UTIL_SCOPE_EXIT { nn::os::FinalizeEvent(&completeEvent); };
            NN_RESULT_DO(
                BaseQueue::EnqueueReference(InvokeFunctionSync<F>, &param, &completeEvent)
            );
            nn::os::WaitEvent(&completeEvent);
            return param.result;
        }

        // @brief 関数をキューに入れます。実行完了を待たずに返ります。
        // @param[in] function 0 個の引数を取る関数オブジェクト。戻り値は捨てられる。
        // @retval ResultInternalRequestQueueAllocationFailed
        template<typename F>
        nn::Result EnqueueAsync(F function) NN_NOEXCEPT
        {
            return BaseQueue::EnqueueCopy(InvokeFunctionAsync<F>, &function, sizeof(F), NN_ALIGNOF(F), nullptr);
        }

        // @brief 関数をキューに入れます。実行完了を待たずに返ります。
        // @param[in] function 0 個の引数を取る関数オブジェクト。戻り値は捨てられる。
        // @retval ResultInternalRequestQueueIsFull
        // @retval ResultInternalRequestQueueAllocationFailed
        template<typename F>
        nn::Result TryEnqueueAsync(F function) NN_NOEXCEPT
        {
            return BaseQueue::TryEnqueueCopy(InvokeFunctionAsync<F>, &function, sizeof(F), NN_ALIGNOF(F), nullptr);
        }

    private:
        template<typename F>
        static void InvokeFunctionSync(void* pArg) NN_NOEXCEPT
        {
            auto pParam = reinterpret_cast<SyncFunctionParameter<F>*>(pArg);
            pParam->result = (*pParam->pFunction)();
        }

        template<typename F>
        static void InvokeFunctionAsync(void* pParam) NN_NOEXCEPT
        {
            F* pFunction = reinterpret_cast<F*>(pParam);
            (*pFunction)();
        }
    };

}}}}
