﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_FunctionInvokationRequestQueue.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    const int MaxSemaphoreCount = 1;

    void FunctionInvokationRequestQueue1::Initialize(
        uintptr_t dequeueWaitHolderTag,
        void* pWorkMemory,
        size_t workMemorySize
        ) NN_NOEXCEPT
    {
        nn::os::InitializeSemaphore(&m_RequesterSemaphore, MaxSemaphoreCount, MaxSemaphoreCount);
        nn::os::InitializeSemaphore(&m_RequesteeSemaphore, 0, MaxSemaphoreCount);
        nn::os::InitializeMultiWaitHolder(&m_RequesteeSemaphoreWaitHolder, &m_RequesteeSemaphore);
        nn::os::SetMultiWaitHolderUserData(&m_RequesteeSemaphoreWaitHolder, dequeueWaitHolderTag);

        m_pParameterHeapMemory = pWorkMemory;
        m_ParameterHeapSize = workMemorySize;
        m_ParameterHeapHandle = nn::lmem::CreateExpHeap(pWorkMemory, workMemorySize, nn::lmem::CreationOption_ThreadSafe);

        m_pRequestedFunction  = nullptr;
        m_pRequestedParameter = nullptr;
        m_IsRequestParameterCopy = false;
        m_pRequestedCompleteEvent = nullptr;

        m_IsRequestProcessing = false;
    }

    void FunctionInvokationRequestQueue1::Finalize() NN_NOEXCEPT
    {
        nn::lmem::DestroyExpHeap(m_ParameterHeapHandle);
        nn::os::FinalizeMultiWaitHolder(&m_RequesteeSemaphoreWaitHolder);
        nn::os::FinalizeSemaphore(&m_RequesteeSemaphore);
        nn::os::FinalizeSemaphore(&m_RequesterSemaphore);
    }

    nn::os::MultiWaitHolderType* FunctionInvokationRequestQueue1::GetDequeueMultiWaitHolder() NN_NOEXCEPT
    {
        return & m_RequesteeSemaphoreWaitHolder;
    }

    // Requester 側：
    //   1. Requester 用セマフォを Acquire
    //   2. パラメータをセット
    //   3. Requestee 用セマフォを Release
    // Server 側：
    //   1. Requestee 用セマフォを Acquire
    //   2. パラメータを受け取る
    //   3. Requester 用セマフォを Release
    //   4. CompleteEvent をシグナル

    nn::Result FunctionInvokationRequestQueue1::TryEnqueueReference(
        void (*pFunction)(void*),
        void* pParam,
        nn::os::EventType* pCompleteEvent
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            nn::os::TryAcquireSemaphore(&m_RequesterSemaphore),
            ResultInternalRequestQueueIsFull()
        );

        m_pRequestedFunction  = pFunction;
        m_pRequestedParameter = pParam;
        m_IsRequestParameterCopy = false;
        m_pRequestedCompleteEvent = pCompleteEvent;
        nn::os::ReleaseSemaphore(&m_RequesteeSemaphore);

        NN_RESULT_SUCCESS;
    }

    nn::Result FunctionInvokationRequestQueue1::EnqueueReference(
        void (*pFunction)(void*),
        void* pParam,
        nn::os::EventType* pCompleteEvent
        ) NN_NOEXCEPT
    {
        nn::os::AcquireSemaphore(&m_RequesterSemaphore),
        m_pRequestedFunction  = pFunction;
        m_pRequestedParameter = pParam;
        m_IsRequestParameterCopy = false;
        m_pRequestedCompleteEvent = pCompleteEvent;
        nn::os::ReleaseSemaphore(&m_RequesteeSemaphore);

        NN_RESULT_SUCCESS;
    }

    nn::Result FunctionInvokationRequestQueue1::TryEnqueueCopy(
        void (*pFunction)(void*),
        void* pParam,
        size_t paramSize,
        size_t paramAlignment,
        nn::os::EventType* pCompleteEvent
        ) NN_NOEXCEPT
    {
        bool isSuccess = false;
        {
            NN_RESULT_THROW_UNLESS(
                nn::os::TryAcquireSemaphore(&m_RequesterSemaphore),
                ResultInternalRequestQueueIsFull()
            );
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    nn::os::ReleaseSemaphore(&m_RequesterSemaphore);
                }
            };

            void* pInternalParam = nullptr;
            if(pParam != nullptr)
            {
                pInternalParam = nn::lmem::AllocateFromExpHeap(m_ParameterHeapHandle, paramSize, static_cast<int>(paramAlignment));
                NN_RESULT_THROW_UNLESS(
                    pInternalParam != nullptr,
                    ResultInternalRequestQueueAllocationFailed()
                );
                std::memcpy(pInternalParam, pParam, paramSize);
            }
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess && pInternalParam != nullptr)
                {
                    nn::lmem::FreeToExpHeap(m_ParameterHeapHandle, pInternalParam);
                }
            };

            m_pRequestedFunction  = pFunction;
            m_pRequestedParameter = pInternalParam;
            m_IsRequestParameterCopy = true;
            m_pRequestedCompleteEvent = pCompleteEvent;
            nn::os::ReleaseSemaphore(&m_RequesteeSemaphore);
            isSuccess = true;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result FunctionInvokationRequestQueue1::EnqueueCopy(
        void (*pFunction)(void*),
        void* pParam,
        size_t paramSize,
        size_t paramAlignment,
        nn::os::EventType* pCompleteEvent
        ) NN_NOEXCEPT
    {
        bool isSuccess = false;
        {
            nn::os::AcquireSemaphore(&m_RequesterSemaphore);
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    nn::os::ReleaseSemaphore(&m_RequesterSemaphore);
                }
            };

            void* pInternalParam = nullptr;
            if(pParam != nullptr)
            {
                pInternalParam = nn::lmem::AllocateFromExpHeap(m_ParameterHeapHandle, paramSize, static_cast<int>(paramAlignment));
                NN_RESULT_THROW_UNLESS(
                    pInternalParam != nullptr,
                    ResultInternalRequestQueueAllocationFailed()
                );
                std::memcpy(pInternalParam, pParam, paramSize);
            }
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess && pInternalParam != nullptr)
                {
                    nn::lmem::FreeToExpHeap(m_ParameterHeapHandle, pInternalParam);
                }
            };

            m_pRequestedFunction  = pFunction;
            m_pRequestedParameter = pInternalParam;
            m_IsRequestParameterCopy = true;
            m_pRequestedCompleteEvent = pCompleteEvent;
            nn::os::ReleaseSemaphore(&m_RequesteeSemaphore);
            isSuccess = true;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result FunctionInvokationRequestQueue1::BeginProcessingRequest(void (**pOutFunction)(void*), void** pParam) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsRequestProcessing);

        NN_RESULT_THROW_UNLESS(
            nn::os::TryAcquireSemaphore(&m_RequesteeSemaphore),
            ResultInternalRequestQueueIsEmpty()
        );

        *pOutFunction = m_pRequestedFunction;
        *pParam       = m_pRequestedParameter;
        m_IsRequestProcessing = true;

        NN_RESULT_SUCCESS;
    }

    void FunctionInvokationRequestQueue1::EndProcessingRequest() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsRequestProcessing);

        if(m_IsRequestParameterCopy)
        {
            if(m_pRequestedParameter != nullptr)
            {
                nn::lmem::FreeToExpHeap(m_ParameterHeapHandle, m_pRequestedParameter);
            }
        }

        auto pCompleteEvent = m_pRequestedCompleteEvent;

        m_pRequestedFunction  = nullptr;
        m_pRequestedParameter = nullptr;
        m_IsRequestParameterCopy = false;
        m_pRequestedCompleteEvent = nullptr;

        m_IsRequestProcessing = false;
        nn::os::ReleaseSemaphore(&m_RequesterSemaphore);

        if(pCompleteEvent)
        {
            nn::os::SignalEvent(pCompleteEvent);
        }
    }

}}}}
