﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_EncryptApplicationId.h"

#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Endian.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/capsrv/capsrv_Result.h>

#include <cstring>

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {
        // TORIAEZU: 鍵をここに書いておく。
        const uint8_t ApplicationIdEncryptionKey[nn::crypto::AesEncryptor128::BlockSize] =
            { 0xB7, 0xED, 0x7A, 0x66, 0xC8, 0x0B, 0x4B, 0x00,
              0x8B, 0xAF, 0x7F, 0x05, 0x89, 0xC0, 0x82, 0x24 };

        // Scoop 準拠の場合は全て 0 となる
        union AlbumFileParameter
        {
            struct
            {
                uint8_t isExtra;
                uint8_t _padding[7];
            };
            // チェック用
            uint8_t valueUint8[8];
            uint64_t valueUint64;
        };
        NN_STATIC_ASSERT(sizeof(AlbumFileParameter) == sizeof(uint64_t));
    }

    EncryptedApplicationId EncryptApplicationId(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        // 純粋に applicationId だけを暗号化する
        return EncryptApplicationId(applicationId, false);
    }

    EncryptedApplicationId EncryptApplicationId(nn::ncm::ApplicationId applicationId, bool isExtra) NN_NOEXCEPT
    {
        nn::crypto::AesEncryptor128 encryptor;
        uint64_t inputBuffer[2] = {};
        EncryptedApplicationId encrypted = {};

        NN_STATIC_ASSERT(encryptor.BlockSize == sizeof(inputBuffer));
        NN_STATIC_ASSERT(encryptor.BlockSize == sizeof(encrypted.value));

        // little endian で書き込み。前半は ApplicationId
        nn::util::StoreLittleEndian(&inputBuffer[0], static_cast<uint64_t>(applicationId.value));

        // 後半は AlbumFileParameter。Scoop 準拠ならオールゼロ
        AlbumFileParameter param = {};
        param.isExtra = isExtra ? 1 : 0;
        std::memcpy(&inputBuffer[1], &param, sizeof(inputBuffer[1]));

        encryptor.Initialize(ApplicationIdEncryptionKey, sizeof(ApplicationIdEncryptionKey));
        encryptor.EncryptBlock(&encrypted.value, sizeof(encrypted.value), inputBuffer, sizeof(inputBuffer));

        return encrypted;
    }

    nn::Result TryDecryptApplicationId(nn::ncm::ApplicationId* pOutValue, bool* pOutIsExtra, const EncryptedApplicationId& encryptedApplicationId) NN_NOEXCEPT
    {
        nn::crypto::AesDecryptor128 decryptor;
        nn::Bit64 outBuffer[2] = {};

        NN_STATIC_ASSERT(decryptor.BlockSize == sizeof(encryptedApplicationId.value));
        NN_STATIC_ASSERT(decryptor.BlockSize == sizeof(outBuffer));

        decryptor.Initialize(ApplicationIdEncryptionKey, sizeof(ApplicationIdEncryptionKey));
        decryptor.DecryptBlock(outBuffer, sizeof(outBuffer), encryptedApplicationId.value, sizeof(encryptedApplicationId.value));

        // 後半部分を AlbumFileParameter 構造体へ変換
        AlbumFileParameter param = {};
        std::memcpy(&param, &outBuffer[1], sizeof(param));

        // 復号後の param.isExtra のチェック
        NN_RESULT_THROW_UNLESS(
            param.valueUint8[0] == static_cast<uint8_t>(0) ||
            param.valueUint8[0] == static_cast<uint8_t>(1),
            ResultAlbumInvalidApplicationId()
        );

        // _padding[] の部分が 0 になっていることを確認
        NN_RESULT_THROW_UNLESS(
            (param.valueUint64 & 0xFFFFFFFFFFFFFF00ull) == 0,
            ResultAlbumInvalidApplicationId()
        );

        *pOutIsExtra = param.isExtra == 1 ? true : false;

        // little endian で読み込み
        pOutValue->value = nn::util::LoadLittleEndian(&outBuffer[0]);

        NN_RESULT_SUCCESS;
    }

}}}}
