﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "decodersrv_DecoderControlServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

#include "../../capsrv_Macro.h"
#include "../capsrvServer_Config.h"
#include "decodersrv_DecoderServerObject.h"

#include "../jpeg/decodersrv_SoftwareJpegDecoderRgba8.h"

//#define NN_CAPSRV_SERVER_DO(Function)  \
//    NN_RESULT_DO(Function());

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(Function());

//#define NN_CAPSRV_SERVER_ASYNC_DO(Function) \
//    NN_RESULT_DO(Function());

namespace nn{ namespace capsrv{ namespace server{

    namespace {

        nn::Result DecodeJpegImpl(
            void* pOutImageBuf,
            size_t outImageBufSize,
            const void* pInJpegBuf,
            size_t inJpegBufSize,
            uint32_t width,
            uint32_t height,
            const nn::capsrv::ScreenShotDecodeOption& option,
            void* pWorkMemory,
            size_t workMemorySize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_PROCESS_START();
            std::memset(pWorkMemory, 0, workMemorySize);
            NN_UTIL_SCOPE_EXIT{ std::memset(pWorkMemory, 0, workMemorySize); };

            std::memset(pOutImageBuf, 0, outImageBufSize);
            NN_CAPSRV_PROCESS_ROLLBACK(std::memset(pOutImageBuf, 0, outImageBufSize));

            NN_RESULT_THROW_UNLESS(width % 16 == 0, ResultAlbumOutOfRange());
            NN_RESULT_THROW_UNLESS(height % 4 == 0, ResultAlbumOutOfRange());
            NN_RESULT_THROW_UNLESS(pOutImageBuf != nullptr, ResultAlbumReadBufferShortage());
            NN_RESULT_THROW_UNLESS(outImageBufSize >= 4 * width * height, ResultAlbumReadBufferShortage());
            NN_RESULT_THROW_UNLESS(pInJpegBuf != nullptr, ResultAlbumInvalidFileData());
            NN_RESULT_THROW_UNLESS(inJpegBufSize > 0, ResultAlbumInvalidFileData());

            jpeg::SoftwareJpegDecoderBufferInputInfo inputInfo = {};
            inputInfo.width = width;
            inputInfo.height = height;
            inputInfo.pBuffer = pInJpegBuf;
            inputInfo.bufferSize = inJpegBufSize;
            inputInfo.isFancyUpsamplingEnabled = option.HasJpegDecoderFlag(ScreenShotJpegDecoderFlag_EnableFancyUpsampling);
            inputInfo.isBlockSmoothingEnabled  = option.HasJpegDecoderFlag(ScreenShotJpegDecoderFlag_EnableBlockSmoothing);

            int outWidth = 0;
            int outHeight = 0;
            jpeg::SoftwareJpegDecoderBufferOutputInfo outputInfo = {};
            outputInfo.pOutWidth = &outWidth;
            outputInfo.pOutHeight = &outHeight;
            outputInfo.pBuffer = pOutImageBuf;
            outputInfo.bufferSize = outImageBufSize;

            NN_RESULT_DO(jpeg::SoftwareJpegDecoderRgba8::DecodeRgba8BufferFromBuffer(outputInfo, inputInfo, pWorkMemory, workMemorySize));

            NN_CAPSRV_PROCESS_SUCCESS();
            NN_RESULT_SUCCESS;
        }

    }

    nn::Result DecoderControlServiceImpl::DecodeJpeg(
        const nn::sf::OutBuffer& outImageBuf,
        const nn::sf::InBuffer& inJpegBuf,
        std::uint32_t width,
        std::uint32_t height,
        const nn::capsrv::ScreenShotDecodeOption& option
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_DC(DecodeJpeg);

        void* pWorkMemory = g_WorkMemory.jpegDecoderWorkMemory;
        size_t workMemorySize = sizeof(g_WorkMemory.jpegDecoderWorkMemory);

        NN_CAPSRV_SERVER_SYNC_DO([&]() -> nn::Result{
            return DecodeJpegImpl(
                outImageBuf.GetPointerUnsafe(),
                outImageBuf.GetSize(),
                inJpegBuf.GetPointerUnsafe(),
                inJpegBuf.GetSize(),
                width,
                height,
                option,
                pWorkMemory,
                workMemorySize
            );
        });

        NN_RESULT_SUCCESS;
    }

}}}
