﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/os/os_SdkThread.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/movie/capsrv_MovieConfig.h>

namespace nn{ namespace capsrv{ namespace server{

    static const int ScreenShotWidth = 1280;
    static const int ScreenShotHeight = 720;

    static const int MovieWidth  = 1280;
    static const int MovieHeight = 720;

    static const int RawImageChannelCount = 4;
    static const int ScreenShotTimeoutMilliseconds = 100;


    // 保存するスクリーンショット 1 枚あたりの容量制限
    static const int ScreenShotImageSizeLimit = 500 * 1024;

    // スクリーンショット保存時の書込み単位
    static const size_t ScreenShotDataBlockSize = 256 * 1024;

    // 保存スクリーンショットの Exif 情報を生成するために使用するワークメモリの大きさ
    static const int ScreenShotExifWorkSize = 6 * 1024;

    // 保存スクリーンショットの Exif 情報を生成するために使用するワークメモリのアライメント
    static const int ScreenShotExifWorkAlignment = NN_ALIGNOF(std::max_align_t);

    // MakerNote の大きさの上限
    static const size_t MakerNoteSizeLimit = 4096;

    NN_STATIC_ASSERT(ScreenShotExifWorkSize > MakerNoteSizeLimit); // nn::image::ExifBuilder 分も必要なのである程度の余裕が必要

    // 動画 1 本あたりの容量制限
    static const int64_t MovieSizeLimit = 0x80000000ll;

    // 動画のデータセクションのブロック数の上限
    static const int64_t MovieDataBlockCountLimit = 8192;

    // 動画のメタデータの大きさ
    static const size_t MovieMetaDataSize = 512 * 1024;

    // 動画のメタデータに含まれる静止画部分の大きさ
    static const size_t MovieMetaImageDataSize = ScreenShotImageSizeLimit; // 500 * 1024

    // 動画のメタデータに含まれる管理情報の大きさ
    static const size_t MovieMetaInfoDataSize = MovieMetaDataSize - MovieMetaImageDataSize; // 12 * 1024

    // 動画のデータブロックサイズ
    static const int64_t MovieDataBlockSize = 256 * 1024;

    // 動画のデータブロックごとのハッシュのサイズ(SHA-256)
    static const size_t MovieDataBlockHashSize = 32;

    // 動画の全体の署名サイズ
    static const size_t MovieSignatureSize = 32;

    // 動画書き出し時の初期ファイルサイズ。
    // MovieFileUnitSize の整数倍にすること。
    // SIGLO-68867:
    //   ファイルの断片化を抑えるため大きめに作っておく。
    //   書込み完了後に書き込んだ大きさに縮小する。
    static const int64_t MovieFileInitialSize = 24 * 1024 * 1024;

    // 動画書き出し時のハッシュファイルの初期サイズ
    // MovieHashFileUnitSize の整数倍にすること。
    // SIGLO-68867: MovieFileUnitSize に同じ。
    static const int64_t MovieHashFileInitialSize = MovieFileInitialSize / MovieDataBlockSize * MovieDataBlockHashSize;

    // 動画書き出し時の単位ファイルサイズ。ファイルサイズが不足した場合このサイズずつ伸長される。
    // MovieDataBlockSize の整数倍にすること。
    // SIGLO-68867: MovieFileInitialSize に同じ。
    static const int64_t MovieFileUnitSize = 4 * 1024 * 1024;

    // 動画書き出し時のハッシュファイルの単位サイズ。ファイルサイズが不足した場合このサイズずつ伸長される。
    // SIGLO-68867: MovieFileUnitSize に同じ。
    static const int64_t MovieHashFileUnitSize = MovieFileUnitSize / MovieDataBlockSize * MovieDataBlockHashSize;

    NN_STATIC_ASSERT(nn::capsrv::server::MovieMetaImageDataSize == nn::capsrv::movie::MovieMetaImageDataSize);
    NN_STATIC_ASSERT(nn::capsrv::server::MovieMetaInfoDataSize == nn::capsrv::movie::MovieMetaInfoDataSize);
    NN_STATIC_ASSERT(nn::capsrv::server::MovieDataBlockSize == nn::capsrv::movie::MovieDataChunkUnitSize);
    NN_STATIC_ASSERT(MovieSizeLimit / MovieDataBlockSize <= MovieDataBlockCountLimit);
    NN_STATIC_ASSERT(MovieMetaDataSize % MovieDataBlockSize == 0);
    NN_STATIC_ASSERT(MovieFileUnitSize % MovieDataBlockSize == 0);
    NN_STATIC_ASSERT(MovieFileInitialSize % MovieFileUnitSize == 0);
    NN_STATIC_ASSERT(MovieHashFileInitialSize % MovieHashFileUnitSize == 0);

    // オーバーレイ用サムネイルの枚数
    static const int OverlayThumbnailImageCount = 2;

    // ファイルリスト読み込み時のファイルリストのバッファのエントリー数（エントリー数）
    static const int AlbumManagerDirectoryEntryBufferLength = 300;

    // サムネイルを先読みするサイズ。最大 65535 バイトな気もするが安全のため 1 ページ多めにしておく
    static const int AlbumManagerThumbnailLoadSize = 68 * 1024;

    // 動画のストリーム数の最大数
    static const int MovieReadStreamCountMax = 4;
    static const int MovieWriteStreamCountMax = 2;

    // ファイルロックのテーブルの最大数
    // 最大数ファイルを同時オープンしても足りるように設定する。
    static const int FileLockTableSize = 10;

    NN_STATIC_ASSERT(FileLockTableSize >= MovieReadStreamCountMax + MovieWriteStreamCountMax);

    // テンポラリファイルの最大数
    static const int TemporaryFileCountMax = 32;

    NN_STATIC_ASSERT(TemporaryFileCountMax >= MovieReadStreamCountMax * 2 + MovieWriteStreamCountMax);

    // テンポラリファイル作成時にリトライする回数の上限
    static const int TemporaryFileNameGenerationRetryCountMax = 100;

//----------------------------------------
// ワークバッファ
//----------------------------------------

    // AlbumManager に割り当てるワークバッファサイズ
    static const size_t AlbumManagerWorkBufferSize = 324 * 1024; // >= 324KB

    // 動画の検証用のワークバッファ
    static const size_t AlbumManagerMovieValidationWorkBufferSize = MovieDataBlockSize;

    // ファイルコピー時のワークバッファ（バイト）
    static const int AlbumManagerFileCopyWorkBufferSize = AlbumManagerWorkBufferSize;

    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= ScreenShotDataBlockSize);
    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= AlbumManagerMovieValidationWorkBufferSize);
    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= AlbumManagerDirectoryEntryBufferLength * sizeof(nn::fs::DirectoryEntry));
    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= AlbumManagerFileCopyWorkBufferSize);
    // LoadScreenShotImage()
    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= AlbumManagerThumbnailLoadSize);
    // LoadScreenShotThumbnailImage()
    NN_STATIC_ASSERT(AlbumManagerWorkBufferSize >= AlbumManagerMovieValidationWorkBufferSize + AlbumManagerThumbnailLoadSize);

    // Jpeg デコーダのワークバッファ
    static const size_t SoftwareJpegDecoderWorkMemorySize = 128 * 1024;

    // Jpeg エンコーダのワークバッファ
    static const size_t SoftwareJpegEncoderWorkMemorySize = 100 * 1024;
    static const nn::image::JpegSamplingRatio SoftwareJpegEncoderSamplingRatio = nn::image::JpegSamplingRatio_422;
    static const int SoftwareJpegEncoderLineAlignment = 4;


//----------------------------------------
// ストレージ
//----------------------------------------

    // ストレージの数
    static const int AlbumStorageCount = 2;

    // 各ストレージのマウント名
    static const int AlbumMountNameLength = 2;

#define NN_CAPSRV_MOUNT_NAME_NAND "NA"
#define NN_CAPSRV_MOUNT_NAME_SD   "SD"

// TORIAEZU: NAND の Album/ 以下を中間ファイル置き場に使う
#define NN_CAPSRV_MOUNT_NAME_TEMP "TM"

//----------------------------------------
// コンテンツ
//----------------------------------------

    // コンテンツの種類の数
    static const int FileContentsCount = 4;
    NN_STATIC_ASSERT(FileContentsCount == nn::capsrv::AlbumFileContentsCount);

    static const AlbumFileContentsType AlbumFileContents_Directory = 0xFF;
    NN_STATIC_ASSERT(AlbumFileContents_Directory > FileContentsCount); // == だと 'Unknown' 扱いの可能性があるので >。

    // 各コンテンツの拡張子
#define NN_CAPSRV_CONTENTS_EXTENSION_SCREENSHOT ".jpg"
#define NN_CAPSRV_CONTENTS_EXTENSION_MOVIE      ".mp4"

//----------------------------------------
// ファイル数制限
//----------------------------------------

    // EnvironmentInfo に設定される初期値
    static const int64_t DefaultNandScreenShotFileCountLimit =  1000;
    static const int64_t DefaultNandMovieFileCountLimit      =   100;
    static const int64_t DefaultSdScreenShotFileCountLimit   = 10000;
    static const int64_t DefaultSdMovieFileCountLimit        =  1000;

//----------------------------------------
// 撮影対象
//----------------------------------------

    static const int LayerStackNameSize = 64;

    // スクリーンショット用のレイヤースタックの既定値
    static const nn::vi::LayerStack DefaultScreenShotLayerStack = nn::vi::LayerStack_Screenshot;
    // 遷移レイヤ用のレイヤースタックの既定値
    static const nn::vi::LayerStack DefaultLastFrameLayerStack = nn::vi::LayerStack_LastFrame;

//----------------------------------------
// アプリケーション管理
//----------------------------------------

    // リソース管理を行うアプリケーションの数の最大値
    static const int ApplicationCountMax = 4;

//----------------------------------------
// ファイルの属性のデフォルト値
//----------------------------------------

    class FileAttributeDefault
    {
    public:
        static const uint32_t ScreenShotFrameCount = 1;
        static const uint32_t ScreenShotFrameRateNumerator = 0;
        static const uint32_t ScreenShotFrameRateDenominator = 0;
        static const uint32_t ScreenShotDataDurationMilliseconds = 0;
        static const uint32_t ScreenShotKeyFrameInterval = 0;
    };

}}}

//----------------------------------------
// デバッグ用
//----------------------------------------

// オーバーレイ用の動画サムネイルを作成した際に「静止画撮影完了通知」を表示させる
// （オーバーレイ用のサムネイルが正しく作れているか確認するため無理矢理表示させる。）
//#define NN_CAPSRV_DEBUG_SEND_SCREENSHOT_OVLN_IN_MOVIE_THUMB_CREATION

// 動画のサムネイル用画像を静止画として保存する
//#define NN_CAPSRV_DEBUG_SAVE_MOVIE_META_DATA_AS_SCREENSHOT

// 自動テスト用の署名を受け付ける
#ifndef NN_SDK_BUILD_RELEASE
#define NN_CAPSRV_DEBUG_ENABLE_FILE_SIGNATURE_FOR_UNIT_TEST
#endif


//----------------------------------------
// 計測
//----------------------------------------

// スクリーンショット撮影の時間計測を有効にするか
//#define NN_CAPSRV_ENABLE_TIME_MEASURE

// IPC の計測を有効にするか
//#define NN_CAPSRV_ENABLE_IPC_TRACE

//----------------------------------------
// ログ
//----------------------------------------

#define NN_CAPSRV_LOG(...) NN_SDK_LOG("[capsrv]" __VA_ARGS__)

// エラーログ
#define NN_CAPSRV_LOG_ERR(...) NN_SDK_LOG("[capsrv][ERROR]" __VA_ARGS__)
// 警告ログ
#define NN_CAPSRV_LOG_WARN(...) NN_SDK_LOG("[capsrv][WARNING]" __VA_ARGS__)
// 実行ログ
#define NN_CAPSRV_LOG_INFO(...) NN_SDK_LOG("[capsrv]" __VA_ARGS__)
// 開発用ログ
//#define NN_CAPSRV_LOG_DEV(...) NN_SDK_LOG("[capsrv][dev]" __VA_ARGS__)

// Result 値ログ
//#define NN_CAPSRV_LOG_RESULT(...) NN_SDK_LOG("[capsrv][result]" __VA_ARGS__)

// MakerNote ログ
//#define NN_CAPSRV_LOG_MAKERNOTE(...) NN_SDK_LOG("[capsrv][mnote]" __VA_ARGS__)


// 実行時間ログ
#ifdef NN_CAPSRV_ENABLE_TIME_MEASURE
#define NN_CAPSRV_LOG_TIME(...) NN_CAPSRV_LOG("[time]" __VA_ARGS__)
#endif

// ファイル操作関数のログ
//#define NN_CAPSRV_LOG_FILE_MANIPULATOR(...) NN_CAPSRV_LOG("[file]" __VA_ARGS__)

// 一時ファイル操作関数のログ
//#define NN_CAPSRV_LOG_TMPFILE_MANIPULATOR(...) NN_CAPSRV_LOG("[tmpf]" __VA_ARGS__)

// Movie のログ
//#define NN_CAPSRV_LOG_MOVIE_DEV(...) NN_CAPSRV_LOG("[mv]" __VA_ARGS__)

// IPC のログ
#ifdef NN_CAPSRV_ENABLE_IPC_TRACE
#define NN_CAPSRV_LOG_IPC(...) NN_CAPSRV_LOG("[ipc]" __VA_ARGS__)
#endif

//---------------------
// Empty Definitions
//---------------------
#ifndef NN_CAPSRV_LOG
#define NN_CAPSRV_LOG(...)
#endif

#ifndef NN_CAPSRV_LOG_ERR
#define NN_CAPSRV_LOG_ERR(...)
#endif
#ifndef NN_CAPSRV_LOG_WARN
#define NN_CAPSRV_LOG_WARN(...)
#endif
#ifndef NN_CAPSRV_LOG_INFO
#define NN_CAPSRV_LOG_INFO(...)
#endif
#ifndef NN_CAPSRV_LOG_DEV
#define NN_CAPSRV_LOG_DEV(...)
#endif
#ifndef NN_CAPSRV_LOG_RESULT
#define NN_CAPSRV_LOG_RESULT(...)
#endif
#ifndef NN_CAPSRV_LOG_MAKERNOTE
#define NN_CAPSRV_LOG_MAKERNOTE(...)
#endif
#ifndef NN_CAPSRV_LOG_TIME
#define NN_CAPSRV_LOG_TIME(...)
#endif
#ifndef NN_CAPSRV_LOG_FILE_MANIPULATOR
#define NN_CAPSRV_LOG_FILE_MANIPULATOR(...)
#endif
#ifndef NN_CAPSRV_LOG_TMPFILE_MANIPULATOR
#define NN_CAPSRV_LOG_TMPFILE_MANIPULATOR(...)
#endif
#ifndef NN_CAPSRV_LOG_MOVIE_DEV
#define NN_CAPSRV_LOG_MOVIE_DEV(...)
#endif
#ifndef NN_CAPSRV_LOG_IPC
#define NN_CAPSRV_LOG_IPC(...)
#endif

//----------------------------------------
// IPC 計測
//----------------------------------------
#if defined(NN_CAPSRV_ENABLE_IPC_TRACE)
    #define NN_CAPSRV_IPC_TRACE_IMPL(port,name)             \
        auto threadId = nn::os::GetThreadId(nn::os::GetCurrentThread());    \
        NN_CAPSRV_LOG_IPC(port "@t%llu:" #name "\n", threadId);             \
        auto ipcTraceTick0 = nn::os::GetSystemTick();       \
        NN_UTIL_SCOPE_EXIT{                                 \
            auto ipcTraceTick1 = nn::os::GetSystemTick();   \
            NN_CAPSRV_LOG_IPC(port "@t%llu:" #name " -> %lldus\n", threadId, (ipcTraceTick1 - ipcTraceTick0).ToTimeSpan().GetMicroSeconds());   \
        };

    #define NN_CAPSRV_SESSION_IPC_TRACE_IMPL(port,name)             \
        auto threadId = nn::os::GetThreadId(nn::os::GetCurrentThread());    \
        uint64_t ipcTraceSessionId = m_SessionId.id;  \
        NN_CAPSRV_LOG_IPC("%llu:" port "@t%llu:" #name "\n", ipcTraceSessionId, threadId); \
        auto ipcTraceTick0 = nn::os::GetSystemTick();       \
        NN_UTIL_SCOPE_EXIT{                                 \
            auto ipcTraceTick1 = nn::os::GetSystemTick();   \
            NN_CAPSRV_LOG_IPC("%llu:" port "@t%llu:" #name " -> %lldus\n", ipcTraceSessionId, threadId, (ipcTraceTick1 - ipcTraceTick0).ToTimeSpan().GetMicroSeconds());   \
        };
#else
    #define NN_CAPSRV_IPC_TRACE_IMPL(port,name)
    #define NN_CAPSRV_SESSION_IPC_TRACE_IMPL(port,name)
#endif
#define NN_CAPSRV_IPC_TRACE_SC(name) NN_CAPSRV_IPC_TRACE_IMPL("sc",name)
#define NN_CAPSRV_IPC_TRACE_SS(name) NN_CAPSRV_IPC_TRACE_IMPL("ss",name)
#define NN_CAPSRV_IPC_TRACE_SU(name) NN_CAPSRV_IPC_TRACE_IMPL("su",name)
#define NN_CAPSRV_IPC_TRACE_C(name) NN_CAPSRV_IPC_TRACE_IMPL("c",name)
#define NN_CAPSRV_IPC_TRACE_A(name) NN_CAPSRV_IPC_TRACE_IMPL("a",name)
#define NN_CAPSRV_IPC_TRACE_U(name) NN_CAPSRV_IPC_TRACE_IMPL("u",name)
#define NN_CAPSRV_IPC_TRACE_DC(name) NN_CAPSRV_IPC_TRACE_IMPL("dc",name)
#define NN_CAPSRV_SESSION_IPC_TRACE_C(name) NN_CAPSRV_SESSION_IPC_TRACE_IMPL("c",name)
#define NN_CAPSRV_SESSION_IPC_TRACE_A(name) NN_CAPSRV_SESSION_IPC_TRACE_IMPL("a",name)
#define NN_CAPSRV_SESSION_IPC_TRACE_U(name) NN_CAPSRV_SESSION_IPC_TRACE_IMPL("u",name)
