﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumControlSessionImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <mutex>

#include "../../capsrv_Macro.h"
#include "../capsrvServer_Config.h"
#include "capsrvServer_AlbumServerObject.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert((::std::unique_lock<decltype(g_AlbumGlobalMutex)>(g_AlbumGlobalMutex), Function())));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO

#endif


namespace nn{ namespace capsrv{ namespace server{

    AlbumControlSessionImpl::AlbumControlSessionImpl() NN_NOEXCEPT
    {
        m_SessionId = SessionId(g_ResourceIdManager.AcquireResourceId());
        m_MovieReadStreamIdTable.Initialize();
        m_MovieWriteStreamIdTable.Initialize();
        NN_CAPSRV_SESSION_IPC_TRACE_C(ctor);
    }

    AlbumControlSessionImpl::~AlbumControlSessionImpl() NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(dtor);
        (void)CloseAllAlbumMovieReadStreams();
        (void)DiscardAllAlbumMovieWriteStreams();
        g_ResourceIdManager.ReleaseResourceId(m_SessionId.id);

        m_MovieWriteStreamIdTable.Finalize();
        m_MovieReadStreamIdTable.Finalize();
        m_SessionId = SessionId::GetInvalidValue();
    }

    //-----------------------------------------------
    // IMovieReadStreamServiceObject
    //-----------------------------------------------

    nn::Result AlbumControlSessionImpl::CloseAllAlbumMovieReadStreams() NN_NOEXCEPT
    {
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            album::MovieStreamId streamId = {};
            while(m_MovieReadStreamIdTable.GetHead(&streamId))
            {
                NN_CAPSRV_LOG_IPC("  Closing MovieReadStream %llu\n", streamId);
                (void)g_AlbumManager.CloseMovieReadStream(streamId);
                m_MovieReadStreamIdTable.Unregister(streamId);
            }
            NN_RESULT_SUCCESS;
        });
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------

    namespace {
        nn::Result CloseAlbumMovieReadStreamSyncImpl(album::MovieStreamId streamId) NN_NOEXCEPT
        {
            NN_CAPSRV_SERVER_SYNC_DO([&](){
                return g_AlbumManager.CloseMovieReadStream(streamId);
            });
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result AlbumControlSessionImpl::OpenAlbumMovieReadStream(nn::sf::Out<nn::capsrv::detail::AlbumMovieReadStreamHandleType> outHandle, const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(OpenAlbumMovieReadStream);

        NN_RESULT_THROW_UNLESS(!m_MovieReadStreamIdTable.IsFull(), ResultAlbumResourceLimit());

        NN_CAPSRV_PROCESS_START();

        album::MovieStreamId streamId = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.OpenMovieReadStream(&streamId, fileId, m_SessionId);
        });
        NN_CAPSRV_PROCESS_ROLLBACK((void)CloseAlbumMovieReadStreamSyncImpl(streamId));

        NN_RESULT_DO(m_MovieReadStreamIdTable.Register(streamId));
        NN_CAPSRV_PROCESS_ROLLBACK(m_MovieReadStreamIdTable.Unregister(streamId));

        NN_CAPSRV_PROCESS_SUCCESS();

        outHandle.Set(streamId.id);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::CloseAlbumMovieReadStream(nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(CloseAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            (void)g_AlbumManager.CloseMovieReadStream(streamId);
            NN_RESULT_SUCCESS;
        });
        (void)m_MovieReadStreamIdTable.Unregister(streamId);

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::GetAlbumMovieReadStreamMovieDataSize(nn::sf::Out<std::int64_t> outSize, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(GetAlbumMovieReadStreamMovieDataSize);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        int64_t size = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieReadStreamMovieDataSize(&size, streamId);
        });

        outSize.Set(size);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::ReadMovieDataFromAlbumMovieReadStream(nn::sf::Out<std::int64_t> outReadSize, const nn::sf::OutBuffer& outBuffer, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle, std::int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(ReadMovieDataFromAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        void* buffer = outBuffer.GetPointerUnsafe();
        size_t bufferSize = outBuffer.GetSize();
        size_t readSize = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.ReadMovieDataFromMovieReadStream(&readSize, buffer, bufferSize, streamId, offset);
        });

        outReadSize.Set(static_cast<int64_t>(readSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::GetAlbumMovieReadStreamImageDataSize(nn::sf::Out<std::int64_t> outSize, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(GetAlbumMovieReadStreamImageDataSize);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        int64_t size = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieReadStreamImageDataSize(&size, streamId);
        });

        outSize.Set(size);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::ReadImageDataFromAlbumMovieReadStream(nn::sf::Out<std::int64_t> outReadSize, const nn::sf::OutBuffer& outBuffer, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle, std::int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(ReadImageDataFromAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        void* buffer = outBuffer.GetPointerUnsafe();
        size_t bufferSize = outBuffer.GetSize();
        size_t readSize = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.ReadImageDataFromMovieReadStream(&readSize, buffer, bufferSize, streamId, offset);
        });

        outReadSize.Set(static_cast<int64_t>(readSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::ReadFileAttributeFromAlbumMovieReadStream(nn::sf::Out<nn::capsrv::detail::ScreenShotAttributeEx0> outAttribute, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(ReadFileAttributeFromAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        album::AlbumFileAttribute attr = {};

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.ReadFileAttributeFromMovieReadStream(&attr, nullptr, nullptr, streamId);
        });

        *outAttribute = attr.attribute.screenshot;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::GetAlbumMovieReadStreamBrokenReason(nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(GetAlbumMovieReadStreamBrokenReason);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        nn::Result reason = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieReadStreamBrokenReason(&reason, streamId);
        });

        NN_RESULT_THROW(reason);
    }

    //-----------------------------------------------
    // IMovieWriteStreamServiceObject
    //-----------------------------------------------

    nn::Result AlbumControlSessionImpl::DiscardAllAlbumMovieWriteStreams() NN_NOEXCEPT
    {
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            album::MovieStreamId streamId = {};
            while(m_MovieWriteStreamIdTable.GetHead(&streamId))
            {
                NN_CAPSRV_LOG_IPC("  Discarding MovieWriteStream %llu\n", streamId);
                (void)g_AlbumManager.DiscardMovieWriteStream(streamId);
                m_MovieWriteStreamIdTable.Unregister(streamId);
            }
            NN_RESULT_SUCCESS;
        });
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------

    namespace {
        nn::Result DiscardAlbumMovieWriteStreamSyncImpl(album::MovieStreamId streamId) NN_NOEXCEPT
        {
            NN_CAPSRV_SERVER_SYNC_DO([&](){
                return g_AlbumManager.DiscardMovieWriteStream(streamId);
            });
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result AlbumControlSessionImpl::OpenAlbumMovieWriteStream(nn::sf::Out<nn::capsrv::detail::AlbumMovieWriteStreamHandleType> outHandle, const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(OpenAlbumMovieWriteStream);

        NN_RESULT_THROW_UNLESS(!m_MovieWriteStreamIdTable.IsFull(), ResultAlbumResourceLimit());

        NN_CAPSRV_PROCESS_START();

        album::MovieStreamId streamId = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.OpenMovieWriteStream(&streamId, fileId, m_SessionId);
        });
        NN_CAPSRV_PROCESS_ROLLBACK((void)DiscardAlbumMovieWriteStreamSyncImpl(streamId));

        NN_RESULT_DO(m_MovieWriteStreamIdTable.Register(streamId));
        NN_CAPSRV_PROCESS_ROLLBACK(m_MovieWriteStreamIdTable.Unregister(streamId));

        NN_CAPSRV_PROCESS_SUCCESS();
        outHandle.Set(streamId.id);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::FinishAlbumMovieWriteStream(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(FinishAlbumMovieWriteStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.FinishMovieWriteStream(streamId);
        });
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::CommitAlbumMovieWriteStream(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(CommitAlbumMovieWriteStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.CommitMovieWriteStream(streamId);
        });
        m_MovieWriteStreamIdTable.Unregister(streamId);

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::DiscardAlbumMovieWriteStream(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(DiscardAlbumMovieWriteStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.DiscardMovieWriteStream(streamId);
        });
        m_MovieWriteStreamIdTable.Unregister(streamId);

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::DiscardAlbumMovieWriteStreamNoDelete(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(DiscardAlbumMovieWriteStreamNoDelete);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.DiscardMovieWriteStreamNoDelete(streamId);
        });
        m_MovieWriteStreamIdTable.Unregister(streamId);

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::StartAlbumMovieWriteStreamDataSection(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(StartAlbumMovieWriteStreamDataSection);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.StartMovieWriteStreamDataSection(streamId);
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::EndAlbumMovieWriteStreamDataSection(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(EndAlbumMovieWriteStreamDataSection);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.EndMovieWriteStreamDataSection(streamId);
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::StartAlbumMovieWriteStreamMetaSection(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(StartAlbumMovieWriteStreamMetaSection);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.StartMovieWriteStreamMetaSection(streamId);
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::EndAlbumMovieWriteStreamMetaSection(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(EndAlbumMovieWriteStreamMetaSection);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.EndMovieWriteStreamMetaSection(streamId);
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::ReadDataFromAlbumMovieWriteStream(nn::sf::Out<std::int64_t> outReadSize, const nn::sf::OutBuffer& outBuffer, nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle, std::int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(ReadDataFromAlbumMovieWriteStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        size_t readSize = 0;
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.ReadDataFromMovieWriteStream(
                &readSize,
                outBuffer.GetPointerUnsafe(),
                outBuffer.GetSize(),
                streamId,
                offset
            );
        });

        outReadSize.Set(static_cast<int64_t>(readSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::WriteDataToAlbumMovieWriteStream(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle, std::int64_t offset, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(WriteDataToAlbumMovieWriteStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.WriteDataToMovieWriteStream(
                streamId,
                offset,
                inBuffer.GetPointerUnsafe(),
                inBuffer.GetSize()
            );
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::WriteMetaToAlbumMovieWriteStream(
        nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle,
        const nn::sf::InBuffer& inBuffer,
        std::uint64_t makerNoteVersion,
        std::int64_t makerNoteOffset,
        std::int64_t makerNoteSize
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(WriteMetaToAlbumMoveiWriteStream);
        NN_UNUSED(inBuffer);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.WriteMetaToMovieWriteStream(streamId, inBuffer.GetPointerUnsafe(), inBuffer.GetSize(), makerNoteVersion, makerNoteOffset, makerNoteSize);
        });

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::GetAlbumMovieWriteStreamBrokenReason(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(GetAlbumMovieWriteStreamBrokenReason);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        nn::Result reason = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieWriteStreamBrokenReason(&reason, streamId);
        });

        NN_RESULT_THROW(reason);
    }

    nn::Result AlbumControlSessionImpl::GetAlbumMovieWriteStreamDataSize(nn::sf::Out<std::int64_t> outValue, nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(GetAlbumMovieWriteStreamDataSize);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        int64_t value = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieWriteStreamDataSize(&value, streamId);
        });

        outValue.Set(value);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumControlSessionImpl::SetAlbumMovieWriteStreamDataSize(nn::capsrv::detail::AlbumMovieWriteStreamHandleType handle, std::int64_t value) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_C(SetAlbumMovieWriteStreamDataSize);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieWriteStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.SetMovieWriteStreamDataSize(streamId, value);
        });

        NN_RESULT_SUCCESS;
    }

}}}
