﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumAccessorApplicationSessionImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <mutex>

#include "../../capsrv_Macro.h"
#include "../capsrvServer_Config.h"
#include "capsrvServer_AlbumServerObject.h"
#include "../detail/capsrvServer_DecryptApplicationAlbumEntry.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert((::std::unique_lock<decltype(g_AlbumGlobalMutex)>(g_AlbumGlobalMutex), Function())));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO

#endif


namespace nn{ namespace capsrv{ namespace server{

    AlbumAccessorApplicationSessionImpl::AlbumAccessorApplicationSessionImpl() NN_NOEXCEPT
    {
        m_SessionId = SessionId(g_ResourceIdManager.AcquireResourceId());
        m_MovieReadStreamIdTable.Initialize();
        NN_CAPSRV_SESSION_IPC_TRACE_U(ctor);
    }

    AlbumAccessorApplicationSessionImpl::~AlbumAccessorApplicationSessionImpl() NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(dtor);
        (void)CloseAllAlbumMovieReadStreams();
        g_ResourceIdManager.ReleaseResourceId(m_SessionId.id);

        m_MovieReadStreamIdTable.Finalize();
        m_SessionId = SessionId::GetInvalidValue();
    }

    //-----------------------------------------------
    // IMovieReadStreamServiceObject
    //-----------------------------------------------

    nn::Result AlbumAccessorApplicationSessionImpl::CloseAllAlbumMovieReadStreams() NN_NOEXCEPT
    {
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            album::MovieStreamId streamId = {};
            while(m_MovieReadStreamIdTable.GetHead(&streamId))
            {
                NN_CAPSRV_LOG_IPC("  Closing MovieReadStream %llu\n", streamId);
                (void)g_AlbumManager.CloseMovieReadStream(streamId);
                m_MovieReadStreamIdTable.Unregister(streamId);
            }
            NN_RESULT_SUCCESS;
        });
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------

    namespace {
        nn::Result CloseAlbumMovieReadStreamSyncImpl(album::MovieStreamId streamId) NN_NOEXCEPT
        {
            NN_CAPSRV_SERVER_SYNC_DO([&](){
                return g_AlbumManager.CloseMovieReadStream(streamId);
            });
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result AlbumAccessorApplicationSessionImpl::OpenAlbumMovieReadStream(nn::sf::Out<nn::capsrv::detail::AlbumMovieReadStreamHandleType> outHandle, const nn::capsrv::ApplicationAlbumFileEntry& srcEntry, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(OpenAlbumMovieReadStreamForApplication);

        NN_RESULT_THROW_UNLESS(!m_MovieReadStreamIdTable.IsFull(), ResultAlbumResourceLimit());

        NN_CAPSRV_PROCESS_START();

        ncm::ApplicationId applicationId = {};
        NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

        AlbumEntry entry = {};
        NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

        album::MovieStreamId streamId = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.OpenMovieReadStream(&streamId, entry.fileId, m_SessionId);
        });
        NN_CAPSRV_PROCESS_ROLLBACK((void)CloseAlbumMovieReadStreamSyncImpl(streamId));

        NN_RESULT_DO(m_MovieReadStreamIdTable.Register(streamId));
        NN_CAPSRV_PROCESS_ROLLBACK(m_MovieReadStreamIdTable.Unregister(streamId));

        NN_CAPSRV_PROCESS_SUCCESS();

        outHandle.Set(streamId.id);
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumAccessorApplicationSessionImpl::CloseAlbumMovieReadStream(nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(CloseAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        NN_CAPSRV_SERVER_SYNC_DO([&](){
            (void)g_AlbumManager.CloseMovieReadStream(streamId);
            NN_RESULT_SUCCESS;
        });
        (void)m_MovieReadStreamIdTable.Unregister(streamId);

        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumAccessorApplicationSessionImpl::GetAlbumMovieReadStreamMovieDataSize(nn::sf::Out<std::int64_t> outSize, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(GetAlbumMovieReadStreamMovieDataSize);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        int64_t size = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieReadStreamMovieDataSize(&size, streamId);
        });

        outSize.Set(size);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumAccessorApplicationSessionImpl::ReadMovieDataFromAlbumMovieReadStream(nn::sf::Out<std::int64_t> outReadSize, const nn::sf::OutBuffer& outBuffer, nn::capsrv::detail::AlbumMovieReadStreamHandleType handle, std::int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(ReadMovieDataFromAlbumMovieReadStream);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        void* buffer = outBuffer.GetPointerUnsafe();
        size_t bufferSize = outBuffer.GetSize();
        size_t readSize = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.ReadMovieDataFromMovieReadStream(&readSize, buffer, bufferSize, streamId, offset);
        });

        outReadSize.Set(static_cast<int64_t>(readSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumAccessorApplicationSessionImpl::GetAlbumMovieReadStreamBrokenReason(nn::capsrv::detail::AlbumMovieReadStreamHandleType handle) NN_NOEXCEPT
    {
        NN_CAPSRV_SESSION_IPC_TRACE_U(GetAlbumMovieReadStreamBrokenReason);

        auto streamId = album::MovieStreamId(handle);
        NN_RESULT_THROW_UNLESS(m_MovieReadStreamIdTable.Contains(streamId), ResultAlbumNotFound());

        nn::Result reason = {};
        NN_CAPSRV_SERVER_SYNC_DO([&](){
            return g_AlbumManager.GetMovieReadStreamBrokenReason(&reason, streamId);
        });

        NN_RESULT_THROW(reason);
    }

}}}
