﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_VerifyMovieFileData.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../../capsrv_Macro.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace {

        struct MovieFileMetaVerificationOutput
        {
            int64_t outMovieDataBlockCount;
            int64_t outMovieDataOffset;
            int64_t outMovieDataSize;
            int64_t outImageDataOffset;
            int64_t outImageDataSize;
            AlbumMovieUtility::MetaHashList outMetaHashList;
        };

        struct MovieFileMetaLoadImageOutput
        {
            size_t  outReadSize;
            int64_t outImageDataSize;
        };

        struct MovieFileMetaLoafImageBuffer
        {
            void*   pBuffer;
            size_t  bufferSize;
        };

        nn::Result VerifyMovieFileDataImpl(
            MovieFileMetaVerificationOutput* pOutVerification,
            MovieFileMetaLoadImageOutput* pOutLoadImage,
            const AlbumFileHandle& hFile,
            int64_t fileSize,
            const nn::util::optional<TemporaryFileHandle> hHashFile,
            const nn::util::optional<MovieFileMetaLoafImageBuffer> imageBuf,
            const EnvironmentInfo& env,
            void* workMemory,
            size_t workMemorySize
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(workMemory);

            NN_CAPSRV_PROCESS_START();
            if(imageBuf)
            {
                std::memset(imageBuf->pBuffer, 0, imageBuf->bufferSize);
            }
            NN_CAPSRV_PROCESS_ROLLBACK(
                if(imageBuf){
                    std::memset(imageBuf->pBuffer, 0, imageBuf->bufferSize);
                }
            );

            // 画像の読込の最大サイズ
            size_t outImageSizeMax = 0;
            if(imageBuf)
            {
                outImageSizeMax = std::min<size_t>(imageBuf->bufferSize, MovieMetaImageDataSize);
            }

            struct FileTail
            {
                AlbumMovieUtility::FileSign sign;
                uint64_t untrustedVersion;
            };

            // 構造体にパディングが入っていないこと
            NN_STATIC_ASSERT(sizeof(FileTail) == MovieSignatureSize + sizeof(uint64_t));

            // ファイルサイズが署名＋バージョンにも満たないならエラー
            NN_RESULT_THROW_UNLESS(fileSize >= sizeof(FileTail), ResultAlbumInvalidFileData());

            // ファイル末尾を読込み
            // ハッシュ部分もまとめて読みたい感じもあるが、単純にするため末尾だけ読む
            FileTail tail = {};
            {
                size_t readSize = 0;
                NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, fileSize - sizeof(FileTail), &tail, sizeof(FileTail)));
                NN_RESULT_THROW_UNLESS(readSize == sizeof(FileTail), ResultAlbumInvalidFileData());
            }

            // 各セクションの大きさを計算
            int64_t dataBlockCount = 0;
            int64_t dataSize = 0;
            int64_t metaSize = 0;
            int64_t dataHashSize = 0;
            int64_t metaHashSize = 0;
            int64_t signSize = 0;
            int64_t versionSize = 0;
            NN_RESULT_DO(AlbumMovieUtility::CalculateSectionSize(
                &dataBlockCount,
                &dataSize,
                &metaSize,
                &dataHashSize,
                &metaHashSize,
                &signSize,
                &versionSize,
                fileSize,
                tail.untrustedVersion
            ));

            // ハッシュを検査
            {
                auto pHashList = reinterpret_cast<AlbumMovieUtility::BlockHash*>(workMemory);
                int64_t countToReadMax = workMemorySize / sizeof(AlbumMovieUtility::BlockHash);

                AlbumMovieUtility::HashSectionValidationState hashState = {};
                NN_RESULT_DO(AlbumMovieUtility::StartHashValidation(&hashState, tail.untrustedVersion));

                int64_t offsetToRead = dataSize + metaSize;
                int64_t offsetToWrite = 0;
                int64_t remain = dataHashSize + metaHashSize;
                while(remain > 0)
                {
                    int64_t sizeToRead = std::min<int64_t>(remain, countToReadMax * sizeof(AlbumMovieUtility::BlockHash));

                    // ファイルから読み込み
                    size_t readSize = 0;
                    NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, offsetToRead, pHashList, sizeToRead));
                    NN_RESULT_THROW_UNLESS(readSize == sizeToRead, ResultAlbumInvalidFileData());

                    // 署名計算を進める
                    int64_t readCount = sizeToRead / sizeof(AlbumMovieUtility::BlockHash);
                    AlbumMovieUtility::ProcessHashValidation(hashState, pHashList, readCount);

                    // 一時ファイルへ書き出し
                    if(hHashFile)
                    {
                        NN_RESULT_DO(TemporaryFileManipulator::WriteFile(*hHashFile, offsetToWrite, pHashList, sizeToRead));
                    }

                    offsetToRead += sizeToRead;
                    offsetToWrite += sizeToRead;
                    remain -= sizeToRead;
                }

                NN_RESULT_TRY(AlbumMovieUtility::FinishHashValidation(hashState, tail.sign))
                    NN_RESULT_CATCH(ResultInternalFileDataVerificationInconsistentSignature)
                    {
                        if(env.IsAlbumMovieFileSignVerificationEnabled())
                        {
                            NN_RESULT_RETHROW;
                        }
                        else
                        {
                            NN_CAPSRV_LOG_WARN("movie file sign verification is disabled\n");
                        }
                    }
                NN_RESULT_END_TRY;
            }

            // メタのハッシュを取得
            typedef AlbumMovieUtility::MetaHashList MetaHashList;
            MetaHashList metaHashList = {};
            for(int i = 0; i < MetaHashList::Length; i++)
            {
                int64_t offset = dataSize + metaSize + dataHashSize + MovieDataBlockHashSize * i;
                size_t readSize = 0;
                NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, offset, &metaHashList.hashList[i], MovieDataBlockHashSize));
                NN_RESULT_THROW_UNLESS(readSize == MovieDataBlockHashSize, ResultAlbumInvalidFileData());
            }

            // メタを検証
            //   imageBuf が渡されていれば、画像データをコピーする。
            size_t writtenImageSize = 0;
            AlbumMovieMetaInfo info = {};
            {
                void* pBlock = workMemory;

                int64_t baseOffset = dataSize;
                for(int i = 0; i < MetaHashList::Length; i++)
                {
                    int64_t offsetToRead = baseOffset + i * MovieDataBlockSize;

                    // ファイルから読み込み
                    size_t readSize = 0;
                    NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, offsetToRead, pBlock, MovieDataBlockSize));
                    NN_RESULT_THROW_UNLESS(readSize == MovieDataBlockSize, ResultAlbumInvalidFileData());

                    // ハッシュを検査
                    NN_RESULT_TRY(AlbumMovieUtility::ValidateBlockData(pBlock, MovieDataBlockSize, metaHashList.hashList[i]))
                        NN_RESULT_CATCH(ResultInternalFileDataVerificationInconsistentBlockHash)
                        {
                            if(env.IsAlbumMovieFileHashVerificationEnabled())
                            {
                                NN_RESULT_RETHROW;
                            }
                            else
                            {
                                NN_CAPSRV_LOG_WARN("movie meta hash verification is disabled\n");
                            }
                        }
                    NN_RESULT_END_TRY;

                    // 画像を出力バッファにコピー
                    //   最後まで検証が通らなかった場合、ロールバック処理でゼロクリアされる。
                    if(imageBuf && writtenImageSize < outImageSizeMax)
                    {
                        size_t sizeToCopy = std::min(static_cast<size_t>(MovieDataBlockSize), outImageSizeMax - writtenImageSize);
                        std::memcpy(reinterpret_cast<char*>(imageBuf->pBuffer) + writtenImageSize, pBlock, sizeToCopy);
                        writtenImageSize += sizeToCopy;
                        NN_SDK_ASSERT_LESS_EQUAL(writtenImageSize, outImageSizeMax);
                    }
                }

                // この時点で、 pBlock にはメタの最後のブロックのデータが入っている

                // ということで、メタの末尾にある Info を解析する
                NN_STATIC_ASSERT(sizeof(nn::capsrv::movie::MovieMetaInfoData) <= MovieDataBlockSize);
                auto* pInfo = reinterpret_cast<const nn::capsrv::movie::MovieMetaInfoData*>(
                    reinterpret_cast<const char*>(pBlock) + MovieDataBlockSize - sizeof(nn::capsrv::movie::MovieMetaInfoData)
                );

                NN_RESULT_DO(AlbumMovieUtility::ExtractMetaInfo(&info, pInfo, tail.untrustedVersion, dataSize));

                // NOTE: ここまで来てようやくファイル末尾のバージョンが正しいことが確認できた
            }

            // 書き込んだサイズを画像データの大きさにフィッティング
            writtenImageSize = std::min(writtenImageSize, static_cast<size_t>(info.GetImageDataSize()));

            NN_CAPSRV_PROCESS_SUCCESS();
            if(pOutVerification)
            {
                pOutVerification->outMovieDataBlockCount = dataBlockCount;
                pOutVerification->outMovieDataOffset = 0;
                pOutVerification->outMovieDataSize = info.GetMovieDataSize();
                pOutVerification->outImageDataOffset = dataSize;
                pOutVerification->outImageDataSize = info.GetImageDataSize();
                pOutVerification->outMetaHashList = metaHashList;
            }
            if(pOutLoadImage)
            {
                pOutLoadImage->outReadSize      = writtenImageSize;
                pOutLoadImage->outImageDataSize = info.GetImageDataSize();
            }
            NN_RESULT_SUCCESS;
        }// NOLINT(impl/function_size)
    }

    nn::Result VerifyMovieFileSignAndMeta(
        int64_t* pOutMovieDataBlockCount,
        int64_t* pOutMovieDataOffset,
        int64_t* pOutMovieDataSize,
        int64_t* pOutImageDataOffset,
        int64_t* pOutImageDataSize,
        AlbumMovieUtility::MetaHashList* pOutMetaHashList,
        const AlbumFileHandle& hFile,
        int64_t fileSize,
        const TemporaryFileHandle& hHashFile,
        const EnvironmentInfo& env,
        void* workMemory,
        size_t workMemorySize
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutMovieDataOffset);
        NN_SDK_REQUIRES_NOT_NULL(pOutMovieDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageDataOffset);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutMetaHashList);

        MovieFileMetaVerificationOutput out = {};
        NN_RESULT_DO(VerifyMovieFileDataImpl(
            &out,
            nullptr,
            hFile,
            fileSize,
            hHashFile,
            nn::util::nullopt /* imageBuf */,
            env,
            workMemory,
            workMemorySize
        ));

        *pOutMovieDataBlockCount = out.outMovieDataBlockCount;
        *pOutMovieDataOffset     = out.outMovieDataOffset;
        *pOutMovieDataSize       = out.outMovieDataSize;
        *pOutImageDataOffset     = out.outImageDataOffset;
        *pOutImageDataSize       = out.outImageDataSize;
        *pOutMetaHashList        = out.outMetaHashList;
        NN_RESULT_SUCCESS;
    }

    nn::Result VerifyMovieFileAndLoadImage(
        size_t* pOutSize,
        int64_t* pOutImageDataSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileHandle& hFile,
        int64_t fileSize,
        const EnvironmentInfo& env,
        void* workMemory,
        size_t workMemorySize
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageDataSize);

        MovieFileMetaLoadImageOutput out = {};
        MovieFileMetaLoafImageBuffer imageBuf = { pBuffer, bufferSize };
        NN_RESULT_DO(VerifyMovieFileDataImpl(
            nullptr,
            &out,
            hFile,
            fileSize,
            nn::util::nullopt /* hHashFile */,
            imageBuf,
            env,
            workMemory,
            workMemorySize
        ));

        *pOutSize          = out.outReadSize;
        *pOutImageDataSize = out.outImageDataSize;
        NN_RESULT_SUCCESS;
    }


}}}}
