﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_TemporaryFileManipulator.h"

#include <atomic>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ServiceConfig.h>
#include "../capsrvServer_ResultPrivate.h"
#include "../capsrvServer_Config.h"
#include "../../capsrv_Macro.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    const size_t TemporaryFileHandle::PathSize;

    TemporaryFileHandle::TemporaryFileHandle() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    //------------------------------------------------

    namespace {

        // @pre size == TemporaryFileHandle::PathSize
        nn::Result MakeTemporaryFilePath(char* buffer, size_t size) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(size, TemporaryFileHandle::PathSize);
            NN_RESULT_THROW_UNLESS(size == TemporaryFileHandle::PathSize, ResultInternalAlbumTemporaryFileError());

            uint64_t value = 0;
            NN_STATIC_ASSERT(sizeof(value) * 2 == TemporaryFileHandle::PathSize - 5); // 1 バイトあたり 2 文字。"TM:/" と '\0' で 5 文字。
            nn::os::GenerateRandomBytes(&value, sizeof(value));

            int length = nn::util::SNPrintf(buffer, size, NN_CAPSRV_MOUNT_NAME_TEMP ":/%016llX", value);
            NN_SDK_ASSERT_EQUAL(length, static_cast<int>(TemporaryFileHandle::PathSize - 1));
            NN_UNUSED(length);

            NN_RESULT_SUCCESS;
        }

        std::atomic_int g_TemporaryFileCount;

    }

    nn::Result TemporaryFileManipulator::OpenNewTemporaryFile(TemporaryFileHandle* pOutHandle, int64_t initialFileSize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("opening new temporary file: %lld bytes\n", initialFileSize);

        NN_RESULT_THROW_UNLESS(!IsQuestMode(), ResultAlbumIsFull());

        TemporaryFileHandle h = {};

        NN_CAPSRV_PROCESS_START();

        // ファイル作成数制限の確認
        int fileCount = ++g_TemporaryFileCount;
        NN_CAPSRV_PROCESS_ROLLBACK(--g_TemporaryFileCount);

        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  checking temporary file count -> %d / %d\n", fileCount, TemporaryFileCountMax);
        NN_RESULT_THROW_UNLESS(fileCount <= TemporaryFileCountMax, ResultInternalAlbumTemporaryFileCountLimit());

        // ぶつからない名前でファイルを作る
        int trial = 0;
        for(trial = 0; trial < TemporaryFileNameGenerationRetryCountMax; trial++)
        {
            std::memset(h.m_Path, 0, sizeof(h.m_Path));
            NN_RESULT_DO(MakeTemporaryFilePath(h.m_Path, sizeof(h.m_Path)));

            // 作ってみる
            auto createResult = nn::fs::CreateFile(h.m_Path, initialFileSize);
            NN_RESULT_TRY(createResult)
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  creating %s -> already exists\n", h.m_Path);
                    continue;
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  creating %s -> failure(%d-%d)\n", h.m_Path, createResult.GetModule(), createResult.GetDescription());
                    NN_RESULT_THROW(ResultInternalAlbumTemporaryFileCreateError());
                }
            NN_RESULT_END_TRY;
            NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  creating %s -> success\n", h.m_Path);
            break;
        }
        NN_RESULT_THROW_UNLESS(trial < TemporaryFileNameGenerationRetryCountMax, ResultInternalAlbumTemporaryFileCreateRetryCountLimit());
        NN_CAPSRV_PROCESS_ROLLBACK((void)nn::fs::DeleteFile(h.m_Path));

        auto openResult = nn::fs::OpenFile(&h.m_Handle, h.m_Path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        NN_RESULT_TRY(openResult)
            NN_RESULT_CATCH_ALL
            {
                NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  opening %s -> failure(%d-%d)\n", h.m_Path, openResult.GetModule(), openResult.GetDescription());
                NN_RESULT_THROW(ResultInternalAlbumTemporaryFileOpenError());
            }
        NN_RESULT_END_TRY;
        NN_CAPSRV_PROCESS_ROLLBACK(nn::fs::CloseFile(h.m_Handle));

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  -> success\n");
        *pOutHandle = h;
        NN_RESULT_SUCCESS;
    }

    void TemporaryFileManipulator::DiscardTemporaryFile(const TemporaryFileHandle& handle) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("discarding temporary file(%s)\n", handle.m_Path);
        nn::fs::FlushFile(handle.m_Handle);
        nn::fs::CloseFile(handle.m_Handle);
        (void)nn::fs::DeleteFile(handle.m_Path);
        int fileCount = --g_TemporaryFileCount;
        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  fileCount -> %d / %d\n", fileCount, TemporaryFileCountMax);
        NN_ABORT_UNLESS_GREATER_EQUAL(fileCount, 0);
        NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("  -> success\n");
    }

    nn::Result TemporaryFileManipulator::GetFileSize(int64_t* pOutValue, const TemporaryFileHandle& handle) NN_NOEXCEPT
    {
        auto result = nn::fs::GetFileSize(pOutValue, handle.m_Handle);
        if(result.IsFailure())
        {
            NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("getting size(%s) -> failure(%d-%d)\n", handle.m_Path, result.GetModule(), result.GetDescription());
            NN_RESULT_THROW(ResultInternalAlbumTemporaryFileGetFileSizeError());
        }
        else
        {
            //NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("getting size(%s) -> success\n", handle.m_Path);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result TemporaryFileManipulator::ReadFile(size_t* pOutReadSize, const TemporaryFileHandle& handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        auto result = nn::fs::ReadFile(pOutReadSize, handle.m_Handle, offset, buffer, size);
        if(result.IsFailure())
        {
            NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("reading (%s, offset=%llu, size=%llu) -> failure(%d-%d)\n", handle.m_Path, offset, size, result.GetModule(), result.GetDescription());
            NN_RESULT_THROW(ResultInternalAlbumTemporaryFileReadFileError());
        }
        else
        {
            //NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("reading (%s, offset=%llu, size=%llu) -> success\n", handle.m_Path, offset, size);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result TemporaryFileManipulator::WriteFile(const TemporaryFileHandle& handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        auto result = nn::fs::WriteFile(handle.m_Handle, offset, buffer, size, nn::fs::WriteOption::MakeValue(0));
        if(result.IsFailure())
        {
            NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("writing (%s, offset=%llu, size=%llu) -> failure(%d-%d)\n", handle.m_Path, offset, size, result.GetModule(), result.GetDescription());
            NN_RESULT_THROW(ResultInternalAlbumTemporaryFileWriteFileError());
        }
        else
        {
            //NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("writing (%s, offset=%llu, size=%llu) -> success\n", handle.m_Path, offset, size);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result TemporaryFileManipulator::ResizeFile(const TemporaryFileHandle& handle, int64_t size) NN_NOEXCEPT
    {
        auto result = nn::fs::SetFileSize(handle.m_Handle, size);
        if(result.IsFailure())
        {
            NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("resizing (%s, size=%llu) -> failure(%d-%d)\n", handle.m_Path, size, result.GetModule(), result.GetDescription());
            NN_RESULT_THROW(ResultInternalAlbumTemporaryFileSetFileSizeError());
        }
        else
        {
            //NN_CAPSRV_LOG_TMPFILE_MANIPULATOR("resizing (%s, size=%llu) -> success\n", handle.m_Path, size);
        }
        NN_RESULT_SUCCESS;
    }


}}}}
