﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"
#include "../capsrvServer_EnvironmentInfo.h"
#include "../capsrvServer_SessionId.h"
#include "capsrvServer_AlbumFileManipulator.h"
#include "capsrvServer_AlbumFileLockTable.h"
#include "capsrvServer_MovieStreamId.h"
#include "capsrvServer_MovieWriteStreamEntry.h"
#include "capsrvServer_MovieReadStreamEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace detail{


        template<typename TEntryType, int TCapacity>
        class MovieStreamEntryList
        {
        public:
            static const int Capacity = TCapacity;
            typedef TEntryType EntryType;

        private:
            enum EntryState
            {
                EntryState_Free = 0,
                EntryState_Used,
            };

        public:
            void Initialize() NN_NOEXCEPT
            {
                std::memset(m_StateList, 0, sizeof(m_StateList));
                std::memset(m_EntryList, 0, sizeof(m_EntryList));
            }

            void Finalize() NN_NOEXCEPT
            {
                std::memset(m_StateList, 0, sizeof(m_StateList));
                std::memset(m_EntryList, 0, sizeof(m_EntryList));
            }

            EntryType& GetEntry(int index) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_RANGE(index, 0, static_cast<int>(Capacity));
                NN_SDK_REQUIRES_NOT_EQUAL(m_StateList[index], EntryState_Free);
                return m_EntryList[index];
            }

            const EntryType& GetEntry(int index) const NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_RANGE(index, 0, static_cast<int>(Capacity));
                NN_SDK_REQUIRES_NOT_EQUAL(m_StateList[index], EntryState_Free);
                return m_EntryList[index];
            }

            // bool F(const EntryType&)
            template<typename F>
            nn::Result FindEntryIndex(int* pOutIndex, F cond) const NN_NOEXCEPT
            {
                for(int i = 0; i < Capacity; i++)
                {
                    if(m_StateList[i] == EntryState_Free)
                    {
                        continue;
                    }
                    if(!cond(static_cast<const EntryType&>(m_EntryList[i])))
                    {
                        continue;
                    }
                    *pOutIndex = i;
                    NN_RESULT_SUCCESS;
                }
                *pOutIndex = -1;
                NN_RESULT_THROW(ResultInternalAlbumStreamNotFound());
            }

            nn::Result AcquireEntryIndex(int* pOutEntryIndex) NN_NOEXCEPT
            {
                int entryIndex = -1;
                for(int i = 0; i < Capacity; i++)
                {
                    if(m_StateList[i] == EntryState_Free)
                    {
                        entryIndex = i;
                        break;
                    }
                }
                NN_RESULT_THROW_UNLESS(entryIndex >= 0, ResultInternalAlbumSessionCountLimit());
                NN_SDK_ASSERT_LESS(entryIndex, static_cast<int>(Capacity));

                std::memset(&m_EntryList[entryIndex], 0, sizeof(EntryType));
                m_StateList[entryIndex] = EntryState_Used;
                *pOutEntryIndex = entryIndex;
                NN_RESULT_SUCCESS;
            }

            void ReleaseEntryIndex(int entryIndex) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_RANGE(entryIndex, 0, static_cast<int>(Capacity));
                NN_SDK_REQUIRES_EQUAL(m_StateList[entryIndex], EntryState_Used);
                m_StateList[entryIndex] = EntryState_Free;
                std::memset(&m_EntryList[entryIndex], 0, sizeof(EntryType));
            }

            // nn::Result func(EntryType&)
            template<typename F>
            nn::Result ForeachEntry(F func) NN_NOEXCEPT
            {
                for(int i = 0; i < Capacity; i++)
                {
                    if(m_StateList[i] == EntryState_Free)
                    {
                        continue;
                    }
                    NN_RESULT_DO(func(static_cast<EntryType&>(m_EntryList[i])));
                }
                NN_RESULT_SUCCESS;
            }

            // nn::Result func(const EntryType&)
            template<typename F>
            nn::Result ForeachEntry(F func) const NN_NOEXCEPT
            {
                for(int i = 0; i < Capacity; i++)
                {
                    if(m_StateList[i] == EntryState_Free)
                    {
                        continue;
                    }
                    NN_RESULT_DO(func(static_cast<const EntryType&>(m_EntryList[i])));
                }
                NN_RESULT_SUCCESS;
            }

        private:
            EntryState m_StateList[Capacity];
            EntryType  m_EntryList[Capacity];
        };

    }

    class MovieStreamManager
    {
    public:
        MovieStreamManager() NN_NOEXCEPT;
        void Initialize(
            const EnvironmentInfo* pEnvInfo,
            ResourceIdManager* pResourceIdMgr,
            AlbumFileLockTable* pReadLockTable,
            AlbumFileLockTable* pWriteLockTable
        ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        void NotifyStorageUnmounted(AlbumStorageType storage) NN_NOEXCEPT;

        //-------------
        // ReadStream
        //-------------

        // @pre pOutStreamId != nullptr
        // @pre sessionId != InvalidValue
        nn::Result OpenReadStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result CloseReadStream(MovieStreamId streamId) NN_NOEXCEPT;

        nn::Result ReadMovieDataFromReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT;
        nn::Result GetReadStreamMovieDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result ReadImageDataFromReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result GetReadStreamImageDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;

        nn::Result GetReadStreamFileId(AlbumFileId* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result GetReadStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;

        //-------------
        // WriteStream
        //-------------

        nn::Result OpenWriteStream(MovieStreamId* pOutStreamId, AlbumCacheDelta* pOutCacheDelta, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT;
        nn::Result StartWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result EndWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result StartWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result EndWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result FinishWriteStream(MovieStreamId streamId, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result CommitWriteStream(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result DiscardWriteStream(AlbumCacheDelta* pOutCacheDelta, MovieStreamId streamId, bool isDeleteSuppressed) NN_NOEXCEPT;

        nn::Result ReadDataFromWriteStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT;
        nn::Result WriteDataToWriteStream(MovieStreamId streamId, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        nn::Result GetWriteStreamDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result SetWriteStreamDataSize(MovieStreamId streamId, int64_t size) NN_NOEXCEPT;
        nn::Result WriteMetaToWriteStream(MovieStreamId streamId, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result GetWriteStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;

    private:
        const EnvironmentInfo* m_pEnvironmentInfo;
        ResourceIdManager* m_pResourceIdManager;
        AlbumFileLockTable* m_pReadLockTable;
        AlbumFileLockTable* m_pWriteLockTable;
        detail::MovieStreamEntryList<MovieReadStreamEntry, MovieReadStreamCountMax> m_ReadStreamEntryList;
        detail::MovieStreamEntryList<MovieWriteStreamEntry, MovieWriteStreamCountMax> m_WriteStreamEntryList;
    };

}}}}
