﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_MovieStreamManager.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../capsrvServer_ResultPrivate.h"
#include "../../capsrv_Macro.h"

#define NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(varEntry, varEntryIndex, StreamId)  \
    int varEntryIndex = -1;                                                         \
    NN_RESULT_DO(m_ReadStreamEntryList.FindEntryIndex(                              \
        &varEntryIndex,                                                             \
        [&](const MovieReadStreamEntry& e){                                         \
            return e.GetStreamId() == StreamId;                                     \
        }                                                                           \
    ));                                                                             \
    auto& varEntry = m_ReadStreamEntryList.GetEntry(varEntryIndex);

#define NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(varEntry, varEntryIndex, StreamId) \
    int varEntryIndex = -1;                                                         \
    NN_RESULT_DO(m_WriteStreamEntryList.FindEntryIndex(                             \
        &varEntryIndex,                                                             \
        [&](const MovieWriteStreamEntry& e){                                        \
            return e.GetStreamId() == StreamId;                                     \
        }                                                                           \
    ));                                                                             \
    auto& varEntry = m_WriteStreamEntryList.GetEntry(varEntryIndex);


namespace nn{ namespace capsrv{ namespace server{ namespace album{

    MovieStreamManager::MovieStreamManager() NN_NOEXCEPT
    {
        m_pEnvironmentInfo = nullptr;
        m_pResourceIdManager = nullptr;
    }

    void MovieStreamManager::Initialize(
        const EnvironmentInfo* pEnvInfo,
        ResourceIdManager* pResourceIdMgr,
        AlbumFileLockTable* pReadLockTable,
        AlbumFileLockTable* pWriteLockTable
    ) NN_NOEXCEPT
    {
        m_ReadStreamEntryList.Initialize();
        m_WriteStreamEntryList.Initialize();

        m_pEnvironmentInfo = pEnvInfo;
        m_pResourceIdManager = pResourceIdMgr;
        m_pReadLockTable = pReadLockTable;
        m_pWriteLockTable = pWriteLockTable;
    }

    void MovieStreamManager::Finalize() NN_NOEXCEPT
    {
        m_ReadStreamEntryList.Finalize();
        m_WriteStreamEntryList.Finalize();

        m_pEnvironmentInfo = nullptr;
        m_pResourceIdManager = nullptr;
        m_pReadLockTable = nullptr;
        m_pWriteLockTable = nullptr;
    }

    //----------------------------------------------------------------

    void MovieStreamManager::NotifyStorageUnmounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        m_WriteStreamEntryList.ForeachEntry([&](MovieWriteStreamEntry& e)->nn::Result
            {
                e.NotifyStorageUnmounted(storage);
                NN_RESULT_SUCCESS;
            }
        );
        m_ReadStreamEntryList.ForeachEntry([&](MovieReadStreamEntry& e)->nn::Result
            {
                e.NotifyStorageUnmounted(storage);
                NN_RESULT_SUCCESS;
            }
        );

        // ストレージのアンマウントでは nn::fs レイヤのファイルはクローズするものの
        // MovieStream としては生きているのでファイルロックを操作しない。
    }

    //----------------------------------------------------------------

    nn::Result MovieStreamManager::OpenReadStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutStreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(sessionId, SessionId::GetInvalidValue());

        NN_CAPSRV_PROCESS_START();

        int entryIndex = -1;
        NN_RESULT_DO(m_ReadStreamEntryList.AcquireEntryIndex(&entryIndex));
        NN_CAPSRV_PROCESS_ROLLBACK(m_ReadStreamEntryList.ReleaseEntryIndex(entryIndex));

        auto streamId = MovieStreamId(m_pResourceIdManager->AcquireResourceId());
        NN_CAPSRV_PROCESS_ROLLBACK(m_pResourceIdManager->ReleaseResourceId(streamId.id));

        auto& entry = m_ReadStreamEntryList.GetEntry(entryIndex);

        NN_RESULT_DO(entry.OpenFile(sessionId, streamId, fileId, workMemory, workMemorySize, *m_pEnvironmentInfo));
        NN_CAPSRV_PROCESS_ROLLBACK(entry.CloseFile());

        NN_RESULT_DO(m_pReadLockTable->LockFile(fileId));
        NN_CAPSRV_PROCESS_ROLLBACK(m_pReadLockTable->UnlockFile(fileId));

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutStreamId   = streamId;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::CloseReadStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        auto fileId = entry.GetAlbumFileId();
        entry.CloseFile();
        m_pResourceIdManager->ReleaseResourceId(streamId.id);
        m_ReadStreamEntryList.ReleaseEntryIndex(entryIndex);
        (void)m_pReadLockTable->UnlockFile(fileId);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::ReadMovieDataFromReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        size_t readSize = 0;
        NN_RESULT_DO(entry.ReadMovieData(&readSize, buffer, size, offset, *m_pEnvironmentInfo));
        *pOutReadSize = readSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetReadStreamMovieDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        int64_t value = {};
        NN_RESULT_DO(entry.GetMovieDataSize(&value));
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::ReadImageDataFromReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        size_t readSize = 0;
        NN_RESULT_DO(entry.ReadImageData(&readSize, buffer, size, offset, *m_pEnvironmentInfo, workMemory, workMemorySize));
        *pOutReadSize = readSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetReadStreamImageDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        int64_t value = {};
        NN_RESULT_DO(entry.GetImageDataSize(&value));
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetReadStreamFileId(AlbumFileId* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        *pOutValue = entry.GetAlbumFileId();
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetReadStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_READSTREAM_BY_STREAMID(entry, entryIndex, streamId);
        *pOutValue = entry.GetBrokenReason();
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------------

    nn::Result MovieStreamManager::OpenWriteStream(MovieStreamId* pOutStreamId, AlbumCacheDelta* pOutCacheDelta, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutStreamId);
        NN_SDK_REQUIRES_NOT_EQUAL(sessionId, SessionId::GetInvalidValue());

        NN_CAPSRV_PROCESS_START();

        int entryIndex = -1;
        NN_RESULT_DO(m_WriteStreamEntryList.AcquireEntryIndex(&entryIndex));
        NN_CAPSRV_PROCESS_ROLLBACK(m_WriteStreamEntryList.ReleaseEntryIndex(entryIndex));

        auto streamId = MovieStreamId(m_pResourceIdManager->AcquireResourceId());
        NN_CAPSRV_PROCESS_ROLLBACK(m_pResourceIdManager->ReleaseResourceId(streamId.id));

        auto& entry = m_WriteStreamEntryList.GetEntry(entryIndex);

        AlbumCacheDelta delta = {};
        NN_RESULT_DO(entry.OpenFile(&delta, sessionId, streamId, fileId, *m_pEnvironmentInfo));
        NN_CAPSRV_PROCESS_ROLLBACK(entry.DiscardFile(&delta, false, *m_pEnvironmentInfo));

        NN_RESULT_DO(m_pWriteLockTable->LockFile(fileId));
        NN_CAPSRV_PROCESS_ROLLBACK(m_pWriteLockTable->UnlockFile(fileId));

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutStreamId   = streamId;
        *pOutCacheDelta = delta;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::StartWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.StartDataSection());
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::EndWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.EndDataSection());
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::StartWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.StartMetaSection());
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::EndWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.EndMetaSection());
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::FinishWriteStream(MovieStreamId streamId, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.FinishFile(workMemory, workMemorySize));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::CommitWriteStream(MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        auto fileId = entry.GetAlbumFileId();
        NN_RESULT_DO(entry.CommitFile());
        m_pResourceIdManager->ReleaseResourceId(streamId.id);
        m_WriteStreamEntryList.ReleaseEntryIndex(entryIndex);
        (void)m_pWriteLockTable->UnlockFile(fileId);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::DiscardWriteStream(AlbumCacheDelta* pOutCacheDelta, MovieStreamId streamId, bool isDeleteSuppressed) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        auto fileId = entry.GetAlbumFileId();
        AlbumCacheDelta delta = {};
        entry.DiscardFile(&delta, isDeleteSuppressed, *m_pEnvironmentInfo);
        m_pResourceIdManager->ReleaseResourceId(streamId.id);
        m_WriteStreamEntryList.ReleaseEntryIndex(entryIndex);
        (void)m_pWriteLockTable->UnlockFile(fileId);
        *pOutCacheDelta = delta;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::ReadDataFromWriteStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        size_t readSize = 0;
        NN_RESULT_DO(entry.ReadData(&readSize, buffer, size, offset));
        *pOutReadSize = readSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::WriteDataToWriteStream(MovieStreamId streamId, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.WriteData(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetWriteStreamDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        int64_t value = {};
        NN_RESULT_DO(entry.GetDataSize(&value));
        *pOutValue = value;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::SetWriteStreamDataSize(MovieStreamId streamId, int64_t size) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.SetDataSize(size));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::WriteMetaToWriteStream(MovieStreamId streamId, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        NN_RESULT_DO(entry.WriteMeta(buffer, size, makerNoteVersion, makerNoteOffset, makerNoteSize, workMemory, workMemorySize));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieStreamManager::GetWriteStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT
    {
        NN_CAPSRV_LOOKUP_WRITESTREAM_BY_STREAMID(entry, entryIndex, streamId);
        *pOutValue = entry.GetBrokenReason();
        NN_RESULT_SUCCESS;
    }

}}}}

