﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_MovieReadStreamEntry.h"

#include <limits>
#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntUtil.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/movie/capsrv_MovieConfig.h>
#include "../../capsrv_Macro.h"
#include "../detail/capsrvServer_AlbumContentsAttribute.h"
#include "../albumsrv/capsrvServer_AlbumServerObject.h"
#include "capsrvServer_AlbumMovieUtility.h"
#include "capsrvServer_VerifyMovieFileData.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    MovieReadStreamEntry::MovieReadStreamEntry() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
        NN_SDK_ASSERT_EQUAL(m_State, MovieReadStreamState_Unused);
    }

    //--------------------------------------------------------

    MovieReadStreamState MovieReadStreamEntry::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    SessionId MovieReadStreamEntry::GetClientSessionId() const NN_NOEXCEPT
    {
        return m_ClientSessionId;
    }

    MovieStreamId MovieReadStreamEntry::GetStreamId() const NN_NOEXCEPT
    {
        return m_StreamId;
    }

    const AlbumFileId& MovieReadStreamEntry::GetAlbumFileId() const NN_NOEXCEPT
    {
        return m_FileId;
    }

    bool MovieReadStreamEntry::IsBroken() const NN_NOEXCEPT
    {
        return m_State == MovieReadStreamState_Broken;
    }

    nn::Result MovieReadStreamEntry::GetBrokenReason() const NN_NOEXCEPT
    {
        return m_BrokenReason;
    }

    //--------------------------------------------------------

    void MovieReadStreamEntry::NotifyStorageUnmounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        if(m_FileId.storage == storage)
        {
            CloseFileInternal();
            if(!IsBroken())
            {
                MarkAsBroken(ResultAlbumIsNotMounted());
            }
        }
    }

    void MovieReadStreamEntry::CloseFileInternal() NN_NOEXCEPT
    {
        if(m_IsFileOpened)
        {
            AlbumFileManipulator::CloseAlbumFile(m_FileHandle);
            TemporaryFileManipulator::DiscardTemporaryFile(m_DataHashFileHandle);
            m_FileHandle = {};
            m_DataHashFileHandle = {};
            m_IsFileOpened = false;
        }
    }

    void MovieReadStreamEntry::MarkAsBroken(nn::Result brokenReason) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_State != MovieReadStreamState_Unused);
        NN_SDK_REQUIRES(m_State != MovieReadStreamState_Broken);
        NN_SDK_REQUIRES(m_BrokenReason.IsSuccess());
        m_State = MovieReadStreamState_Broken;
        m_BrokenReason = brokenReason;
    }

    nn::Result MovieReadStreamEntry::MarkAsBrokenIfFailure(nn::Result result) NN_NOEXCEPT
    {
        if(result.IsFailure())
        {
            MarkAsBroken(result);
        }
        return result;
    }

    //--------------------------------------------------------
    // 状態遷移

    nn::Result MovieReadStreamEntry::OpenFile(
        SessionId clientSessionId,
        MovieStreamId streamId,
        const AlbumFileId& fileId,
        void* workMemory,
        size_t workMemorySize,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, MovieReadStreamState_Unused);
        NN_SDK_REQUIRES(!m_IsFileOpened);
        NN_SDK_REQUIRES(m_BrokenReason.IsSuccess());

        NN_CAPSRV_PROCESS_START();

        NN_RESULT_THROW_UNLESS(fileId.contents == nn::capsrv::AlbumFileContents_Movie || fileId.contents == nn::capsrv::AlbumFileContents_ExtraMovie, ResultAlbumInvalidFileContents());

        // 一時ファイルをオープン
        TemporaryFileHandle hHashFile = {};
        NN_RESULT_DO(TemporaryFileManipulator::OpenNewTemporaryFile(&hHashFile, 0));
        NN_CAPSRV_PROCESS_ROLLBACK(TemporaryFileManipulator::DiscardTemporaryFile(hHashFile));

        AlbumFilePath filepath = {};
        AlbumFileHandle hFile = {};
        NN_RESULT_DO(AlbumFileManipulator::GetFilePath(&filepath, fileId, AlbumStorageDirection_Source, env));
        NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForLoad(&hFile, filepath));
        NN_CAPSRV_PROCESS_ROLLBACK(AlbumFileManipulator::CloseAlbumFile(hFile));

        int64_t fileSize = 0;
        NN_RESULT_DO(AlbumFileManipulator::GetFileSize(&fileSize, hFile));
        // 大きすぎるファイルは見つからない扱い
        NN_RESULT_THROW_UNLESS(fileSize <= nn::capsrv::server::detail::AlbumContentsAttribute::GetFileSizeLimit(fileId.contents, env), ResultAlbumFileNotFound());

        // オープン時のチェック
        // - ハッシュ領域を検証して一時ファイルにコピー
        // - メタ領域を検証
        int64_t movieDataBlockCount = 0;
        int64_t movieDataOffset = 0;
        int64_t movieDataSize   = 0;
        int64_t imageDataOffset = 0;
        int64_t imageDataSize   = 0;
        MetaHashList metaHashList = {};
        NN_RESULT_DO(VerifyMovieFileSignAndMeta(
            &movieDataBlockCount,
            &movieDataOffset,
            &movieDataSize,
            &imageDataOffset,
            &imageDataSize,
            &metaHashList,
            hFile,
            fileSize,
            hHashFile,
            env,
            workMemory,
            workMemorySize
        ));
        NN_SDK_ASSERT_MINMAX(movieDataBlockCount, 0, MovieDataBlockCountLimit);

        NN_CAPSRV_PROCESS_SUCCESS();
        m_State = MovieReadStreamState_Reading;
        m_BrokenReason = nn::ResultSuccess();
        m_ClientSessionId = clientSessionId;
        m_StreamId = streamId;
        m_FileId = fileId;
        m_FileHandle = hFile;
        m_IsFileOpened = true;
        m_MetaHashList = metaHashList;
        m_MovieDataBlockCount = movieDataBlockCount;
        m_MovieDataOffset = movieDataOffset;
        m_MovieDataSize = movieDataSize;
        m_ImageDataOffset = imageDataOffset;
        m_ImageDataSize = imageDataSize;
        m_DataHashFileHandle = hHashFile;
        NN_RESULT_SUCCESS;
    }

    void MovieReadStreamEntry::CloseFile() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieReadStreamState_Unused);

        CloseFileInternal();

        m_State = MovieReadStreamState_Unused;
        m_BrokenReason = nn::ResultSuccess();
        m_ClientSessionId = {};
        m_StreamId = {};
        m_FileId = {};
        m_MetaHashList = {};
        m_MovieDataBlockCount = 0;
        m_MovieDataOffset = 0;
        m_MovieDataSize = 0;
        m_ImageDataOffset = 0;
        m_ImageDataSize = 0;
    }

    //--------------------------------------------------------
    // ファイル操作

    nn::Result MovieReadStreamEntry::ReadMovieData(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset, const EnvironmentInfo& env) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieReadStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieReadStreamState_Reading,
            ResultAlbumDenied()
        );

        NN_CAPSRV_LOG_MOVIE_DEV("reading movie data(offset=%llu, size=%llu)\n", offset, size);

        // アクセスが UnitSize 単位になっているかチェック
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(size) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());
        NN_RESULT_THROW_UNLESS(static_cast<int64_t>(offset) % MovieDataBlockSize == 0, ResultAlbumOutOfRange());

        NN_CAPSRV_PROCESS_START();
        std::memset(buffer, 0, size);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(buffer, 0, size));

        if(size == 0)
        {
            *pOutReadSize = 0;
            NN_RESULT_SUCCESS;
        }

        // オーバーフローチェック
        //   0 <= offset
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultAlbumOutOfRange());
        //   0 < size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(size), ResultAlbumOutOfRange());
        const int64_t sSize = static_cast<int64_t>(size);
        NN_SDK_ASSERT(sSize > 0);
        //   0 < offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(sSize <= std::numeric_limits<int64_t>::max() - offset, ResultAlbumOutOfRange());
        //   0 < DataOffset + offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(m_MovieDataOffset <= std::numeric_limits<int64_t>::max() - offset - sSize, ResultAlbumOutOfRange());

        // ファイルの範囲内は FS から読み込む
        // パディング領域まで読み込む。
        // 範囲外はゼロ埋めして読めたことにする
        int64_t paddedSize = m_ImageDataOffset - m_MovieDataOffset;
        NN_RESULT_THROW_UNLESS(paddedSize >= m_MovieDataSize, ResultInternalError());
        NN_RESULT_THROW_UNLESS(paddedSize <= ((m_MovieDataSize + MovieDataBlockSize - 1) / MovieDataBlockSize) * MovieDataBlockSize, ResultInternalError());
        if(offset < paddedSize)
        {
            size_t sizeToRead = static_cast<size_t>(std::min(sSize, paddedSize - offset));
            size_t readSize = 0;
            NN_RESULT_DO(MarkAsBrokenIfFailure(
                AlbumFileManipulator::ReadFile(&readSize, m_FileHandle, m_MovieDataOffset + offset, buffer, sizeToRead)
            ));
            if(readSize < size)
            {
                std::memset(reinterpret_cast<char*>(buffer) + readSize, 0, size - readSize);
            }
        }
        else
        {
            std::memset(buffer, 0, size);
        }

        // 読込んだデータの正当性を確認
        //   （特に SD カードの）ファイルの内容は任意タイミングで外部から書き換えられる可能性があるので
        //   たとえ一度検査したことがある場所でも読込む度にチェックする
        int64_t blockIndexBase  = offset / MovieDataBlockSize;
        int64_t blockIndexCount = sSize / MovieDataBlockSize;
        for(int64_t i = 0; i < blockIndexCount; i++)
        {
            auto pBlock = reinterpret_cast<const char*>(buffer) + MovieDataBlockSize * i;
            int64_t blockIndex = blockIndexBase + i;

            AlbumMovieUtility::BlockHash hash = {};
            if(blockIndex >= 0 && blockIndex < m_MovieDataBlockCount)
            {
                // ハッシュを読み込み
                size_t readSize = 0;
                int64_t hashOffset = MovieDataBlockHashSize * blockIndex;
                NN_RESULT_DO(TemporaryFileManipulator::ReadFile(&readSize, m_DataHashFileHandle, hashOffset, &hash, MovieDataBlockHashSize));
                NN_SDK_ASSERT_EQUAL(readSize, MovieDataBlockHashSize);
            }
            else
            {
                // 範囲外ならゼロ埋め
                hash = g_AlbumMovieUtilityStaticObject.GetZeroBlockHash();
            }

            // ハッシュを検査
            NN_RESULT_TRY(AlbumMovieUtility::ValidateBlockData(pBlock, MovieDataBlockSize, hash))
                NN_RESULT_CATCH(ResultInternalFileDataVerificationInconsistentBlockHash)
                {
                    if(env.IsAlbumMovieFileHashVerificationEnabled())
                    {
                        NN_RESULT_RETHROW;
                    }
                    else
                    {
                        NN_CAPSRV_LOG_WARN("movie data hash verification is disabled\n");
                    }
                }
            NN_RESULT_END_TRY;
        }

        NN_CAPSRV_LOG_MOVIE_DEV("-> success\n");
        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutReadSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieReadStreamEntry::GetMovieDataSize(int64_t* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieReadStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieReadStreamState_Reading,
            ResultAlbumDenied()
        );

        *pOutValue = m_MovieDataSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieReadStreamEntry::ReadImageData(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset, const EnvironmentInfo& env, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieReadStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieReadStreamState_Reading,
            ResultAlbumDenied()
        );

        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, MovieDataBlockSize);
        NN_RESULT_THROW_UNLESS(workMemory != nullptr, ResultInternalError());
        NN_RESULT_THROW_UNLESS(workMemorySize >= MovieDataBlockSize, ResultInternalError());

        NN_CAPSRV_LOG_MOVIE_DEV("reading image data(offset=%llu, size=%llu)\n", offset, size);

        NN_CAPSRV_PROCESS_START();
        std::memset(buffer, 0, size);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(buffer, 0, size));

        if(size == 0)
        {
            *pOutReadSize = 0;
            NN_RESULT_SUCCESS;
        }

        // オーバーフローチェック
        //   0 <= offset
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultAlbumOutOfRange());
        //   0 < size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(size), ResultAlbumOutOfRange());
        const int64_t sSize = static_cast<int64_t>(size);
        NN_SDK_ASSERT(sSize > 0);
        //   0 < offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(sSize <= std::numeric_limits<int64_t>::max() - offset, ResultAlbumOutOfRange());
        //   0 < DataOffset + offset + size <= INT64_MAX
        NN_RESULT_THROW_UNLESS(m_ImageDataOffset <= std::numeric_limits<int64_t>::max() - offset - sSize, ResultAlbumOutOfRange());

        char* pBlockData = reinterpret_cast<char*>(workMemory);
        size_t blockDataSize = MovieDataBlockSize;

        // ファイルの範囲内は一時ファイルから読み込む
        // 範囲外はゼロ埋めして読めたことにする
        {
            ptrdiff_t posInOutBuf = 0;
            int64_t posInImageData = offset;
            int64_t remain = sSize;
            while(remain > 0)
            {
                NN_STATIC_ASSERT(MovieMetaDataSize % MovieDataBlockSize == 0);
                if(posInImageData < m_ImageDataSize)
                {
                    int64_t blockIndex = posInImageData / MovieDataBlockSize;
                    NN_SDK_ASSERT_RANGE(blockIndex, 0, MetaHashList::Length);

                    // ファイルから読込
                    size_t readSize = 0;
                    NN_RESULT_DO(MarkAsBrokenIfFailure(
                        AlbumFileManipulator::ReadFile(&readSize, m_FileHandle, m_ImageDataOffset + blockIndex * MovieDataBlockSize, pBlockData, blockDataSize)
                    ));
                    NN_RESULT_THROW_UNLESS(readSize == MovieDataBlockSize, ResultAlbumInvalidFileData());

                    // ハッシュを検査
                    NN_RESULT_TRY(AlbumMovieUtility::ValidateBlockData(pBlockData, blockDataSize, m_MetaHashList.hashList[blockIndex]))
                        NN_RESULT_CATCH(ResultInternalFileDataVerificationInconsistentBlockHash)
                        {
                            if(env.IsAlbumMovieFileHashVerificationEnabled())
                            {
                                NN_RESULT_RETHROW;
                            }
                            else
                            {
                                NN_CAPSRV_LOG_WARN("movie meta hash verification is disabled\n");
                            }
                        }
                    NN_RESULT_END_TRY;

                    // 出力バッファにコピー
                    int64_t posInBlock = posInImageData % MovieDataBlockSize;
                    int64_t sizeToCopy = remain;
                    sizeToCopy = std::min(sizeToCopy, MovieDataBlockSize - posInBlock);
                    sizeToCopy = std::min(sizeToCopy, m_ImageDataSize - posInImageData);
                    std::memcpy(reinterpret_cast<char*>(buffer) + posInOutBuf, pBlockData + posInBlock, sizeToCopy);

                    posInOutBuf  += static_cast<ptrdiff_t>(sizeToCopy);
                    posInImageData += sizeToCopy;
                    remain  -= sizeToCopy;
                    NN_SDK_ASSERT_GREATER_EQUAL(remain, 0);
                }
                else
                {
                    std::memset(reinterpret_cast<char*>(buffer) + posInOutBuf, 0, remain);
                    break;
                }
            }
        }

        NN_CAPSRV_LOG_MOVIE_DEV("-> success\n");
        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutReadSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieReadStreamEntry::GetImageDataSize(int64_t* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, MovieReadStreamState_Unused);
        NN_RESULT_THROW_UNLESS(
            m_State == MovieReadStreamState_Reading,
            ResultAlbumDenied()
        );

        *pOutValue = m_ImageDataSize;
        NN_RESULT_SUCCESS;
    }

}}}}
