﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_MountAlbumDirectory.h"

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_StringUtil.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../capsrvServer_Config.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace {

        // マウント名のリスト
        static const char* MountNameList[] = {
            NN_CAPSRV_MOUNT_NAME_NAND,
            NN_CAPSRV_MOUNT_NAME_SD,
        };
        // fs で定義されているストレージの識別子
        static const nn::fs::ImageDirectoryId ImageDirectoryIdList[] = {
            nn::fs::ImageDirectoryId::Nand,
            nn::fs::ImageDirectoryId::SdCard,
        };
        static const int MountNameCount = sizeof(MountNameList) / sizeof(MountNameList[0]);

        nn::Result ConvertFsResultForMountAlbum(AlbumStorageType storage, nn::Result result) NN_NOEXCEPT
        {
            NN_UNUSED(storage);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::fs::ResultMountNameAlreadyExists)
                {
                    NN_ABORT("MountAlbum: Already mounted (storage=%d)\n", storage);
                }
                NN_RESULT_CATCH(nn::fs::ResultAllocationMemoryFailed)
                {
                    NN_ABORT("MountAlbum: AllocationMemoryFailed\n");
                }
                NN_RESULT_CATCH(nn::fs::ResultSdCardAccessFailed)
                {
                    // SD カードがささっていない場合このエラー。
                    NN_SDK_ASSERT_EQUAL(storage, AlbumStorage_Sd);
                    NN_CAPSRV_LOG_INFO("Couldn't access to SD card\n");
                    NN_RESULT_THROW(ResultAlbumMountFailed());
                }
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    // SD カードに /Nintendo/Album という名前のファイルが存在すると
                    // アルバムディレクトリが作成できずにこのエラーが返ってくるらしい。
                    NN_CAPSRV_LOG_INFO("Couldn't mount Album directory (maybe Album 'file' exists)\n");
                    NN_RESULT_THROW(ResultAlbumStructureCorrupted());
                }
                NN_RESULT_CATCH_ALL
                {
                    // 不明なエラー。
                    NN_CAPSRV_LOG_DEV("Unknown MountAlbum Error: 0x%08X(%d-%d)\n", static_cast<int>(result.GetInnerValueForDebug()), result.GetModule(), result.GetDescription());
                    NN_RESULT_THROW(ResultAlbumError());
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::Result GetMountName(const char** pOutValue, AlbumStorageType storage) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(
                storage >= 0 && storage < MountNameCount,
                ResultAlbumInvalidStorage()
            );

            *pOutValue = MountNameList[storage];

            NN_RESULT_SUCCESS;
        }

        nn::Result GetImageDirectoryId(nn::fs::ImageDirectoryId* pOutValue, AlbumStorageType storage) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(
                storage >= 0 && storage < MountNameCount,
                ResultAlbumInvalidStorage()
            );

            *pOutValue = ImageDirectoryIdList[storage];

            NN_RESULT_SUCCESS;
        }

    }

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumMountFailed
    nn::Result MountAlbumDirectory(AlbumStorageType storage, const EnvironmentInfo& env) NN_NOEXCEPT
    {
        const char* p = env.GetPcHostAlbumDirectoryPath();
        if (storage == AlbumStorage_Sd && p != nullptr)
        {
            // fwdbg でアルバム保存先が PC-Host に設定されている場合、
            // "SD:" で PC-Host に接続するようにする
            auto result = fs::MountHost(NN_CAPSRV_MOUNT_NAME_SD, p);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(fs::ResultUnexpected)
                {
                    // TargetManager が未接続の場合は再マウントが必要
                    NN_CAPSRV_LOG_INFO("TargetManager may be not connected.\n");
                    NN_RESULT_THROW(ResultNeedsRetryAlbumMount());
                }
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    // fwdbg の "album_directory_path" のパスが見つからない
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAlbumPathOnPcHostNotFound());
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            NN_RESULT_END_TRY;
            NN_CAPSRV_LOG_INFO("Mounted PC-Host: '%s'\n", p);
            NN_RESULT_SUCCESS;
        }

        const char* mountName = nullptr;
        nn::fs::ImageDirectoryId id;
        NN_RESULT_DO(GetMountName(&mountName, storage));
        NN_RESULT_DO(GetImageDirectoryId(&id, storage));

        NN_RESULT_DO(ConvertFsResultForMountAlbum(storage,
            nn::fs::MountImageDirectory(mountName, id)
        ));
        NN_CAPSRV_LOG_INFO("Mounted '%s'\n", mountName);

        NN_RESULT_SUCCESS;
    }

    nn::Result UnmountAlbumDirectory(AlbumStorageType storage, const EnvironmentInfo& env) NN_NOEXCEPT
    {
        const char* p = env.GetPcHostAlbumDirectoryPath();
        if (storage == AlbumStorage_Sd && p != nullptr)
        {
            fs::Unmount(NN_CAPSRV_MOUNT_NAME_SD);
            NN_CAPSRV_LOG_INFO("Unmounted PC-Host: '%s'\n", p);
            NN_RESULT_SUCCESS;
        }

        const char* mountName = nullptr;
        NN_RESULT_DO(GetMountName(&mountName, storage));

        nn::fs::Unmount(mountName);
        NN_CAPSRV_LOG_INFO("Unmounted '%s'\n", mountName);

        NN_RESULT_SUCCESS;
    }

}}}}
