﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ExtractJpegThumbnail.h"

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include "../capsrvServer_ResultPrivate.h"
#include "../capsrvServer_Config.h"
#include "../detail/capsrvServer_ExifWorkStorage.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    nn::Result ExtractJpegThumbnail(
        size_t*     pOutSize,
        void*       pOutBuffer,
        size_t      bufferSize,
        const void* pJpegHeadData,
        size_t      jpegHeadDataSize
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT_NOT_NULL(pOutBuffer);
        NN_SDK_ASSERT_GREATER(bufferSize, 0u);
        NN_SDK_ASSERT_NOT_NULL(pJpegHeadData);

        const void* pExif = nullptr;
        size_t exifSize = 0;
        NN_RESULT_THROW_UNLESS(
            nn::image::JpegStatus_Ok == nn::image::JpegDecoder::GetExifData(&pExif, &exifSize, pJpegHeadData, jpegHeadDataSize),
            ResultInternalFileDataVerificationExifExtractionFailed()
        );

        detail::ExifExtractWorkStorage work;
        nn::image::ExifExtractor extractor(&work, sizeof(work));
        extractor.SetExifData(pExif, exifSize);
        NN_RESULT_THROW_UNLESS(
            nn::image::JpegStatus_Ok == extractor.Analyze(),
            ResultInternalFileDataVerificationExifAnalyzationFailed()
        );

        const void* pThumb = nullptr;
        size_t thumbSize = 0;
        pThumb = extractor.ExtractThumbnail(&thumbSize);
        NN_RESULT_THROW_UNLESS(
            pThumb != nullptr && thumbSize > 0,
            ResultAlbumFileNoThumbnail()
        );

        // サムネイルが JPEG ファイルかどうかの確認
        nn::image::JpegDecoder decoder;
        decoder.SetImageData(pThumb, thumbSize);
        NN_RESULT_THROW_UNLESS(
            decoder.Analyze() == nn::image::JpegStatus_Ok,
            ResultAlbumInvalidFileData()
        );

        // サムネイルのサイズが正しいかどうかの確認
        nn::image::Dimension dim = decoder.GetAnalyzedDimension();
        NN_RESULT_THROW_UNLESS(
            dim.width == nn::capsrv::ViewerThumbnailImageSize_Width && dim.height == nn::capsrv::ViewerThumbnailImageSize_Height,
            ResultAlbumInvalidFileData()
        );

        // 渡したバッファに収まっているか確認
        NN_RESULT_THROW_UNLESS(
            reinterpret_cast<const char*>(pThumb) + thumbSize <= reinterpret_cast<const char*>(pJpegHeadData) + jpegHeadDataSize,
            ResultAlbumInvalidFileData()
        );

        // 書込み先に入りきるか確認
        NN_RESULT_THROW_UNLESS(
            thumbSize <= bufferSize,
            ResultAlbumReadBufferShortage()
        );

        std::memcpy(pOutBuffer, pThumb, thumbSize);
        *pOutSize = thumbSize;

        NN_RESULT_SUCCESS;
    }

}}}}
