﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumPathUtility.h"

#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <cctype>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/capsrv/capsrv_Result.h>
#include "../detail/capsrvServer_EncryptApplicationId.h"
#include "../detail/capsrvServer_AlbumContentsAttribute.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    const int AlbumPathUtility::PathLength;
    const int AlbumPathUtility::PathSize;
    const int AlbumPathUtility::ExtraPathSize;

    int AlbumPathUtility::GetSubDirectoryNameLengthForRegular(int depth) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(SubDirectoryDepth));
        const int List[SubDirectoryDepth] =
        {
            SubDirectoryNameLength0,
            SubDirectoryNameLength1,
            SubDirectoryNameLength2
        };
        return List[depth];
    }

    int AlbumPathUtility::GetSubDirectoryNameLengthForExtra(int depth) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(ExtraSubDirectoryDepth));
        const int List[ExtraSubDirectoryDepth] =
        {
            ExtraSubDirectoryNameLength0,
            ExtraSubDirectoryNameLength1,
            ExtraSubDirectoryNameLength2,
            ExtraSubDirectoryNameLength3,
            ExtraSubDirectoryNameLength4
        };
        return List[depth];
    }

    int AlbumPathUtility::GetSubDirectoryNameLength(int depth, bool isExtra) NN_NOEXCEPT
    {
        if (isExtra)
        {
            return GetSubDirectoryNameLengthForExtra(depth);
        }
        else
        {
            return GetSubDirectoryNameLengthForRegular(depth);
        }
    }

    int AlbumPathUtility::GetSubDirectoryPathLengthForRegular(int depth) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(SubDirectoryDepth));
        const int List[SubDirectoryDepth] =
        {
            SubDirectoryPathLength0,
            SubDirectoryPathLength1,
            SubDirectoryPathLength2
        };
        return List[depth];
    }

    int AlbumPathUtility::GetSubDirectoryPathLengthForExtra(int depth) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(ExtraSubDirectoryDepth));
        const int List[ExtraSubDirectoryDepth] =
        {
            ExtraSubDirectoryPathLength0,
            ExtraSubDirectoryPathLength1,
            ExtraSubDirectoryPathLength2,
            ExtraSubDirectoryPathLength3,
            ExtraSubDirectoryPathLength4,
        };
        return List[depth];
    }

    int AlbumPathUtility::GetSubDirectoryPathLength(int depth, bool isExtra) NN_NOEXCEPT
    {
        if (isExtra)
        {
            return GetSubDirectoryPathLengthForExtra(depth);
        }
        else
        {
            return GetSubDirectoryPathLengthForRegular(depth);
        }
    }

    //----------------------------------------------
    // 検査関数
    //----------------------------------------------

    nn::Result AlbumPathUtility::ValidateFileId(
        const AlbumFileId* pFileId,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileId);
        NN_RESULT_DO(ValidateApplicationId(pFileId->applicationId));
        NN_RESULT_DO(ValidateTime(&pFileId->time));
        NN_RESULT_DO(ValidateStorage(pFileId->storage));
        NN_RESULT_DO(ValidateFileContents(pFileId->contents, env));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateSubDirectoryId(
        const AlbumFileId* pDirectoryId,
        int depth,
        bool isExtra
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryId);
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(ExtraSubDirectoryDepth));

        NN_RESULT_DO(ValidateStorage(pDirectoryId->storage));
        if (isExtra)
        {
            if(depth >= 1)
            {
                NN_RESULT_DO(ValidateApplicationId(pDirectoryId->applicationId));
            }
            if(depth >= 2)
            {
                NN_RESULT_DO(ValidateYear(pDirectoryId->time.year));
            }
            if(depth >= 3)
            {
                NN_RESULT_DO(ValidateMonth(pDirectoryId->time.month));
            }
            if(depth >= 4)
            {
                NN_RESULT_DO(ValidateDay(pDirectoryId->time.day));
            }
        }
        else
        {
            if(depth >= 0)
            {
                NN_RESULT_DO(ValidateYear(pDirectoryId->time.year));
            }
            if(depth >= 1)
            {
                NN_RESULT_DO(ValidateMonth(pDirectoryId->time.month));
            }
            if(depth >= 2)
            {
                NN_RESULT_DO(ValidateDay(pDirectoryId->time.day));
            }
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateYear(uint16_t value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            value >= 0 && value <= 9999, // 4 桁以下なら OK
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateMonth(uint8_t value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            value >= 1 && value <= 12,
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateDay(uint8_t value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            value >= 1 && value <= 31,
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateTime(
        const AlbumFileDateTime* pTime
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTime);

        NN_RESULT_DO(ValidateYear(pTime->year));
        NN_RESULT_DO(ValidateMonth(pTime->month));
        NN_RESULT_DO(ValidateDay(pTime->day));

        NN_RESULT_THROW_UNLESS(
            pTime->hour >= 0 && pTime->hour <= 24, // 24 にはならないかもしれないが…
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_THROW_UNLESS(
            pTime->minute >= 0 && pTime->minute <= 60, // 60 にはならないかもしれないが…
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_THROW_UNLESS(
            pTime->second >= 0 && pTime->second <= 60, // うるう秒？
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_THROW_UNLESS(
            pTime->id >= 0 && pTime->id <= 99,
            ResultAlbumInvalidTimestamp()
        );

        // 作成日時が nn::time で扱えないものは不正とする
        NN_RESULT_THROW_UNLESS(pTime->IsValidDateTime(), ResultAlbumInvalidTimestamp());
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateApplicationId(
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            applicationId.value != ncm::ApplicationId::GetInvalidId().value,
            ResultAlbumInvalidTimestamp()
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateStorage(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            storage >= 0 && storage < AlbumStorageCount,
            ResultAlbumInvalidStorage()
        );
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ValidateFileContents(
        AlbumFileContentsType contents,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(env.IsAlbumFileContentsSupported(contents), ResultAlbumInvalidFileContents());
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------
    // パス生成関数
    //----------------------------------------------
    void AlbumPathUtility::GetFilename(
        int* pOutPathLength,
        char* pOutPath,
        size_t outPathBufferSize,
        const AlbumFileDateTime* pTime,
        nn::ncm::ApplicationId applicationId,
        AlbumFileContentsType contents,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPathLength);
        NN_SDK_REQUIRES_NOT_NULL(pOutPath);
        NN_SDK_REQUIRES_GREATER_EQUAL(outPathBufferSize, static_cast<size_t>(ExtraFilenameSize));
        NN_SDK_REQUIRES(ValidateTime(pTime).IsSuccess());
        NN_SDK_REQUIRES(ValidateApplicationId(applicationId).IsSuccess());
        NN_SDK_REQUIRES(ValidateFileContents(contents, env).IsSuccess());

        const char* extension = detail::AlbumContentsAttribute::GetFileExtension(contents, env);
        NN_SDK_ASSERT_NOT_NULL(extension);

        detail::EncryptedApplicationId encryptedApplicationId = detail::EncryptApplicationId(applicationId, IsExtraAlbumFileContents(contents));
        char encryptedApplicationIdStr[2 * sizeof(encryptedApplicationId) + 1];
        {
            char* p = encryptedApplicationIdStr;
            char* pEnd = encryptedApplicationIdStr + sizeof(encryptedApplicationIdStr);
            for(int i = 0; i < sizeof(encryptedApplicationId); i++, p += 2)
            {
                int val = static_cast<int>(static_cast<uint8_t>(encryptedApplicationId.value[i]));
                int l = nn::util::SNPrintf(p, static_cast<size_t>(pEnd - p), "%02X", val);
                NN_UNUSED(l);
                NN_SDK_ASSERT_EQUAL(l, 2);
            }
        }

        int length;
        if (IsExtraAlbumFileContents(contents))
        {
            length = nn::util::SNPrintf(pOutPath, outPathBufferSize, "%04d%02d%02d%02d%02d%02d%02d-%s%c%s",
                static_cast<int>(pTime->year),
                static_cast<int>(pTime->month),
                static_cast<int>(pTime->day),
                static_cast<int>(pTime->hour),
                static_cast<int>(pTime->minute),
                static_cast<int>(pTime->second),
                static_cast<int>(pTime->id),
                encryptedApplicationIdStr,
                'X',
                extension
                );
            NN_SDK_ASSERT_EQUAL(length, static_cast<int>(ExtraFilenameLength));
        }
        else
        {
            length = nn::util::SNPrintf(pOutPath, outPathBufferSize, "%04d%02d%02d%02d%02d%02d%02d-%s%s",
                static_cast<int>(pTime->year),
                static_cast<int>(pTime->month),
                static_cast<int>(pTime->day),
                static_cast<int>(pTime->hour),
                static_cast<int>(pTime->minute),
                static_cast<int>(pTime->second),
                static_cast<int>(pTime->id),
                encryptedApplicationIdStr,
                extension
                );
            NN_SDK_ASSERT_EQUAL(length, static_cast<int>(FilenameLength));
        }

        *pOutPathLength = length;
    }

    namespace {
        int GetPathDelimiter(char* pOutPath, int outPathSize) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutPath);
            NN_SDK_REQUIRES_GREATER_EQUAL(outPathSize, 2);
            NN_UNUSED(outPathSize);

            pOutPath[0] = '/';
            pOutPath[1] = '\0';

            return 1;
        }

        int GetPathRoot(char* pOutPath, int outPathSize, AlbumStorageType storage) NN_NOEXCEPT
        {
            static const char* MountNames[]
            = {
                NN_CAPSRV_MOUNT_NAME_NAND ":/",
                NN_CAPSRV_MOUNT_NAME_SD ":/",
            };
            // MountNames の順番と enum の値が一致していることを確認。
            NN_STATIC_ASSERT(AlbumStorage_Nand == 0);
            NN_STATIC_ASSERT(AlbumStorage_Sd == 1);

            NN_SDK_REQUIRES_GREATER_EQUAL(outPathSize, AlbumPathUtility::RootPathLength + 1);
            NN_SDK_REQUIRES_RANGE(storage, 0, sizeof(MountNames) / sizeof(MountNames[0]));

            int length = nn::util::Strlcpy(pOutPath, MountNames[storage], outPathSize);
            NN_SDK_ASSERT(length == AlbumPathUtility::MountNameLength + 2);

            return length;
        }

        int GetSubDirectoryNameForRegular(char* pOutPath, int outPathSize, const AlbumFileId& fileId, int depth) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutPath);
            NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(AlbumPathUtility::SubDirectoryDepth));
            NN_SDK_REQUIRES_GREATER_EQUAL(outPathSize, AlbumPathUtility::GetSubDirectoryNameLengthForRegular(depth) + 1);

            int length = 0;
            switch(depth)
            {
            case 0:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateYear(fileId.time.year).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%04d", static_cast<int>(fileId.time.year));
                break;
            case 1:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateMonth(fileId.time.month).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%02d", static_cast<int>(fileId.time.month));
                break;
            case 2:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateDay(fileId.time.day).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%02d", static_cast<int>(fileId.time.day));
                break;
            default: NN_UNEXPECTED_DEFAULT;
            };

            NN_SDK_ASSERT_EQUAL(length, AlbumPathUtility::GetSubDirectoryNameLengthForRegular(depth));

            return length;
        }


        int GetSubDirectoryNameForExtra(char* pOutPath, int outPathSize, const AlbumFileId& fileId, int depth) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutPath);
            NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(AlbumPathUtility::ExtraSubDirectoryDepth));
            NN_SDK_REQUIRES_GREATER_EQUAL(outPathSize, AlbumPathUtility::GetSubDirectoryNameLengthForExtra(depth) + 1);

            int length = 0;
            switch(depth)
            {
            case 0:
                length = nn::util::SNPrintf(pOutPath, outPathSize, NN_CAPSRV_DIRECTORY_NAME_FOR_EXTRA);
                break;
            case 1:
                {
                    auto encryptedApplicationId = detail::EncryptApplicationId(fileId.applicationId);
                    char encryptedApplicationIdStr[2 * sizeof(encryptedApplicationId) + 1];
                    char* p = encryptedApplicationIdStr;
                    char* pEnd = encryptedApplicationIdStr + sizeof(encryptedApplicationIdStr);
                    for(int i = 0; i < sizeof(encryptedApplicationId); i++, p += 2)
                    {
                        int val = static_cast<int>(static_cast<uint8_t>(encryptedApplicationId.value[i]));
                        int l = nn::util::SNPrintf(p, static_cast<size_t>(pEnd - p), "%02X", val);
                        NN_UNUSED(l);
                        NN_SDK_ASSERT_EQUAL(l, 2);
                    }
                    length = nn::util::SNPrintf(pOutPath, outPathSize, "%s", encryptedApplicationIdStr);
                }
                break;
            case 2:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateYear(fileId.time.year).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%04d", static_cast<int>(fileId.time.year));
                break;
            case 3:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateMonth(fileId.time.month).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%02d", static_cast<int>(fileId.time.month));
                break;
            case 4:
                NN_SDK_REQUIRES(AlbumPathUtility::ValidateDay(fileId.time.day).IsSuccess());
                length = nn::util::SNPrintf(pOutPath, outPathSize, "%02d", static_cast<int>(fileId.time.day));
                break;
            default: NN_UNEXPECTED_DEFAULT;
            };

            NN_SDK_ASSERT_EQUAL(length, AlbumPathUtility::GetSubDirectoryNameLengthForExtra(depth));

            return length;
        }

        int GetSubDirectoryName(char* pOutPath, int outPathSize, const AlbumFileId& fileId, int depth, bool isExtra) NN_NOEXCEPT
        {
            if (isExtra)
            {
                return GetSubDirectoryNameForExtra(pOutPath, outPathSize, fileId, depth);
            }
            else
            {
                return GetSubDirectoryNameForRegular(pOutPath, outPathSize, fileId, depth);
            }
        }
    }

    void AlbumPathUtility::GetStorageRootPath(
        int* pOutPathLength,
        char* pOutPath,
        size_t outPathBufferSize,
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPathLength);
        NN_SDK_REQUIRES_NOT_NULL(pOutPath);
        NN_SDK_REQUIRES_GREATER_EQUAL(outPathBufferSize, static_cast<size_t>(ExtraPathSize));
        NN_SDK_REQUIRES(ValidateStorage(storage).IsSuccess());

        int length = 0;
        length += GetPathRoot(pOutPath, static_cast<int>(outPathBufferSize), storage);

        NN_SDK_ASSERT_EQUAL(length, static_cast<int>(RootPathLength));
        *pOutPathLength = length;
    }

    void AlbumPathUtility::GetSubDirectoryPath(
        int* pOutPathLength,
        char* pOutPath,
        size_t outPathBufferSize,
        const AlbumFileId& fileId,
        int depth,
        bool isExtra
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPathLength);
        NN_SDK_REQUIRES_NOT_NULL(pOutPath);
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(ExtraSubDirectoryDepth));

        int length = 0;
        int size = static_cast<int>(outPathBufferSize);
        char* p = pOutPath;

        length += GetPathRoot(p + length, size - length, fileId.storage);
        for(int i = 0; i <= depth; i++)
        {
            length += GetSubDirectoryName(p + length, size - length, fileId, i, isExtra);
            length += GetPathDelimiter(p + length, size - length);
        }

        NN_SDK_ASSERT_EQUAL(length, GetSubDirectoryPathLength(depth, isExtra));
        *pOutPathLength = length;
    }

    void AlbumPathUtility::GetFilepath(
        int* pOutPathLength,
        char* pOutPath,
        size_t outPathBufferSize,
        const AlbumFileId& fileId,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutPathLength);
        NN_SDK_REQUIRES_NOT_NULL(pOutPath);
        NN_SDK_REQUIRES_GREATER_EQUAL(outPathBufferSize, static_cast<size_t>(ExtraPathSize));
        NN_SDK_REQUIRES(ValidateFileId(&fileId, env).IsSuccess());

        int length = 0;
        int size = static_cast<int>(outPathBufferSize);
        char* p = pOutPath;

        // sub directory
        {
            int n = 0;
            if (IsExtraAlbumFileContents(fileId.contents))
            {
                GetSubDirectoryPath(&n, p + length, size - length, fileId, ExtraSubDirectoryDepth - 1, true);
            }
            else
            {
                GetSubDirectoryPath(&n, p + length, size - length, fileId, SubDirectoryDepth - 1, false);
            }
            length += n;
        }
        // filename
        {
            int n = 0;
            GetFilename(&n, p + length, size - length, &fileId.time, fileId.applicationId, fileId.contents, env);
            length += n;
        }

        NN_SDK_ASSERT(length == static_cast<int>(PathLength) ||
                      length == static_cast<int>(ExtraPathLength));
        *pOutPathLength = length;
    }

    //----------------------------------------------
    // パス解析関数
    //----------------------------------------------

    namespace {

        template<typename TIntType>
        bool ParseDecimal1(TIntType* pOutValue, char c) NN_NOEXCEPT
        {
            if(c < '0' || c > '9')
            {
                return false;
            }
            *pOutValue = static_cast<TIntType>(c - '0');
            return true;
        }

        template<typename TIntType>
        bool ParseDecimal(TIntType* pOutValue, const char* str, int strLength) NN_NOEXCEPT
        {
            TIntType value = 0;
            for(int i = 0; i < strLength; i++)
            {
                TIntType v;
                if(!ParseDecimal1(&v, str[i]))
                {
                    return false;
                }
                value *= 10;
                value += v;
            }
            *pOutValue = value;
            return true;
        }

        template<typename TIntType>
        bool ParseHexaUpper1(TIntType* pOutValue, char c) NN_NOEXCEPT
        {
            if(ParseDecimal1(pOutValue, c))
            {
                return true;
            }
            if(c < 'A' || c > 'F')
            {
                return false;
            }
            *pOutValue = static_cast<TIntType>(c - 'A' + 10);
            return true;
        }

        bool ParseHexaByte(const char** pOutNext, char* pOutValue, const char* str, ptrdiff_t strLength) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(strLength, 2);
            NN_UNUSED(strLength);

            uint8_t value = 0;
            for(int i = 0; i < 2; i++)
            {
                uint8_t v;
                if(!ParseHexaUpper1(&v, str[i]))
                {
                    return false;
                }
                value <<= 4;
                value += v;
            }
            *pOutValue = static_cast<char>(value);
            *pOutNext = str + 2;
            return true;
        }

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidTimestamp
        nn::Result ParseDateTime(const char** pOutNextString, AlbumFileDateTime* pOutValue, const char* str, int strLength) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(strLength, 16);
            AlbumFileDateTime time;
            const char* p = str;
            const char* const pEnd = str + strLength;
            NN_UNUSED(pEnd);
            // year
            NN_SDK_ASSERT(p + 4 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.year, p, 4),
                ResultAlbumInvalidTimestamp()
            );
            p += 4;
            // month
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.month, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;
            // day
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.day, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;
            // hour
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.hour, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;
            // minute
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.minute, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;
            // second
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.second, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;
            // id
            NN_SDK_ASSERT(p + 2 <= pEnd);
            NN_RESULT_THROW_UNLESS(
                ParseDecimal(&time.id, p, 2),
                ResultAlbumInvalidTimestamp()
            );
            p += 2;

            NN_RESULT_DO(AlbumPathUtility::ValidateTime(&time));

            *pOutValue = time;
            *pOutNextString = p;
            NN_RESULT_SUCCESS;
        }

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidApplicationId
        nn::Result ParseEncryptedApplicationId(const char** pOutNextString, detail::EncryptedApplicationId* pOutValue, const char* str, int strLength) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER_EQUAL(strLength, static_cast<int>(2 * sizeof(detail::EncryptedApplicationId)));
            detail::EncryptedApplicationId id = {};
            const char* p = str;
            const char* const pEnd = str + strLength;
            NN_UNUSED(pEnd);

            for(int i = 0; i < sizeof(detail::EncryptedApplicationId); i++)
            {
                NN_RESULT_THROW_UNLESS(
                    ParseHexaByte(&p, &id.value[i], p, pEnd - p),
                    ResultAlbumInvalidApplicationId()
                );
            }

            *pOutValue = id;
            *pOutNextString = p;
            NN_RESULT_SUCCESS;
        }

        bool MatchString(const char** pOutNextString, const char* str, int strLength, const char* pattern, int patternLength) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(nn::util::Strnlen(pattern, patternLength), patternLength);
            if(strLength < patternLength)
            {
                return false;
            }

            int result = nn::util::Strncmp(str, pattern, patternLength);
            if(result != 0)
            {
                return false;
            }

            *pOutNextString = str + patternLength;
            return true;
        }

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidFilename
        nn::Result ParseExtraChar(const char** pOutNextString, char* pOutExtraChar, const char* str, int strLength) NN_NOEXCEPT
        {
            const int patternLength = 1;
            NN_SDK_REQUIRES_GREATER_EQUAL(strLength, patternLength);

            char extraChar = static_cast<char>(std::toupper(*str));
            NN_RESULT_THROW_UNLESS(extraChar == 'X', ResultAlbumInvalidFilename());

            *pOutExtraChar  = extraChar;
            *pOutNextString = str + patternLength;
            NN_RESULT_SUCCESS;
        }

        nn::Result ParseFileExtension(
            const char** pOutNext,
            AlbumFileContentsType* pOutContents,
            const char* str,
            int strLength,
            bool isExtra,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT
        {
            AlbumFileContentsType contents = {};

            const char* p = str;
            const char* const pEnd = str + strLength;

            auto testExtension = [&env](const char** pOutNext, const char* p, const char* pEnd, AlbumFileContentsType contents) -> bool
            {
                return MatchString(pOutNext, p, static_cast<int>(pEnd - p),
                    detail::AlbumContentsAttribute::GetFileExtension(contents, env),
                    detail::AlbumContentsAttribute::GetFileExtensionLength(contents, env)
                );
            };

            if (!isExtra && env.IsAlbumScreenShotFileSupported() && testExtension(&p, p, pEnd, AlbumFileContents_ScreenShot))
            {
                NN_CAPSRV_LOG_DEV("screenshot %s\n", str);
                contents = AlbumFileContents_ScreenShot;
            }
            else if (!isExtra && env.IsAlbumMovieFileSupported() && testExtension(&p, p, pEnd, AlbumFileContents_Movie))
            {
                NN_CAPSRV_LOG_DEV("movie %s\n", str);
                contents = AlbumFileContents_Movie;
            }
            else if (isExtra && env.IsAlbumExtraScreenShotFileSupported() && testExtension(&p, p, pEnd, AlbumFileContents_ExtraScreenShot))
            {
                NN_CAPSRV_LOG_DEV("extra-screenshot %s\n", str);
                contents = AlbumFileContents_ExtraScreenShot;
            }
            else if (isExtra && env.IsAlbumExtraMovieFileSupported() && testExtension(&p, p, pEnd, AlbumFileContents_ExtraMovie))
            {
                NN_CAPSRV_LOG_DEV("extra-movie %s\n", str);
                contents = AlbumFileContents_ExtraMovie;
            }
            else
            {
                NN_CAPSRV_LOG_DEV("unknown extension %s\n", str);
                NN_RESULT_THROW(ResultAlbumInvalidFileContents());
            }

            *pOutNext = p;
            *pOutContents = contents;
            NN_RESULT_SUCCESS;
        }

        // 通常の Scoop 準拠アルバムファイル
        nn::Result ParseFilenameRegular(
            const char** pOutNext,
            const char** pOutEnd,
            AlbumFileDateTime* pOutTime,
            detail::EncryptedApplicationId* pOutEncryptedApplicationId,
            const char* filename,
            size_t filenameSize,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT
        {
            AlbumFileDateTime time = {};
            detail::EncryptedApplicationId encryptedApplicationId = {};

            const char* p = filename;
            const char* pEnd = filename + filenameSize;
            // date and time
            NN_RESULT_DO(ParseDateTime(&p, &time, p, static_cast<int>(pEnd - p)));
            // "-"
            NN_RESULT_THROW_UNLESS(
                MatchString(&p, p, static_cast<int>(pEnd - p), "-", 1),
                ResultAlbumInvalidFilename()
            );
            // encoded title id
            NN_RESULT_DO(ParseEncryptedApplicationId(&p, &encryptedApplicationId, p, static_cast<int>(pEnd - p)));

            *pOutTime = time;
            *pOutEncryptedApplicationId = encryptedApplicationId;
            *pOutNext = p;
            *pOutEnd  = pEnd;
            NN_RESULT_SUCCESS;
        }

        // Extra 属性付きのアプリ保存ファイル
        nn::Result ParseFilenameWithExtraChar(
            const char** pOutNext,
            const char** pOutEnd,
            AlbumFileDateTime* pOutTime,
            detail::EncryptedApplicationId* pOutEncryptedApplicationId,
            char* pOutExtra,
            const char* filename,
            size_t filenameSize,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT
        {
            AlbumFileDateTime time = {};
            detail::EncryptedApplicationId encryptedApplicationId = {};

            const char* p = filename;
            const char* pEnd = filename + filenameSize;
            // date and time
            NN_RESULT_DO(ParseDateTime(&p, &time, p, static_cast<int>(pEnd - p)));
            // "-"
            NN_RESULT_THROW_UNLESS(
                MatchString(&p, p, static_cast<int>(pEnd - p), "-", 1),
                ResultAlbumInvalidFilename()
            );
            // encoded title id
            NN_RESULT_DO(ParseEncryptedApplicationId(&p, &encryptedApplicationId, p, static_cast<int>(pEnd - p)));

            // Extra attribute
            char extraChar;
            NN_RESULT_DO(ParseExtraChar(&p, &extraChar, p,  static_cast<int>(pEnd - p)));

            *pOutExtra = extraChar;
            *pOutTime = time;
            *pOutEncryptedApplicationId = encryptedApplicationId;
            *pOutNext = p;
            *pOutEnd  = pEnd;
            NN_RESULT_SUCCESS;
        }

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidFilename
        // @retval nn::capsrv::ResultAlbumInvalidTimestamp
        // @retval nn::capsrv::ResultAlbumInvalidApplicationId
        // @retval nn::capsrv::ResultAlbumInvalidFileContents
        nn::Result ParseFilename(
            const char** pOutNext,
            const char** pOutEnd,
            AlbumFileDateTime* pOutTime,
            detail::EncryptedApplicationId* pOutEncryptedApplicationId,
            char* pOutExtra,
            const char* filename,
            size_t filenameSize,
            const EnvironmentInfo& env
            ) NN_NOEXCEPT
        {
            int length = nn::util::Strnlen(filename, static_cast<int>(filenameSize));
            if (length == AlbumPathUtility::FilenameLength)
            {
                *pOutExtra = '\0';
                return ParseFilenameRegular(pOutNext, pOutEnd, pOutTime, pOutEncryptedApplicationId, filename, filenameSize, env);
            }
            else if (length == AlbumPathUtility::ExtraFilenameLength)
            {
                return ParseFilenameWithExtraChar(pOutNext, pOutEnd, pOutTime, pOutEncryptedApplicationId, pOutExtra, filename, filenameSize, env);
            }

            NN_RESULT_THROW( ResultAlbumInvalidFilename() );
        }
    }

    nn::Result AlbumPathUtility::ParseNullOrEoi(
        const char** pOutNext,
        const char* pString,
        size_t length
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutNext);
        NN_SDK_REQUIRES_NOT_NULL(pString);

        const char* p = pString;
        if(length == 0)
        {
            *pOutNext = p;
            NN_RESULT_SUCCESS;
        }
        if(*p == '\0')
        {
            *pOutNext = p;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW(ResultAlbumInvalidFileId());
    }

    nn::Result AlbumPathUtility::ParseAlbumStorageName(
        const char** pOutNext,
        AlbumStorageType* pOutValue,
        const char* pString,
        size_t length
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutNext);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pString);

        NN_RESULT_THROW_UNLESS(length >= static_cast<size_t>(MountNameLength), ResultAlbumInvalidStorage());

        const char* p = pString;
        const char* pEnd = pString + length;
        if(MatchString(&p, p, static_cast<int>(pEnd - p), NN_CAPSRV_MOUNT_NAME_NAND, MountNameLength))
        {
            *pOutValue = AlbumStorage_Nand;
            *pOutNext  = p;
            NN_RESULT_SUCCESS;
        }
        else if(MatchString(&p, p, static_cast<int>(pEnd - p), NN_CAPSRV_MOUNT_NAME_SD, MountNameLength))
        {
            *pOutValue = AlbumStorage_Sd;
            *pOutNext  = p;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW(ResultAlbumInvalidStorage());
    }

    nn::Result AlbumPathUtility::ParseSubDirectoryNameForRegular(
        const char** pOutNext,
        AlbumFileId* pOutValue,
        const char* pString,
        size_t length,
        int depth
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pString);
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(SubDirectoryDepth));

        size_t directoryNameLength = GetSubDirectoryNameLengthForRegular(depth);
        NN_RESULT_THROW_UNLESS(length >= directoryNameLength, ResultAlbumInvalidTimestamp());

        AlbumFileId result = *pOutValue;

        const char* p = pString;
        switch(depth)
        {
        case 0:
            {
                uint16_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateYear(v));
                result.time.year = v;
                break;
            }
        case 1:
            {
                uint8_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateMonth(v));
                result.time.month = v;
                break;
            }
        case 2:
            {
                uint8_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateDay(v));
                result.time.day = v;
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = result;
        *pOutNext = p;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumPathUtility::ParseSubDirectoryNameForExtra(
        const char** pOutNext,
        AlbumFileId* pOutValue,
        const char* pString,
        size_t length,
        int depth
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pString);
        NN_SDK_REQUIRES_RANGE(depth, 0, static_cast<int>(ExtraSubDirectoryDepth));

        size_t directoryNameLength = GetSubDirectoryNameLengthForExtra(depth);
        NN_RESULT_THROW_UNLESS(length >= directoryNameLength, ResultAlbumInvalidTimestamp());

        AlbumFileId result = *pOutValue;

        const char* p = pString;
        switch(depth)
        {
        case 0:
            {
                NN_RESULT_THROW_UNLESS(std::memcmp(p, NN_CAPSRV_DIRECTORY_NAME_FOR_EXTRA, directoryNameLength) == 0, ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                break;
            }
        case 1:
            {
                detail::EncryptedApplicationId encryptedId;
                char buf[3];
                buf[2] = '\0';
                for (int i=0; i<sizeof(encryptedId.value); ++i)
                {
                    buf[0] = p[i * 2 + 0];
                    buf[1] = p[i * 2 + 1];
                    char* outPtr;
                    int val = std::strtol(buf, &outPtr, 16);
                    NN_RESULT_THROW_UNLESS(buf + 2 == outPtr, ResultAlbumInvalidTimestamp());
                    NN_RESULT_THROW_UNLESS(val >= 0x00 && val <= 0xff, ResultAlbumInvalidTimestamp());
                    encryptedId.value[i] = val;
                }
                p += directoryNameLength;

                nn::ncm::ApplicationId id;
                bool isExtra;
                NN_RESULT_DO(TryDecryptApplicationId(&id, &isExtra, encryptedId));
                // ApplicationId 以外のフラグはセットされていない状態で
                // Encrypt されているはずである。
                NN_RESULT_THROW_UNLESS(!isExtra, ResultAlbumInvalidTimestamp());
                // Extra の場合は applicationId のフィールドを埋めておく。
                // 以後、AlbumFileId を directoryId として使用する箇所において、
                // applicationId != 0 なら Extra なディレクトリ探索中を示す。
                result.applicationId = id;
                break;
            }
        case 2:
            {
                uint16_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateYear(v));
                result.time.year = v;
                break;
            }
        case 3:
            {
                uint8_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateMonth(v));
                result.time.month = v;
                break;
            }
        case 4:
            {
                uint8_t v = 0;
                NN_RESULT_THROW_UNLESS(ParseDecimal(&v, p, directoryNameLength), ResultAlbumInvalidTimestamp());
                p += directoryNameLength;
                NN_RESULT_DO(AlbumPathUtility::ValidateDay(v));
                result.time.day = v;
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }

        *pOutValue = result;
        *pOutNext  = p;
        NN_RESULT_SUCCESS;
    }

    // 0 階層目のディレクトリ名から以下のいずれを呼ぶかを判別する
    // - AlbumPathUtility::ParseSubDirectoryNameForRegular()
    // - AlbumPathUtility::ParseSubDirectoryNameForExtra()
    nn::Result AlbumPathUtility::ParseSubDirectoryName(
        const char** pOutNext,
        AlbumFileId* pOutValue,
        const char* pString,
        size_t length,
        int depth,
        bool isExtra
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pString);

        if (isExtra)
        {
            return ParseSubDirectoryNameForExtra(pOutNext, pOutValue, pString, length, depth);
        }
        else
        {
            return ParseSubDirectoryNameForRegular(pOutNext, pOutValue, pString, length, depth);
        }
    }

    nn::Result AlbumPathUtility::ParseAlbumFilename(
        const char** pOutNext,
        AlbumFileId* pOutValue,
        const char* pString,
        size_t length,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutNext);
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pString);

        AlbumFileDateTime time;
        detail::EncryptedApplicationId encryptedApplicationId;
        AlbumFileContentsType contents;
        char extraChar = '\0';

        const char* p = pString;
        const char* pEnd;
        NN_RESULT_DO(ParseFilename(&p, &pEnd, &time, &encryptedApplicationId, &extraChar, p, length, env));

        nn::ncm::ApplicationId applicationId;
        bool isExtra;
        NN_RESULT_DO(TryDecryptApplicationId(&applicationId, &isExtra, encryptedApplicationId));
        NN_RESULT_THROW_UNLESS(
            (!isExtra && extraChar == '\0') || (isExtra && extraChar == 'X'),
            ResultAlbumInvalidFilename()
        );
        NN_RESULT_DO(ParseFileExtension(&p, &contents, p, static_cast<int>(pEnd - p), isExtra, env));

        pOutValue->applicationId = applicationId;
        pOutValue->time = time;
        pOutValue->contents = contents;
        *pOutNext = p;
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumPathUtility::CheckFileIdConsistency(
        const AlbumFileId& fileId,
        const AlbumFileId& directoryId
        ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(fileId.time.year == directoryId.time.year, ResultAlbumInvalidTimestamp());
        NN_RESULT_THROW_UNLESS(fileId.time.month == directoryId.time.month, ResultAlbumInvalidTimestamp());
        NN_RESULT_THROW_UNLESS(fileId.time.day == directoryId.time.day, ResultAlbumInvalidTimestamp());
        NN_RESULT_SUCCESS;
    }

}}}}
