﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/capsrv/movie/capsrv_MovieMetaData.h>
#include "../capsrvServer_Config.h"
#include "capsrvServer_AlbumFileManipulator.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    class AlbumMovieMetaInfo
    {
    public:
        AlbumMovieMetaInfo() NN_NOEXCEPT;
        AlbumMovieMetaInfo(
            int64_t movieDataSize,
            int64_t imageDataSize,
            uint64_t version
        ) NN_NOEXCEPT;

        int64_t GetMovieDataSize() const NN_NOEXCEPT;
        int64_t GetImageDataSize() const NN_NOEXCEPT;
        uint64_t GetVersion() const NN_NOEXCEPT;

    private:
        int64_t m_MovieDataSize;
        int64_t m_ImageDataSize;
        uint64_t m_Version;
    };

    class AlbumMovieUtility
    {
    public:
        struct BlockHash
        {
            char value[MovieDataBlockHashSize];
        };

        struct FileSign
        {
            char value[MovieSignatureSize];
        };

        struct MetaHashList
        {
        public:
            static const int Length = MovieMetaDataSize / MovieDataBlockSize;

        public:
            AlbumMovieUtility::BlockHash hashList[Length];
        };

        struct HashSectionValidationState
        {
            uint64_t untrustedVersion;
            nn::crypto::Sha256Generator hasher;
        };

        struct FileSignGenerationState
        {
            nn::crypto::Sha256Generator hasher;
        };

    public:
        // ファイルサイズとバージョンから各セクションの大きさを計算する
        static nn::Result CalculateSectionSize(
            int64_t* pOutDataBlockCount,
            int64_t* pOutDataSize,
            int64_t* pOutMetaSize,
            int64_t* pOutDataHashSize,
            int64_t* pOutMetaHashSize,
            int64_t* pOutSignSize,
            int64_t* pOutVersionSize,
            int64_t fileSize,
            uint64_t untrustedVersion
        ) NN_NOEXCEPT;

        static void SetCalculateFileSign(void (*f)(FileSign* pOut, const void* data, size_t dataSize)) NN_NOEXCEPT;

        // ハッシュに対して署名チェックを行う
        static nn::Result StartHashValidation(
            HashSectionValidationState* pOutState,
            uint64_t untrustedVersion
        ) NN_NOEXCEPT;

        static void ProcessHashValidation(
            HashSectionValidationState& state,
            const BlockHash* pHashList,
            int64_t hashCount
        ) NN_NOEXCEPT;

        static nn::Result FinishHashValidation(
            HashSectionValidationState& state,
            const FileSign& sign
        ) NN_NOEXCEPT;

        // ハッシュから署名を生成する
        static nn::Result StartFileSignGeneration(
            FileSignGenerationState* pState
        ) NN_NOEXCEPT;

        static void ProcessFileSignGeneration(
            FileSignGenerationState& state,
            const BlockHash* pHashList,
            int64_t hashCount
        );

        static nn::Result FinishFileSignGeneration(
            FileSign* pOut,
            FileSignGenerationState& state
        );

        // ブロックに対してハッシュの検査を行う
        // @pre pBlock != nullptr
        // @pre blockSize == MovieDataBlockSize
        static nn::Result ValidateBlockData(
            const void* pBlock,
            size_t blockSize,
            const BlockHash& expectedHash
        ) NN_NOEXCEPT;

        // メタ情報の解析を行う
        // @pre pMetaInfoData != nullptr
        static nn::Result ExtractMetaInfo(
            AlbumMovieMetaInfo* pOutValue,
            const nn::capsrv::movie::MovieMetaInfoData* pMetaInfoData,
            uint64_t untrustedVersion,
            int64_t dataSectionSize
        ) NN_NOEXCEPT;
    };

    class AlbumMovieUtilityStaticObject
    {
    public:
        void Initialize(void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        // ゼロ埋めされたデータに対するハッシュ値を取得する
        AlbumMovieUtility::BlockHash GetZeroBlockHash() const NN_NOEXCEPT;

    private:
        AlbumMovieUtility::BlockHash m_ZeroBlockHash;

    };


}}}}
