﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumManager.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../detail/capsrvServer_AlbumContentsAttribute.h"
#include "../detail/capsrvServer_CalculateJpegMac.h"
#include "capsrvServer_AlbumPathUtility.h"
#include "capsrvServer_AlbumFileManipulator.h"
#include "capsrvServer_ExtractJpegThumbnail.h"
#include "capsrvServer_VerifyScreenShotFileData.h"
#include "capsrvServer_AlbumManagerCommon.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumError
    // @retval CheckMounted()
    // @retval OpenFileForSaveFile()
    // @retval ConvertFsResultForSaveFile()
    nn::Result AlbumManager::SaveScreenShotFile(
        const AlbumFileId* pFileId,
        const void* data,
        size_t dataSize,
        uint64_t makerNoteVersion,
        int64_t makerNoteOffset,
        int64_t makerNoteSize
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        NN_SDK_REQUIRES_NOT_NULL(pFileId);

        auto storage = pFileId->storage;
        auto contents = pFileId->contents;
        NN_CAPSRV_CHECK_FILE_NOT_READ_LOCKED(*pFileId, ResultAlbumAccessCorrupted());
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pFileId, ResultAlbumAccessCorrupted());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(storage, AlbumStorageDirection_Destination));
        NN_RESULT_DO(CheckAlbumLimitation(storage, contents, 1));

        NN_CAPSRV_PROCESS_START();

        nn::capsrv::server::detail::Signature signature = {};
        int64_t signatureOffset = 0;
        NN_RESULT_DO(nn::capsrv::server::detail::CalculateJpegMac(&signature, &signatureOffset, data, dataSize, makerNoteVersion, makerNoteOffset, makerNoteSize));

        AlbumFileHandle hFile = {};
        NN_CAPSRV_CREATE_FILEPATH(filepath, *pFileId, AlbumStorageDirection_Destination);
        {
            AlbumCacheDelta delta = {};
            NN_UTIL_SCOPE_EXIT {
                m_AlbumStorageCache.UpdateEntry(delta);
            };
            NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForSave(
                &hFile, &delta, filepath, static_cast<int64_t>(dataSize)
            ));
        }
        NN_UTIL_SCOPE_EXIT{
            AlbumFileManipulator::CloseAlbumFile(hFile);
        };

        NN_CAPSRV_PROCESS_ROLLBACK({
            AlbumCacheDelta delta = {};
            NN_UTIL_SCOPE_EXIT {
                m_AlbumStorageCache.UpdateEntry(delta);
            };
            (void)AlbumFileManipulator::DeleteAlbumFile(&delta, filepath);
        });

        NN_STATIC_ASSERT(2 * ScreenShotDataBlockSize >= ScreenShotImageSizeLimit);
        char* pBuffer = reinterpret_cast<char*>(m_Memory);
        // 前半
        {
            size_t sizeToWrite = std::min<size_t>(dataSize, ScreenShotDataBlockSize);
            std::memcpy(pBuffer, data, sizeToWrite);
            NN_SDK_ASSERT(sizeToWrite >= signatureOffset + sizeof(signature));
            NN_RESULT_THROW_UNLESS(sizeToWrite >= signatureOffset + sizeof(signature), ResultInternalError());
            // 署名部分を置き換え
            std::memcpy(pBuffer + signatureOffset, &signature, sizeof(signature));
            // 書込み
            NN_RESULT_DO(AlbumFileManipulator::WriteFile(hFile, 0, pBuffer, sizeToWrite));
        }
        // 後半
        if(dataSize > ScreenShotDataBlockSize)
        {
            size_t sizeToWrite = dataSize - ScreenShotDataBlockSize;
            NN_RESULT_DO(AlbumFileManipulator::WriteFile(
                hFile,
                ScreenShotDataBlockSize,
                reinterpret_cast<const char*>(data) + ScreenShotDataBlockSize,
                sizeToWrite
            ));
        }

        NN_CAPSRV_PROCESS_SUCCESS();

        NN_RESULT_SUCCESS;
    }

    //------------------------
    nn::Result AlbumManager::LoadScreenShotImageFileImpl(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutFileSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId& fileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileSize);
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_ScreenShot ||
                        fileId.contents == AlbumFileContents_ExtraScreenShot);

        NN_CAPSRV_PROCESS_START();

        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);

        // ファイルがバッファに収まるか確認
        NN_RESULT_THROW_UNLESS(
            static_cast<uintmax_t>(fileSize) <= static_cast<uintmax_t>(bufferSize),
            ResultAlbumReadBufferShortage()
        );

        if(fileSize > 0)
        {
            size_t readSize;
            NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, 0, pBuffer, static_cast<size_t>(fileSize)));
            if(static_cast<uintmax_t>(readSize) != static_cast<uintmax_t>(fileSize))
            {
                NN_RESULT_THROW(ResultAlbumError());
            }
        }

        AlbumFileAttribute attribute = {};
        // ファイルの内容が正当か検査
        NN_RESULT_DO(VerifyScreenShotFileDataEx(
            &attribute.attribute.screenshot,
            pOutAppletData,
            pOutApplicationData,
            pOutSystemReservedInfo,
            &fileId,
            pBuffer,
            static_cast<size_t>(fileSize),
            *m_pEnvironmentInfo
        ));
        attribute.content = fileId.contents;

        NN_CAPSRV_PROCESS_SUCCESS();
        if(pOutAttribute)
        {
            *pOutAttribute = attribute;
        }
        *pOutFileSize = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumManager::LoadScreenShotThumbnailFileImpl(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId& fileId,
        void* workMemory,
        size_t workSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_ScreenShot ||
                        fileId.contents == AlbumFileContents_ExtraScreenShot);

        NN_CAPSRV_PROCESS_START();

        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);

        NN_RESULT_THROW_UNLESS(
            fileSize > 0,
            ResultAlbumInvalidFileData()
        );

        AlbumFileAttribute attribute = {};
        size_t thumbSize = 0;

        // 先頭部分だけ読み込む。
        size_t sizeToRead = std::min(static_cast<size_t>(fileSize), static_cast<size_t>(AlbumManagerThumbnailLoadSize));
        NN_SDK_ASSERT_LESS_EQUAL(sizeToRead, workSize);
        NN_RESULT_THROW_UNLESS(sizeToRead <= workSize, nn::capsrv::ResultAlbumWorkMemoryError());

        size_t readSize;
        NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, 0, workMemory, sizeToRead));
        if(static_cast<uintmax_t>(readSize) != static_cast<uintmax_t>(sizeToRead))
        {
            NN_RESULT_THROW(ResultAlbumError());
        }

        // ファイルの検査と属性の取得
        NN_RESULT_DO(VerifyScreenShotFileHeaderEx(&attribute.attribute.screenshot, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, &fileId, workMemory, readSize, *m_pEnvironmentInfo));
        attribute.content = fileId.contents;

        // サムネイル部分を抽出
        NN_RESULT_DO(ExtractJpegThumbnail(&thumbSize, pBuffer, bufferSize, workMemory, readSize));

        NN_CAPSRV_PROCESS_SUCCESS();
        if(pOutAttribute)
        {
            *pOutAttribute = attribute;
        }
        *pOutSize = thumbSize;
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumManager::VerifyAndLoadScreenShotMakerNoteInfo(
        uint64_t* pOutFileSize,
        nn::capsrv::server::detail::MakerNoteInfo* pOutMakerNoteInfo,
        const AlbumFileId& fileId,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(fileId.contents == AlbumFileContents_ScreenShot ||
                        fileId.contents == AlbumFileContents_ExtraScreenShot);

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_CREATE_FILEPATH(filepath, fileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, fileId, hFile);

#if 1
        // ファイルの先頭部分だけ読む（ヘッダ相当の workBufferSize でよい）
        // SIGLO-74160 の account::Uid でフィルタしながらファイルリストを取得
        // する場合には、個々のファイルの MakerNote を確認する必要があるため、
        // 高速化のため、ファイルの先頭部分だけを読むことにする。
        size_t sizeToRead = std::min(static_cast<size_t>(fileSize), static_cast<size_t>(AlbumManagerThumbnailLoadSize));
        NN_RESULT_THROW_UNLESS(sizeToRead <= workBufferSize, nn::capsrv::ResultAlbumWorkMemoryError());
        auto fileDataRange = FileDataRange_HeaderOnly;
#else
        // ファイル全体を読む（ファイルサイズ相当の workBufferSize が必要）
        size_t sizeToRead = fileSize;
        NN_RESULT_THROW_UNLESS(sizeToRead <= workBufferSize, nn::capsrv::ResultAlbumReadBufferShortage());
        auto fileDataRange = FileDataRange_WholeFile;
#endif

        // ファイルがバッファに収まるか確認
        if (sizeToRead > 0)
        {
            size_t readSize;
            NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hFile, 0, pWorkBuffer, sizeToRead));
            NN_RESULT_THROW_UNLESS(readSize == sizeToRead, ResultAlbumError());
        }

        // ファイルの内容が正当か検査
        NN_RESULT_DO(VerifyAndLoadScreenShotMakerNoteInfoImpl(
            pOutMakerNoteInfo,
            &fileId,
            pWorkBuffer,
            static_cast<size_t>(fileSize),
            *m_pEnvironmentInfo,
            fileDataRange
        ));
        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutFileSize = static_cast<uint64_t>(fileSize);
        NN_RESULT_SUCCESS;
    }

}}}}

