﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumManager.h"

#include <algorithm>
#include <iterator>
#include <limits>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_AlbumFileCountLimit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include "../albumsrv/capsrvServer_AlbumServerObject.h"
#include "../capsrvServer_ConvertFileSystemResult.h"
#include "../detail/capsrvServer_AlbumContentsAttribute.h"
#include "../detail/capsrvServer_GetApplicationAlbumEntry.h"
#include "capsrvServer_AlbumPathUtility.h"
#include "capsrvServer_MountAlbumDirectory.h"
#include "capsrvServer_VerifyScreenShotFileData.h"
#include "capsrvServer_ExtractJpegThumbnail.h"
#include "capsrvServer_AlbumManagerCommon.h"


namespace nn{ namespace capsrv{ namespace server{ namespace album{

namespace {

    bool IsScreenShotFileImpl(const AlbumFileId& fileId) NN_NOEXCEPT
    {
        if ((g_EnvironmentInfo.IsAlbumScreenShotFileSupported() && fileId.contents == AlbumFileContents_ScreenShot) ||
            (g_EnvironmentInfo.IsAlbumExtraScreenShotFileSupported() && fileId.contents == AlbumFileContents_ExtraScreenShot))
        {
            return true;
        }
        return false;
    }

    bool IsMovieFileImpl(const AlbumFileId& fileId) NN_NOEXCEPT
    {
        if ((g_EnvironmentInfo.IsAlbumMovieFileSupported() && fileId.contents == AlbumFileContents_Movie) ||
            (g_EnvironmentInfo.IsAlbumExtraMovieFileSupported() && fileId.contents == AlbumFileContents_ExtraMovie))
        {
            return true;
        }
        return false;
    }

}   // namespace

    void AlbumManager::StorageStatus::SetAvailableNotTried() NN_NOEXCEPT
    {
        mountStatus = StorageMountStatus::AvailableNotTried;
        mountResult = ResultAlbumIsNotMounted();
    }

    void AlbumManager::StorageStatus::SetAvailable() NN_NOEXCEPT
    {
        mountStatus = StorageMountStatus::AvailableMounted;
        mountResult = nn::ResultSuccess();
    }

    void AlbumManager::StorageStatus::SetUnavailable() NN_NOEXCEPT
    {
        mountStatus = StorageMountStatus::Unavailable;
        mountResult = ResultAlbumIsNotMounted();
    }

    void AlbumManager::StorageStatus::SetUnavailable(nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(result.IsFailure());
        mountStatus = StorageMountStatus::Unavailable;
        mountResult = result;
    }

    //-----------------------------------

    AlbumManager::AlbumManager() NN_NOEXCEPT
        : m_Memory(NULL)
        , m_MemorySize(0)
        , m_pEnvironmentInfo(nullptr)
        , m_pResourceIdManager(nullptr)
    {
        for(int i = 0; i < AlbumStorageCount; i++)
        {
            m_StorageStatusList[i].SetUnavailable();
        }
    }

    void AlbumManager::Initialize(const EnvironmentInfo* pEnvironmentInfo, ResourceIdManager* pResourceIdMgr) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pEnvironmentInfo);
        m_pEnvironmentInfo = pEnvironmentInfo;
        m_pResourceIdManager = pResourceIdMgr;

        NN_RESULT_ABORTING_BLOCK
        {
            const fs::SystemSaveDataId systemSaveDataId{0x8000000000000140llu};
            NN_RESULT_TRY(fs::CreateSystemSaveData(systemSaveDataId, 512 * 1024, 512 * 1024, 0))
                NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
                {
                    // ignore
                }
            NN_RESULT_END_TRY
            NN_RESULT_DO(fs::MountSystemSaveData(NN_CAPSRV_MOUNT_NAME_TEMP, systemSaveDataId));
            NN_RESULT_SUCCESS;
        };

        // 初期状態ではすべてのストレージを無効にしておく
        for(int i = 0; i < AlbumStorageCount; i++)
        {
            m_StorageStatusList[i].SetUnavailable();
        }

        m_ReadLockFileTable.Initialize();
        m_WriteLockFileTable.Initialize();
        m_MovieStreamManager.Initialize(pEnvironmentInfo, pResourceIdMgr, &m_ReadLockFileTable, &m_WriteLockFileTable);
    }

    void AlbumManager::Finalize() NN_NOEXCEPT
    {

        for(int s = 0; s < AlbumStorageCount; s++)
        {
            m_MovieStreamManager.NotifyStorageUnmounted(static_cast<AlbumStorageType>(s));
            if(m_StorageStatusList[s].mountStatus == StorageMountStatus::AvailableMounted)
            {
                UnmountAlbumDirectory(static_cast<AlbumStorageType>(s), *m_pEnvironmentInfo);
            }
            m_StorageStatusList[s].SetUnavailable();
        }

        fs::Unmount(NN_CAPSRV_MOUNT_NAME_TEMP);

        m_MovieStreamManager.Finalize();
        m_WriteLockFileTable.Finalize();
        m_ReadLockFileTable.Finalize();

        m_pEnvironmentInfo = nullptr;
        m_pResourceIdManager = nullptr;
    }

    void AlbumManager::SetWorkMemory(void* memory, size_t size) NN_NOEXCEPT
    {
        m_Memory = memory;
        m_MemorySize = size;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumInvalidStorage
    // @retval nn::capsrv::ResultAlbumIsNotMounted
    nn::Result AlbumManager::EnsureMounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            storage >= 0 && storage < AlbumStorageCount,
            ResultAlbumInvalidStorage()
        );

        auto& storageStatus = m_StorageStatusList[storage];

        // マウントを試していなければマウントしてみる
        if(storageStatus.mountStatus == StorageMountStatus::AvailableNotTried)
        {
            auto mountResult = MountAlbumDirectory(storage, *m_pEnvironmentInfo);
            if(mountResult.IsSuccess())
            {
                storageStatus.SetAvailable();
            }
            else if(ResultAlbumStructureCorrupted::Includes(mountResult))
            {
                storageStatus.SetUnavailable(ResultAlbumStructureCorrupted());
            }
            else if(ResultNeedsRetryAlbumMount::Includes(mountResult))
            {
                storageStatus.SetAvailableNotTried();
                NN_RESULT_THROW(storageStatus.mountResult);
            }
            else
            {
                storageStatus.SetUnavailable();
            }
        }

        // マウント成功 → Success
        // マウント失敗 → Failure
        NN_SDK_ASSERT(
            (storageStatus.mountStatus == StorageMountStatus::AvailableMounted && storageStatus.mountResult.IsSuccess()) ||
            (storageStatus.mountStatus == StorageMountStatus::Unavailable && storageStatus.mountResult.IsFailure())
        );

        NN_RESULT_THROW(storageStatus.mountResult);
    }

    nn::Result AlbumManager::EnsureMounted(AlbumStorageType storage, AlbumStorageDirection direction) NN_NOEXCEPT
    {
        NN_RESULT_TRY(EnsureMounted(storage))
            NN_RESULT_CATCH(ResultAlbumStructureCorrupted)
            {
                switch(direction)
                {
                case AlbumStorageDirection_Source:
                    NN_RESULT_THROW(ResultAlbumSourceStructureCorrupted());
                case AlbumStorageDirection_Destination:
                    NN_RESULT_THROW(ResultAlbumDestinationStructureCorrupted());
                default:
                    NN_RESULT_THROW(ResultAlbumStructureCorrupted());
                }
            }
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumManager::ForceUnmounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_WARN("Storage %d is forced to be unmounted\n", storage);
        NN_RESULT_THROW_UNLESS(
            storage >= 0 && storage < AlbumStorageCount,
            ResultAlbumInvalidStorage()
        );

        auto& storageStatus = m_StorageStatusList[storage];

        // マウント中ならアンマウント（マウントしていなければ無視）
        if(storageStatus.mountStatus == StorageMountStatus::AvailableMounted)
        {
            m_MovieStreamManager.NotifyStorageUnmounted(storage);
            NN_RESULT_DO(UnmountAlbumDirectory(storage, *m_pEnvironmentInfo));
        }

        storageStatus.SetUnavailable();
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::ResetMountStatus(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_INFO("Reset storage %d's mount status\n", storage);
        NN_RESULT_THROW_UNLESS(
            storage >= 0 && storage < AlbumStorageCount,
            ResultAlbumInvalidStorage()
        );

        auto& storageStatus = m_StorageStatusList[storage];

        // マウント中ならアンマウント（マウントしていなければ無視）
        if(storageStatus.mountStatus == StorageMountStatus::AvailableMounted)
        {
            m_MovieStreamManager.NotifyStorageUnmounted(storage);
            NN_RESULT_DO(UnmountAlbumDirectory(storage, *m_pEnvironmentInfo));
        }

        storageStatus.SetAvailableNotTried();
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::RefreshStorageCache(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(EnsureMounted(storage));

        AlbumStorageCache::Entry entryList[FileContentsCount] = {};

        auto visitor = MakeAlbumVisitor(
            [&](const AlbumFileId&, int) -> nn::Result { NN_RESULT_SUCCESS; },
            [&]() -> nn::Result { NN_RESULT_SUCCESS; },
            // managed file
            [&](int64_t size, const AlbumFileId& fileId) -> nn::Result {
                if(fileId.contents >= 0 && fileId.contents < FileContentsCount)
                {
                    entryList[fileId.contents].fileCount++;
                }
                NN_RESULT_SUCCESS;
            },
            // unmanaged file. 管理外のファイルはカウントしない。
            [&](int64_t) -> nn::Result { NN_RESULT_SUCCESS; }
        );

        NN_SDK_ASSERT(m_MemorySize >= sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            storage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(m_Memory),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        for(AlbumFileContentsType c = 0; c < FileContentsCount; c++)
        {
            NN_CAPSRV_LOG_DEV("Updated album cache: Storage=%d, Contents=%d, FileCount=%lld\n",
                static_cast<int>(storage), c, entryList[c].fileCount);
        }

        for(AlbumFileContentsType c = 0; c < FileContentsCount; c++)
        {
            m_AlbumStorageCache.SetEntry(storage, c, entryList[c]);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::GetStorageCache(AlbumCacheData* pOutValue, AlbumStorageType storage, AlbumFileContentsType contents) NN_NOEXCEPT
    {
        std::memset(pOutValue, 0, sizeof(AlbumCacheData));
        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(AlbumPathUtility::ValidateFileContents(contents, *m_pEnvironmentInfo));

        *pOutValue = m_AlbumStorageCache.GetEntry(storage, contents);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::CheckAlbumLimitation(AlbumStorageType storage, AlbumFileContentsType contents, int64_t fileCountDelta) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(fileCountDelta, 0);
        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(AlbumPathUtility::ValidateFileContents(contents, *m_pEnvironmentInfo));

        auto countLimit = m_pEnvironmentInfo->GetFileCountLimit(storage, contents);
        auto cache = m_AlbumStorageCache.GetEntry(storage, contents);

        int64_t postCount = cache.fileCount + fileCountDelta;
        NN_RESULT_THROW_UNLESS(postCount >= cache.fileCount, ResultInternalAlbumLimitationFileCountLimit()); // 念の為。
        NN_RESULT_THROW_UNLESS(postCount <= countLimit, ResultInternalAlbumLimitationFileCountLimit());
        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumError
    // @retval CheckMounted()
    // @retval ConvertFsResultForLoadDirectory()
    nn::Result AlbumManager::GetFileCount(
        int* pOutCount,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        *pOutCount = 0;

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(EnsureMounted(storage, AlbumStorageDirection_Source));

        int fileCount = 0;

        auto visitor = MakeAlbumVisitor(
            [&](const AlbumFileId&, int) -> nn::Result { NN_RESULT_SUCCESS; },
            [&]() -> nn::Result { NN_RESULT_SUCCESS; },
            [&](int64_t, const AlbumFileId& fileId) -> nn::Result
            {
                // 対象ファイル種別でなければカウントしない
                NN_RESULT_THROW_UNLESS(contentsMask.Test(fileId.contents), ResultSuccess());

                // 書込み中のファイルはカウントしない
                NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, nn::ResultSuccess());

                fileCount++;
                NN_RESULT_SUCCESS;
            },
            [&](int64_t) -> nn::Result { NN_RESULT_SUCCESS; }
        );
        NN_SDK_ASSERT(m_MemorySize >= sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            storage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(m_Memory),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        *pOutCount = fileCount;

        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capstv::ResultAlbumError
    // @retval CheckMounted()
    // @retval ConvertFsResultForLoadDirectory()
    nn::Result AlbumManager::GetFileList(
        int* pOutCount,
        AlbumEntry* pOutList,
        int listCapacity,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        *pOutCount = 0;
        std::memset(pOutList, 0, sizeof(AlbumEntry) * listCapacity);

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(EnsureMounted(storage, AlbumStorageDirection_Source));

        int count = 0;
        auto visitor = MakeAlbumVisitor(
            [&](const AlbumFileId&, int) -> nn::Result { NN_RESULT_SUCCESS; },
            [&]() -> nn::Result { NN_RESULT_SUCCESS; },
            [&](int64_t size, const AlbumFileId& fileId) -> nn::Result {
                if(count >= listCapacity)
                {
                    NN_RESULT_SUCCESS;
                }

                // 対象ファイル種別でなければリストに含めない
                NN_RESULT_THROW_UNLESS(contentsMask.Test(fileId.contents), ResultSuccess());

                // 書込み中のファイルはリストに含めない
                NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, nn::ResultSuccess())

                pOutList[count].fileId = fileId;
                pOutList[count].size   = size;
                count++;
                NN_RESULT_SUCCESS;
            },
            [&](int64_t) -> nn::Result { NN_RESULT_SUCCESS; }
        );

        NN_SDK_ASSERT(m_MemorySize >= sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            storage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(m_Memory),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        *pOutCount = count;
        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capstv::ResultAlbumError
    // @retval CheckMounted()
    // @retval ConvertFsResultForLoadDirectory()
    nn::Result AlbumManager::GetFileListForApplication(
        int* pOutCount,
        ApplicationAlbumFileEntry* pOutList,
        int listCapacity,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask,
        const account::Uid* pUid,
        AlbumFileDateTime beginDateTime,
        AlbumFileDateTime endDateTime,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        *pOutCount = 0;
        std::memset(pOutList, 0, sizeof(ApplicationAlbumFileEntry) * listCapacity);

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(EnsureMounted(storage, AlbumStorageDirection_Source));

        server::detail::ApplicationAlbumEntryKey key = {};
        NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationAlbumEntryKeyFromApplicationId(&key, applicationId));

        auto compare = [](const ApplicationAlbumFileEntry& lhs, const ApplicationAlbumFileEntry& rhs) -> bool
        {
            // 比較演算子によってソート方向が以下のようになる。
            //
            // - ">" なら、日付が新しいものから古いものという順序になる
            // - "<" なら、日付が古いものから新しいものという順序になる
            //
            return lhs.dateTime > rhs.dateTime;
        };

        int count = 0;
        auto visitor = MakeAlbumVisitor(
            [&](const AlbumFileId&, int) -> nn::Result { NN_RESULT_SUCCESS; },
            [&]() -> nn::Result { NN_RESULT_SUCCESS; },
            [&](int64_t size, const AlbumFileId& fileId) -> nn::Result {
                if(count >= listCapacity)
                {
                    NN_RESULT_SUCCESS;
                }

                // 対象ファイル種別でなければリストに含めない
                NN_RESULT_THROW_UNLESS(contentsMask.Test(fileId.contents), ResultSuccess());

                // ApplicationId が異なる場合はリストに含めない
                NN_RESULT_THROW_UNLESS(fileId.applicationId == applicationId, ResultSuccess());

                // 作成日時が指定された期間内でなければリストに含めない
                NN_RESULT_THROW_UNLESS(fileId.time >= beginDateTime && fileId.time < endDateTime, ResultSuccess());

                // 書込み中のファイルはリストに含めない
                NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, nn::ResultSuccess())

                // Uid フィルタ有効時は、MakerNote 内の Uid をチェック
                if (pUid)
                {
                    // fileId を使って MakerNoteInfo 内の UidList と照合
                    uint64_t fileSize;
                    capsrv::server::detail::MakerNoteInfo makerNoteInfo;
                    NN_RESULT_THROW_UNLESS(this->LoadMakerNoteInfo(&fileSize, &makerNoteInfo, fileId, m_Memory, AlbumManagerThumbnailLoadSize).IsSuccess(), ResultSuccess());
                    auto& uidList = makerNoteInfo.systemReservedInfo.uidList;
                    bool isUidMatched = false;
                    for (int i=0; i<uidList.uidCount; ++i)
                    {
                        if (uidList.uid[i] == *pUid)
                        {
                            isUidMatched = true;
                            break;
                        }
                    }
                    NN_RESULT_THROW_UNLESS(isUidMatched, ResultSuccess());
                }

                // AlbumEntry を構築
                AlbumEntry albumEntry = {};
                albumEntry.size   = size;
                albumEntry.fileId = fileId;

                // アプリ向けの ApplicationAlbumFileEntry に変換
                ApplicationAlbumFileEntry outEntry = {};
                outEntry._encrypted = server::detail::GetApplicationAlbumEntryFromAlbumEntry(albumEntry, key);
                outEntry.dateTime   = fileId.time;

                // アライメントを考慮して最後尾に挿入
                pOutList[count] = outEntry;
                ++count;
                // ソート用にヒープ化しておく
                std::push_heap(pOutList, pOutList + count, compare);
                NN_RESULT_SUCCESS;
            },
            [&](int64_t) -> nn::Result { NN_RESULT_SUCCESS; }
        );

        NN_SDK_ASSERT(m_MemorySize >= AlbumManagerThumbnailLoadSize + sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            storage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(reinterpret_cast<char*>(m_Memory) + AlbumManagerThumbnailLoadSize),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        // ヒープソートを実行
        std::sort_heap(pOutList, pOutList + count, compare);
        *pOutCount = count;
        NN_RESULT_SUCCESS;
    }


    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumError
    // @retval nn::capsrv::ResultAlbumInvalidFileId
    // @retval CheckMounted() のうち nn::capsrv::ResultAlbumInvalidStorage 以外
    // @retval OpenFileForLoadFile()
    // @retval ConvertFsResultForLoadFile()
    nn::Result AlbumManager::GetFileSize(
        size_t* pOutValue,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        *pOutValue = 0;

        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pFileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(pFileId->storage, AlbumStorageDirection_Source));

        NN_CAPSRV_CREATE_FILEPATH(filepath, *pFileId, AlbumStorageDirection_Source);
        NN_CAPSRV_OPEN_FILE_TO_READ(hFile, filepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, *pFileId, hFile);

        *pOutValue = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
    }



    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumError
    // @retval nn::capsrv::ResultAlbumInvalidFileId
    // @retval nn::capsrv::ResultAlbumReadBufferShortage
    // @retval CheckMounted() のうち nn::capsrv::ResultAlbumInvalidStorage 以外
    // @retval OpenFileForLoadFile()
    // @retval ConvertFsResultForLoadFile()
    nn::Result AlbumManager::LoadFile(
        size_t* pOutFileSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        return LoadFileEx(nullptr, nullptr, nullptr, nullptr, pOutFileSize, pBuffer, bufferSize, pFileId);
    }

    nn::Result AlbumManager::LoadFileEx(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutFileSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileSize);
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();

        auto cleanupOutput = [&](){
            if(pOutAttribute)
            {
                std::memset(pOutAttribute, 0, sizeof(AlbumFileAttribute));
            }
            if(pOutAppletData)
            {
                std::memset(pOutAppletData, 0, sizeof(AppletData));
            }
            if(pOutApplicationData)
            {
                std::memset(pOutApplicationData, 0, sizeof(ApplicationData));
            }
            if(pOutSystemReservedInfo)
            {
                std::memset(pOutSystemReservedInfo, 0, sizeof(SystemReservedInfo));
            }
            *pOutFileSize = 0;
            std::memset(pBuffer, 0, bufferSize);
        };

        NN_CAPSRV_PROCESS_START();
        cleanupOutput();
        NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pFileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(pFileId->storage, AlbumStorageDirection_Source));

        if (IsScreenShotFileImpl(*pFileId))
        {
            NN_RESULT_DO(LoadScreenShotImageFileImpl(
                pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo,  pOutFileSize, pBuffer, bufferSize, *pFileId
            ));
        }
        else if (IsMovieFileImpl(*pFileId))
        {
            NN_RESULT_DO(LoadMovieImageFileImpl(
                pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo,  pOutFileSize, pBuffer, bufferSize, *pFileId, m_Memory, m_MemorySize
            ));
        }
        else
        {
            NN_RESULT_THROW(ResultAlbumInvalidFileContents());
        }

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::LoadFileThumbnail(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        return LoadFileThumbnailEx(nullptr, nullptr, nullptr, nullptr, pOutSize, pBuffer, bufferSize, pFileId);
    }

    nn::Result AlbumManager::LoadFileThumbnailEx(
        AlbumFileAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();

        auto cleanupOutput = [&](){
            if(pOutAttribute)
            {
                std::memset(pOutAttribute, 0, sizeof(AlbumFileAttribute));
            }
            if(pOutAppletData)
            {
                std::memset(pOutAppletData, 0, sizeof(AppletData));
            }
            if(pOutApplicationData)
            {
                std::memset(pOutApplicationData, 0, sizeof(ApplicationData));
            }
            if(pOutSystemReservedInfo)
            {
                std::memset(pOutSystemReservedInfo, 0, sizeof(SystemReservedInfo));
            }
            *pOutSize = 0;
            std::memset(pBuffer, 0, bufferSize);
        };

        NN_CAPSRV_PROCESS_START();
        cleanupOutput();
        NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pFileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(pFileId->storage, AlbumStorageDirection_Source));

        if (IsScreenShotFileImpl(*pFileId))
        {
            NN_RESULT_DO(LoadScreenShotThumbnailFileImpl(
                pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, pOutSize, pBuffer, bufferSize, *pFileId, m_Memory, m_MemorySize
            ));
        }
        else if (IsMovieFileImpl(*pFileId))
        {
            NN_RESULT_DO(LoadMovieThumbnailFileImpl(
                pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, pOutSize, pBuffer, bufferSize, *pFileId, m_Memory, m_MemorySize
            ));
        }
        else
        {
            NN_RESULT_THROW(ResultAlbumInvalidFileContents());
        }

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumInvalidFileId
    // @retval CheckMounted() のうち nn::capsrv::ResultAlbumInvalidStorage 以外
    // @retval ConvertFsResultForDeleteFile()
    nn::Result AlbumManager::DeleteFile(const AlbumFileId* pFileId) NN_NOEXCEPT
    {
        NN_CAPSRV_CHECK_FILE_NOT_READ_LOCKED(*pFileId, ResultAlbumDestinationAccessCorrupted());
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pFileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(pFileId->storage, AlbumStorageDirection_Destination));

        NN_CAPSRV_CREATE_FILEPATH(filepath, *pFileId, AlbumStorageDirection_Destination);

#if 0
        // SIGLO-79412
        // ファイル削除時の事前ファイルサイズチェックを行わないようにする。
        // 計測結果では、1 ファイル当たり約 500usec 程度の高速化となる。

        // ファイルサイズを取得
        int64_t fileSize = 0;
        {
            AlbumFileHandle hFile = {};
            NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForLoad(&hFile, filepath));
            NN_UTIL_SCOPE_EXIT{
                AlbumFileManipulator::CloseAlbumFile(hFile);
            };

            NN_RESULT_DO(AlbumFileManipulator::GetFileSize(&fileSize, hFile));
            NN_RESULT_THROW_UNLESS(fileSize >= 0, ResultAlbumError());
        }

        // 大きすぎるファイルは存在しない扱い
        NN_RESULT_THROW_UNLESS(
            fileSize <= static_cast<int64_t>(nn::capsrv::server::detail::AlbumContentsAttribute::GetFileSizeLimit(pFileId->contents, *m_pEnvironmentInfo)),
            ResultAlbumFileNotFound()
        );
#endif

        {
            AlbumCacheDelta delta = {};
            NN_UTIL_SCOPE_EXIT{
                m_AlbumStorageCache.UpdateEntry(delta);
            };
            NN_RESULT_DO(AlbumFileManipulator::DeleteAlbumFile(&delta, filepath));
        }

        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval nn::capsrv::ResultAlbumInvalidFileId
    // @retval CheckMounted()
    // @retval OpenFileForLoadFile()
    // @retval ConvertFsResultForLoadFile()
    // @retval OpenFileForSaveFile()
    // @retval ConvertFsResultForSaveFile()
    nn::Result AlbumManager::StorageCopyFile(
        const AlbumFileId* pSourceFileId,
        AlbumStorageType destinationStorage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();

        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(*pSourceFileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(pSourceFileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(destinationStorage));
        NN_RESULT_DO(EnsureMounted(pSourceFileId->storage, AlbumStorageDirection_Source));
        NN_RESULT_DO(EnsureMounted(destinationStorage, AlbumStorageDirection_Destination));
        NN_RESULT_THROW_UNLESS(pSourceFileId->storage != destinationStorage, ResultAlbumInvalidStorage());

        AlbumFileId dstFileId = *pSourceFileId;
        dstFileId.storage = destinationStorage;
        // TORIAEZU: 書込み中ファイルに対する削除操作に合わせて直す必要があるかもしれない。
        NN_CAPSRV_CHECK_FILE_NOT_READ_LOCKED(dstFileId, ResultAlbumDestinationAccessCorrupted());
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(dstFileId, ResultAlbumDestinationAccessCorrupted());

        NN_CAPSRV_CREATE_FILEPATH(srcFilepath, *pSourceFileId, AlbumStorageDirection_Source);
        NN_CAPSRV_CREATE_FILEPATH(dstFilepath, dstFileId, AlbumStorageDirection_Destination);

        NN_CAPSRV_OPEN_FILE_TO_READ(hSrcFile, srcFilepath);
        NN_CAPSRV_GET_FILESIZE_TO_READ(fileSize, *pSourceFileId, hSrcFile);


        NN_RESULT_DO(CheckAlbumLimitation(destinationStorage, pSourceFileId->contents, 1));

        bool isSuccess = false;
        {
            // dst を src と同じファイルサイズで作成
            AlbumFileHandle hDstFile = {};
            {
                AlbumCacheDelta delta = {};
                NN_UTIL_SCOPE_EXIT {
                    m_AlbumStorageCache.UpdateEntry(delta);
                };
                NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForSave(
                    &hDstFile, &delta, dstFilepath, static_cast<int64_t>(fileSize)
                ));
            }
            NN_UTIL_SCOPE_EXIT{
                AlbumFileManipulator::CloseAlbumFile(hDstFile);
                if(!isSuccess)
                {
                    AlbumCacheDelta delta = {};
                    NN_UTIL_SCOPE_EXIT {
                        m_AlbumStorageCache.UpdateEntry(delta);
                    };
                    AlbumFileManipulator::DeleteAlbumFile(&delta, dstFilepath);
                }
            };

            // 中身をコピー
            int64_t copiedSize = 0;
            void* pWorkMemory = m_Memory;
            while(copiedSize < fileSize)
            {
                int64_t transferSize = std::min(fileSize - copiedSize, static_cast<int64_t>(m_MemorySize));
                size_t readSize = 0;
                NN_RESULT_DO(AlbumFileManipulator::ReadFile(&readSize, hSrcFile, copiedSize, pWorkMemory, static_cast<size_t>(transferSize)));
                NN_RESULT_DO(AlbumFileManipulator::WriteFile(hDstFile, copiedSize, pWorkMemory, static_cast<size_t>(transferSize)));
                copiedSize += transferSize;
            }

            isSuccess = true;
        }

        NN_RESULT_SUCCESS;
    }

    // @retval nn::ResultSuccess
    // @retval CheckMounted()
    // @retval ConvertFsResultForLoadDirectory()
    nn::Result AlbumManager::GetUsage(
        nn::capsrv::AlbumContentsUsage* pOutUnknownUsage,
        nn::capsrv::AlbumContentsUsage* pOutUsageArray,
        int usageArrayLength,
        AlbumStorageType storage,
        AlbumFileContentsFlag requestedContentsMask
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutUnknownUsage);
        NN_SDK_REQUIRES_NOT_NULL(pOutUsageArray);
        NN_SDK_REQUIRES_GREATER_EQUAL(usageArrayLength, 0);
        NN_STATIC_ASSERT(AlbumFileContentsFlag::StorageBitCount >= AlbumFileContentsCount);
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();

        NN_CAPSRV_PROCESS_START();
        std::memset(pOutUnknownUsage, 0, sizeof(nn::capsrv::AlbumContentsUsage));
        std::memset(pOutUsageArray, 0, sizeof(nn::capsrv::AlbumContentsUsage) * usageArrayLength);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(pOutUnknownUsage, 0, sizeof(nn::capsrv::AlbumContentsUsage)));
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(pOutUsageArray, 0, sizeof(nn::capsrv::AlbumContentsUsage) * usageArrayLength));

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        NN_RESULT_DO(EnsureMounted(storage, AlbumStorageDirection_Source));


        // 初期化
        AlbumFileContentsFlag supportedMask = {};
        supportedMask
            .Set(AlbumFileContents_ScreenShot)
            .Set(AlbumFileContents_Movie)
            .Set(AlbumFileContents_ExtraScreenShot)
            .Set(AlbumFileContents_ExtraMovie);
        AlbumFileContentsFlag contentsMask = (requestedContentsMask & supportedMask);
        for (int c = 0; c < std::min(usageArrayLength, AlbumFileContentsCount); c++)
        {
            if(contentsMask.Test(c))
            {
                pOutUsageArray[c].contents = static_cast<AlbumFileContentsType>(c);
            }
        }
        pOutUnknownUsage->flags |= AlbumContentsUsageFlag_IsUnknownContents;

        auto visitor = MakeAlbumVisitor(
            [&](const AlbumFileId&, int) -> nn::Result { NN_RESULT_SUCCESS; },
            // unmanaged directory
            [&]() -> nn::Result {
                pOutUnknownUsage->flags |= AlbumContentsUsageFlag_HasGreaterUsage;
                NN_RESULT_SUCCESS;
            },
            // managed file
            [&](int64_t size, const AlbumFileId& fileId) -> nn::Result {
                bool isUnknown = true;

                // 書込み中ファイルは無視
                NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, nn::ResultSuccess());

                if(fileId.contents >= 0 && fileId.contents < usageArrayLength)
                {
                    // カウント対象だったらカウント
                    if(contentsMask.Test(fileId.contents))
                    {
                        auto pContentsUsage = &pOutUsageArray[fileId.contents];
                        pContentsUsage->count++;
                        pContentsUsage->size += size;
                        isUnknown = false;
                    }
                }
                // カウント対象外なら Unknown に計上
                if(isUnknown)
                {
                    pOutUnknownUsage->count++;
                    pOutUnknownUsage->size += size;
                }
                NN_RESULT_SUCCESS;
            },
            // unmanaged file
            [&](int64_t size) -> nn::Result {
                pOutUnknownUsage->count++;
                pOutUnknownUsage->size += size;
                NN_RESULT_SUCCESS;
            }
        );

        NN_SDK_ASSERT(m_MemorySize >= sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            storage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(m_Memory),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumManager::GetRequiredStorageSpaceSizeToCopyAll(size_t* pOutSize, AlbumStorageType destinationStorage, AlbumStorageType sourceStorage) NN_NOEXCEPT
    {
        NN_CAPSRV_ALBUMMANAGER_REQUIRES_AND_CLEAR_WORKMEMORY();
        *pOutSize = 0;

        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(destinationStorage));
        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(sourceStorage));
        NN_RESULT_DO(EnsureMounted(sourceStorage, AlbumStorageDirection_Source));
        NN_RESULT_DO(EnsureMounted(destinationStorage, AlbumStorageDirection_Destination));

        if(sourceStorage == destinationStorage)
        {
            *pOutSize = 0;
            NN_RESULT_SUCCESS;
        }

        static const int64_t ClusterSize = 64 * 1024;
        static const int64_t MarginSize  = 1 * 1024 * 1024;
        int64_t newFileDataClusterCount = 0;
        int64_t newFileCount = 0;
        int64_t newDirectoryCount = 0;

        auto getFileSize = [](int64_t* pOutValue, const AlbumFileId& fileId, AlbumStorageDirectionType direction, const EnvironmentInfo& env) -> nn::Result {
            AlbumFileHandle h;
            AlbumFilePath path;
            NN_RESULT_DO(AlbumFileManipulator::GetFilePath(&path, fileId, direction, env));
            NN_RESULT_DO(AlbumFileManipulator::OpenAlbumFileForLoad(&h, path));
            NN_UTIL_SCOPE_EXIT {
                AlbumFileManipulator::CloseAlbumFile(h);
            };
            int64_t size = 0;
            NN_RESULT_DO(AlbumFileManipulator::GetFileSize(&size, h));
            *pOutValue = size;
            NN_RESULT_SUCCESS;
        };

        // src 側の管理対象のディレクトリ／ファイルすべてについて、 dst 側に既に存在するかを見て増分を計算する。
        auto visitor = MakeAlbumVisitor(
            // managed directory
            [&](const AlbumFileId& directoryId, int depth) -> nn::Result {
                AlbumFileId dstDirectoryId = directoryId;
                dstDirectoryId.storage = destinationStorage;

                bool isDstExist = false;
                bool isExtra = (directoryId.applicationId.value != 0);
                (void)AlbumFileManipulator::CheckSubDirectoryExist(&isDstExist, dstDirectoryId, depth, isExtra, AlbumStorageDirection_Destination);
                if(!isDstExist)
                {
                    newDirectoryCount++;
                }
                NN_RESULT_SUCCESS;
            },
            // unmanaged directory
            [&]() -> nn::Result { NN_RESULT_SUCCESS; },
            // managed file
            [&](int64_t size, const AlbumFileId& fileId) -> nn::Result {
                // 書込み中ファイルは無視
                NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, nn::ResultSuccess());

                AlbumFileId dstFileId = fileId;
                dstFileId.storage = destinationStorage;

                int64_t srcClusterCount = (size + ClusterSize - 1) / ClusterSize;

                bool isDstExist = false;
                (void)AlbumFileManipulator::CheckFileExist(&isDstExist, dstFileId, AlbumStorageDirection_Destination, *m_pEnvironmentInfo);

                if(isDstExist)
                {
                    int64_t dstFileSize = 0;
                    (void)getFileSize(&dstFileSize, dstFileId, AlbumStorageDirection_Destination, *m_pEnvironmentInfo);
                    int64_t dstClusterCount = (dstFileSize + ClusterSize - 1) / ClusterSize;
                    newFileDataClusterCount += srcClusterCount - dstClusterCount;
                }
                else
                {
                    newFileCount++;
                    newFileDataClusterCount += srcClusterCount;
                }
                NN_RESULT_SUCCESS;
            },
            // unmanaged file
            [&](int64_t) -> nn::Result { NN_RESULT_SUCCESS; }
        );

        NN_SDK_ASSERT(m_MemorySize >= sizeof(nn::fs::DirectoryEntry) * AlbumManagerDirectoryEntryBufferLength);
        NN_RESULT_DO(AlbumFileManipulator::VisitAlbumStorage(
            sourceStorage,
            visitor,
            reinterpret_cast<nn::fs::DirectoryEntry*>(m_Memory),
            AlbumManagerDirectoryEntryBufferLength,
            AlbumStorageDirection_Source,
            *m_pEnvironmentInfo
        ));

        newFileDataClusterCount = std::max(static_cast<int64_t>(0), newFileDataClusterCount);
        newFileCount            = std::max(static_cast<int64_t>(0), newFileCount);
        newDirectoryCount       = std::max(static_cast<int64_t>(0), newDirectoryCount);


        NN_CAPSRV_LOG_DEV("cluster %d, file %d, dir %d\n", static_cast<int>(newFileDataClusterCount), static_cast<int>(newFileCount), static_cast<int>(newDirectoryCount));

        int64_t result = ClusterSize * newFileDataClusterCount + ClusterSize * newDirectoryCount + (newFileCount + newDirectoryCount) * ClusterSize / 512 + MarginSize;

        *pOutSize = static_cast<size_t>(result);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumManager::LoadMakerNoteInfo(
        uint64_t* pOutFileSize,
        nn::capsrv::server::detail::MakerNoteInfo* pOutInfo,
        const AlbumFileId& fileId,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        *pOutFileSize = 0;
        std::memset(pOutInfo, 0, sizeof(*pOutInfo));

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_CHECK_FILE_NOT_WRITE_LOCKED(fileId, ResultAlbumFileNotFound());
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(&fileId, *m_pEnvironmentInfo));
        NN_RESULT_DO(EnsureMounted(fileId.storage, AlbumStorageDirection_Source));

        if (IsScreenShotFileImpl(fileId))
        {
            NN_RESULT_DO(VerifyAndLoadScreenShotMakerNoteInfo(pOutFileSize, pOutInfo, fileId, pWorkBuffer, workBufferSize));
        }
        else if (IsMovieFileImpl(fileId))
        {
            NN_RESULT_DO(VerifyAndLoadMovieMakerNoteInfo(pOutFileSize, pOutInfo, fileId, pWorkBuffer, workBufferSize));
        }
        else
        {
            NN_RESULT_THROW(ResultAlbumInvalidFileContents());
        }

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

}}}}
