﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumFileManipulator.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"
#include "../capsrvServer_ConvertFileSystemResult.h"
#include "../detail/capsrvServer_AlbumContentsAttribute.h"
#include "capsrvServer_AlbumPathUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace {

        int GetAlbumStorageWorkspaceEntrySize(int depth, bool isExtra) NN_NOEXCEPT
        {
            const int WorkspaceEntrySizeSubDirectoryTop = 20;
            const int WorkspaceEntrySizeSubDirectoryApplicationId = 100;
            const int WorkspaceEntrySizeSubDirectoryYear = 20;
            const int WorkspaceEntrySizeSubDirectoryMonth = 12;
            const int WorkspaceEntrySizeSubDirectoryDay = 31;

            const int requiredWorkSpaceEntrySizeForRegular[] =
            {
                WorkspaceEntrySizeSubDirectoryTop,
                WorkspaceEntrySizeSubDirectoryMonth,
                WorkspaceEntrySizeSubDirectoryDay,
                AlbumManagerDirectoryEntryBufferLength
                    - WorkspaceEntrySizeSubDirectoryTop
                    - WorkspaceEntrySizeSubDirectoryMonth
                    - WorkspaceEntrySizeSubDirectoryDay,
            };

            const int requiredWorkSpaceEntrySizeForExtra[] =
            {
                WorkspaceEntrySizeSubDirectoryTop,
                WorkspaceEntrySizeSubDirectoryApplicationId,
                WorkspaceEntrySizeSubDirectoryYear,
                WorkspaceEntrySizeSubDirectoryMonth,
                WorkspaceEntrySizeSubDirectoryDay,
                AlbumManagerDirectoryEntryBufferLength
                    - WorkspaceEntrySizeSubDirectoryTop
                    - WorkspaceEntrySizeSubDirectoryApplicationId
                    - WorkspaceEntrySizeSubDirectoryYear
                    - WorkspaceEntrySizeSubDirectoryMonth
                    - WorkspaceEntrySizeSubDirectoryDay,
            };

            if (isExtra)
            {
                const auto& list = requiredWorkSpaceEntrySizeForExtra;
                NN_SDK_ASSERT(sizeof(list) / sizeof(list[0]) == AlbumPathUtility::ExtraSubDirectoryDepth + 1);
                NN_SDK_ASSERT(depth >= 0 && depth <= AlbumPathUtility::ExtraSubDirectoryDepth);
                return list[depth];
            }
            else
            {
                const auto& list = requiredWorkSpaceEntrySizeForRegular;
                NN_SDK_ASSERT(sizeof(list) / sizeof(list[0]) == AlbumPathUtility::SubDirectoryDepth + 1);
                NN_SDK_ASSERT(depth >= 0 && depth <= AlbumPathUtility::SubDirectoryDepth);
                return list[depth];
            }
        }

        nn::Result CheckFilename(AlbumFileId* pOutValue, const char* pString, size_t size, const EnvironmentInfo& env, bool isExtra) NN_NOEXCEPT
        {
            const char* p = pString;
            const char* pEnd = p + size;
            AlbumFileId fileId = *pOutValue;
            NN_RESULT_DO(AlbumPathUtility::ParseAlbumFilename(&p, &fileId, p, pEnd - p, env));
            NN_RESULT_THROW_UNLESS(IsExtraAlbumFileContents(fileId.contents) == isExtra, ResultAlbumInvalidFilename());
            NN_RESULT_DO(AlbumPathUtility::ParseNullOrEoi(&p, p, pEnd - p));
            *pOutValue = fileId;
            NN_RESULT_SUCCESS;
        }

        nn::Result CheckSubDirectoryName(AlbumFileId* pOutValue, const char* pString, size_t size, int depth, bool isExtra) NN_NOEXCEPT
        {
            const char* p = pString;
            const char* pEnd = p + size;
            AlbumFileId directoryId = *pOutValue;
            NN_RESULT_DO(AlbumPathUtility::ParseSubDirectoryName(&p, &directoryId, p, pEnd - p, depth, isExtra));
            NN_RESULT_DO(AlbumPathUtility::ParseNullOrEoi(&p, p, pEnd - p));
            *pOutValue = directoryId;
            NN_RESULT_SUCCESS;
        }

        template<typename F>
        nn::Result ForeachDirectoryEntry(
            const AlbumFilePath& directoryPath,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            F function
        ) NN_NOEXCEPT
        {
            nn::fs::DirectoryHandle h = {};
            NN_RESULT_DO(ConvertFileSystemResult(
                nn::fs::OpenDirectory(&h, directoryPath.Get(), nn::fs::OpenDirectoryMode_All), directoryPath.GetDirection()
            ));
            NN_UTIL_SCOPE_EXIT {
                nn::fs::CloseDirectory(h);
            };

            for(;;)
            {
                int64_t entryCount = 0;
                nn::fs::DirectoryEntry* entries = pWorkspace;

                NN_RESULT_DO(ConvertFileSystemResult(
                    nn::fs::ReadDirectory(&entryCount, entries, h, workSize), directoryPath.GetDirection()
                ));

                if(entryCount == 0)
                {
                    break;
                }

                for(int64_t entryIndex = 0; entryIndex < entryCount; entryIndex++)
                {
                    nn::fs::DirectoryEntry& entry = entries[entryIndex];
                    NN_RESULT_DO(function(entry));
                }
            }
            NN_RESULT_SUCCESS;
        }


        nn::Result VisitAlbumStorageSubDirectory2(
            AlbumFileId directoryId,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env,
            int currentDepth,
            bool isExtra
        ) NN_NOEXCEPT
        {
            const int nextDepth = currentDepth + 1;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, isExtra);
            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);
            NN_UNUSED(workSize);

            AlbumFilePath path = {};
            NN_RESULT_DO(AlbumFileManipulator::GetSubDirectoryPath(&path, directoryId, currentDepth, isExtra, direction));

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        bool isManaged = false;

                        for(;;)
                        {
                            if(!CheckFilename(&fileId, entry.name, sizeof(entry.name), env, isExtra).IsSuccess())
                            {
                                break;
                            }
                            if(!AlbumPathUtility::CheckFileIdConsistency(fileId, directoryId).IsSuccess())
                            {
                                break;
                            }
                            if(entry.fileSize > static_cast<int64_t>(nn::capsrv::server::detail::AlbumContentsAttribute::GetFileSizeLimit(fileId.contents, env)))
                            {
                                break;
                            }

                            isManaged = true;
                            break;
                        }

                        if(isManaged)
                        {
                            if(visitor.pOnManagedFileFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedFileFoundCallbackFunction(entry.fileSize, fileId, visitor.pUserParam));
                            }
                        }
                        else
                        {
                            if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                            }
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

        nn::Result VisitAlbumStorageSubDirectory1(
            AlbumFileId directoryId,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env,
            int currentDepth,
            bool isExtra
        ) NN_NOEXCEPT
        {
            const int nextDepth = currentDepth + 1;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, isExtra);
            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);

            AlbumFilePath path = {};
            AlbumFileManipulator::GetSubDirectoryPath(&path, directoryId, currentDepth, isExtra, direction);

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, isExtra).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectory2(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth, isExtra));
                        }
                        else
                        {
                            if(visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                            }
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

        nn::Result VisitAlbumStorageSubDirectory0(
            AlbumFileId directoryId,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env,
            int currentDepth,
            bool isExtra
        ) NN_NOEXCEPT
        {
            const int nextDepth = currentDepth + 1;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, isExtra);
            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);

            AlbumFilePath path = {};
            NN_RESULT_DO(AlbumFileManipulator::GetSubDirectoryPath(&path, directoryId, currentDepth, isExtra, direction));

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, isExtra).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectory1(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth, isExtra));
                        }
                        else
                        {
                            if(visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                            }
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

        nn::Result VisitAlbumStorageSubDirectoryApplicationId(
            AlbumFileId directoryId,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env,
            int currentDepth,
            bool isExtra
        ) NN_NOEXCEPT
        {
            const int nextDepth = currentDepth + 1;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, isExtra);
            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);

            AlbumFilePath path = {};
            NN_RESULT_DO(AlbumFileManipulator::GetSubDirectoryPath(&path, directoryId, currentDepth, isExtra, direction));

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, isExtra).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectory0(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth, isExtra));
                        }
                        else
                        {
                            if(visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                            }
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

        nn::Result VisitAlbumStorageSubDirectoryExtra(
            AlbumFileId directoryId,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env,
            int currentDepth
        ) NN_NOEXCEPT
        {
            const bool isExtra = true;
            const int nextDepth = currentDepth + 1;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, isExtra);
            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);

            AlbumFilePath path = {};
            NN_RESULT_DO(AlbumFileManipulator::GetSubDirectoryPath(&path, directoryId, currentDepth, isExtra, direction));

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, isExtra).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectoryApplicationId(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth, isExtra));
                        }
                        else
                        {
                            if(!isExtra && visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                            }
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

        nn::Result VisitAlbumStorageRoot(
            AlbumStorageType storage,
            const AlbumVisitorType& visitor,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env
        ) NN_NOEXCEPT
        {
            const int nextDepth = 0;
            const int entrySize = GetAlbumStorageWorkspaceEntrySize(nextDepth, true);
            NN_SDK_ASSERT(GetAlbumStorageWorkspaceEntrySize(nextDepth, false) == entrySize);
            AlbumFileId directoryId = {};
            directoryId.storage = storage;

            NN_SDK_REQUIRES_GREATER_EQUAL(workSize, entrySize);

            AlbumFilePath path = {};
            AlbumFileManipulator::GetStorageRootPath(&path, storage, direction);

            NN_RESULT_DO(ForeachDirectoryEntry(
                path,
                pWorkspace,
                entrySize,
                [&](const nn::fs::DirectoryEntry& entry) -> nn::Result
                {
                    if(entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("D %s%s\n", path.Get(), entry.name);
                        AlbumFileId fileId = directoryId;
                        // 通常アルバムの管理対象ディレクトリか？
                        if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, false).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectory0(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth, false));
                        }
                        // Extra なアルバムの管理対象ディレクトリか？
                        else if(CheckSubDirectoryName(&fileId, entry.name, sizeof(entry.name), nextDepth, true).IsSuccess())
                        {
                            if(visitor.pOnManagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnManagedDirectoryFoundCallbackFunction(fileId, nextDepth, visitor.pUserParam));
                            }
                            NN_RESULT_DO(VisitAlbumStorageSubDirectoryExtra(fileId, visitor, pWorkspace + entrySize, workSize - entrySize, direction, env, nextDepth));
                        }
                        // 未知のディレクトリ
                        else
                        {
                            if(visitor.pOnUnmanagedDirectoryFoundCallbackFunction)
                            {
                                NN_RESULT_DO(visitor.pOnUnmanagedDirectoryFoundCallbackFunction(visitor.pUserParam));
                            }
                        }
                    }
                    else
                    {
                        NN_CAPSRV_LOG_FILE_MANIPULATOR("F %s%s\n", path.Get(), entry.name);
                        if(visitor.pOnUnmanagedFileFoundCallbackFuncton)
                        {
                            NN_RESULT_DO(visitor.pOnUnmanagedFileFoundCallbackFuncton(entry.fileSize, visitor.pUserParam));
                        }
                    }
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }

    }

    nn::Result AlbumFileManipulator::VisitAlbumStorage(
        AlbumStorageType storage,
        const AlbumVisitorType& visitor,
        nn::fs::DirectoryEntry* pWorkspace,
        int workSize,
        AlbumStorageDirectionType direction,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        return VisitAlbumStorageRoot(storage, visitor, pWorkspace, workSize, direction, env);
    }

}}}}
