﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumFileManipulator.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/capsrv/capsrv_ServiceConfig.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"
#include "../capsrvServer_ConvertFileSystemResult.h"
#include "capsrvServer_AlbumPathUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    const int AlbumFilePath::Size;

    AlbumFilePath::AlbumFilePath() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    const char* AlbumFilePath::Get() const NN_NOEXCEPT
    {
        return m_Value;
    }

    AlbumStorageType AlbumFilePath::GetStorage() const NN_NOEXCEPT
    {
        return m_Storage;
    }

    AlbumFileContentsType AlbumFilePath::GetContents() const NN_NOEXCEPT
    {
        return m_Contents;
    }

    AlbumStorageDirectionType AlbumFilePath::GetDirection() const NN_NOEXCEPT
    {
        return m_Direction;
    }

    AlbumFileHandle::AlbumFileHandle() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    nn::fs::FileHandle AlbumFileHandle::GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

    AlbumStorageType AlbumFileHandle::GetStorage() const NN_NOEXCEPT
    {
        return m_Storage;
    }

    AlbumFileContentsType AlbumFileHandle::GetContents() const NN_NOEXCEPT
    {
        return m_Contents;
    }

    AlbumStorageDirectionType AlbumFileHandle::GetDirection() const NN_NOEXCEPT
    {
        return m_Direction;
    }

    bool AlbumFileHandle::IsFlushOnCloseRequired() const NN_NOEXCEPT
    {
        return m_IsFlushOnCloseRequired;
    }

    //-----------------------------------------------------------------

    namespace {
        nn::Result EmitAlbumStructureCorruptedError(AlbumStorageDirectionType direction) NN_NOEXCEPT
        {
            switch(direction)
            {
            case AlbumStorageDirection_Source:
                NN_RESULT_THROW(ResultAlbumSourceStructureCorrupted());
            case AlbumStorageDirection_Destination:
                NN_RESULT_THROW(ResultAlbumDestinationStructureCorrupted());
            default:
                NN_RESULT_THROW(ResultAlbumStructureCorrupted());
            }
        }

        nn::Result CreateSubDirectories_I(char* path, int length, AlbumStorageDirectionType direction) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(path[length], '/');
            path[length] = '\0';
            NN_UTIL_SCOPE_EXIT {
                path[length] = '/';
            };
            NN_RESULT_TRY(nn::fs::CreateDirectory(path))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    nn::fs::DirectoryEntryType type = {};
                    nn::fs::GetEntryType(&type, path);
                    // 存在するのがディレクトリなら OK
                    NN_RESULT_THROW_UNLESS(type == nn::fs::DirectoryEntryType_Directory, EmitAlbumStructureCorruptedError(direction));
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::Result CreateSubDirectories(char* path, AlbumStorageDirectionType direction, bool isExtra) NN_NOEXCEPT
        {
            if (isExtra)
            {
                int length = AlbumPathUtility::MountNameLength;
                length += 2 + AlbumPathUtility::ExtraSubDirectoryNameLength0;
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::ExtraSubDirectoryNameLength1;
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::ExtraSubDirectoryNameLength2;
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::ExtraSubDirectoryNameLength3;
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::ExtraSubDirectoryNameLength4;
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
            }
            else
            {
                int length = AlbumPathUtility::MountNameLength;          // ST
                length += 2 + AlbumPathUtility::SubDirectoryNameLength0; // ST:/YYYY
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::SubDirectoryNameLength1; // ST:/YYYY/MM
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
                length += 1 + AlbumPathUtility::SubDirectoryNameLength2; // ST:/YYYY/MM/DD
                NN_RESULT_DO(CreateSubDirectories_I(path, length, direction));
            }

            NN_SDK_ASSERT_MINMAX(nn::util::Strnlen(path, AlbumPathUtility::ExtraPathSize), AlbumPathUtility::PathSize - 1, AlbumPathUtility::ExtraPathSize - 1);
            NN_RESULT_SUCCESS;
        }

        nn::Result DeleteSubDirectoriesIfEmpty_I(char* path, int length, AlbumStorageDirectionType direction) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(length, 0, AlbumPathUtility::ExtraPathSize);
            NN_RESULT_THROW_UNLESS(path[length] == '/', ResultAlbumInvalidFilename());
            path[length] = '\0';
            NN_UTIL_SCOPE_EXIT {
                path[length] = '/';
            };
            NN_RESULT_DO(ConvertFileSystemResult(nn::fs::DeleteDirectory(path), direction));
            NN_CAPSRV_LOG_DEV("[album] Deleted empty subdirectory: %s\n", path);
            NN_RESULT_SUCCESS;
        }

        nn::Result DeleteSubDirectoriesIfEmpty(char* path, AlbumStorageDirectionType direction) NN_NOEXCEPT
        {
            int length =
                AlbumPathUtility::MountNameLength +             // ST
                2 + AlbumPathUtility::SubDirectoryNameLength0 + // :/YYYY
                1 + AlbumPathUtility::SubDirectoryNameLength1 + // /MM
                1 + AlbumPathUtility::SubDirectoryNameLength2;  // /DD
            NN_RESULT_DO(DeleteSubDirectoriesIfEmpty_I(path, length, direction)); // ST:/YYYY/MM/DD
            NN_SDK_ASSERT_MINMAX(nn::util::Strnlen(path, AlbumPathUtility::ExtraPathSize), AlbumPathUtility::PathSize - 1, AlbumPathUtility::ExtraPathSize - 1);
            length -= 1 + AlbumPathUtility::SubDirectoryNameLength2;
            NN_RESULT_DO(DeleteSubDirectoriesIfEmpty_I(path, length, direction)); // ST:/YYYY/MM
            NN_SDK_ASSERT_MINMAX(nn::util::Strnlen(path, AlbumPathUtility::ExtraPathSize), AlbumPathUtility::PathSize - 1, AlbumPathUtility::ExtraPathSize - 1);
            length -= 1 + AlbumPathUtility::SubDirectoryNameLength1;
            NN_RESULT_DO(DeleteSubDirectoriesIfEmpty_I(path, length, direction)); // ST:/YYYY
            NN_SDK_ASSERT_MINMAX(nn::util::Strnlen(path, AlbumPathUtility::ExtraPathSize), AlbumPathUtility::PathSize - 1, AlbumPathUtility::ExtraPathSize - 1);
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result AlbumFileManipulator::GetFilePath(
        AlbumFilePath* pOutValue,
        const AlbumFileId& fileId,
        AlbumStorageDirectionType direction,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        NN_RESULT_DO(AlbumPathUtility::ValidateFileId(&fileId, env));
        AlbumFilePath path = {};
        int n = 0;
        AlbumPathUtility::GetFilepath(&n, path.m_Value, sizeof(path.m_Value), fileId, env);
        path.m_Storage = fileId.storage;
        path.m_Contents = fileId.contents;
        path.m_Direction = direction;
        *pOutValue = path;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::GetSubDirectoryPath(
        AlbumFilePath* pOutValue,
        const AlbumFileId& directoryId,
        int depth,
        bool isExtra,
        AlbumStorageDirectionType direction
    ) NN_NOEXCEPT
    {
        NN_RESULT_DO(AlbumPathUtility::ValidateSubDirectoryId(&directoryId, depth, isExtra));
        AlbumFilePath path = {};
        int n = 0;
        AlbumPathUtility::GetSubDirectoryPath(&n, path.m_Value, sizeof(path.m_Value), directoryId, depth, isExtra);
        path.m_Storage = directoryId.storage;
        path.m_Contents = AlbumFileContents_Directory;
        path.m_Direction = direction;
        *pOutValue = path;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::GetStorageRootPath(
        AlbumFilePath* pOutValue,
        AlbumStorageType storage,
        AlbumStorageDirectionType direction
    ) NN_NOEXCEPT
    {
        NN_RESULT_DO(AlbumPathUtility::ValidateStorage(storage));
        AlbumFilePath path = {};
        int n = 0;
        AlbumPathUtility::GetStorageRootPath(&n, path.m_Value, sizeof(path.m_Value), storage);
        path.m_Storage = storage;
        path.m_Contents = AlbumFileContents_Directory;
        path.m_Direction = direction;
        *pOutValue = path;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::OpenAlbumFileForSaveImpl(
        AlbumFileHandle* pOutHandle,
        AlbumCacheDelta* pOutCacheDelta,
        AlbumFilePath& filepath,
        int64_t fileSize,
        int openMode
    ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsQuestMode(), ResultAlbumIsFull());

        {
            AlbumFilePath path = filepath;
            NN_RESULT_DO(ConvertFileSystemResult(CreateSubDirectories(path.m_Value, path.m_Direction, IsExtraAlbumFileContents(path.m_Contents)), path.GetDirection()));
        }

        nn::fs::FileHandle hFile = {};
        AlbumCacheDelta delta = {};
        nn::Result result = nn::fs::CreateFile(filepath.Get(), fileSize);
        if(result.IsSuccess())
        {
            NN_RESULT_DO(ConvertFileSystemResult(nn::fs::OpenFile(&hFile, filepath.Get(), openMode), filepath.GetDirection()));
            delta = AlbumCacheDelta(filepath.GetStorage(), filepath.GetContents(), 1);
        }
        else if(nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            NN_RESULT_DO(ConvertFileSystemResult(nn::fs::OpenFile(&hFile, filepath.Get(), openMode), filepath.GetDirection()));
            NN_RESULT_DO(ConvertFileSystemResult(nn::fs::SetFileSize(hFile, fileSize), filepath.GetDirection()));
            delta = AlbumCacheDelta(filepath.GetStorage(), filepath.GetContents(), 0);
        }
        else
        {
            NN_RESULT_THROW(ConvertFileSystemResult(result, filepath.m_Direction));
        }

        pOutHandle->m_Handle = hFile;
        pOutHandle->m_Storage = filepath.m_Storage;
        pOutHandle->m_Contents = filepath.m_Contents;
        pOutHandle->m_Direction = filepath.m_Direction;
        pOutHandle->m_IsFlushOnCloseRequired = true;
        *pOutCacheDelta = delta;
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::OpenAlbumFileForSave(
        AlbumFileHandle* pOutHandle,
        AlbumCacheDelta* pOutCacheDelta,
        AlbumFilePath& filepath,
        int64_t fileSize
    ) NN_NOEXCEPT
    {
        return OpenAlbumFileForSaveImpl(pOutHandle, pOutCacheDelta, filepath, fileSize, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    }

    nn::Result AlbumFileManipulator::OpenAlbumFileForWriteStream(
        AlbumFileHandle* pOutHandle,
        AlbumCacheDelta* pOutCacheDelta,
        AlbumFilePath& filepath,
        int64_t fileSize
    ) NN_NOEXCEPT
    {
        return OpenAlbumFileForSaveImpl(pOutHandle, pOutCacheDelta, filepath, fileSize, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    }

    nn::Result AlbumFileManipulator::OpenAlbumFileForLoad(
        AlbumFileHandle* pOutHandle,
        const AlbumFilePath& filepath
    ) NN_NOEXCEPT
    {
        nn::fs::FileHandle hFile;
        NN_RESULT_DO(ConvertFileSystemResult(nn::fs::OpenFile(&hFile, filepath.m_Value, nn::fs::OpenMode_Read), filepath.m_Direction));
        pOutHandle->m_Handle = hFile;
        pOutHandle->m_Storage = filepath.m_Storage;
        pOutHandle->m_Contents = filepath.m_Contents;
        pOutHandle->m_Direction = filepath.m_Direction;
        pOutHandle->m_IsFlushOnCloseRequired = false;
        NN_RESULT_SUCCESS;
    }

    void AlbumFileManipulator::CloseAlbumFile(const AlbumFileHandle& handle) NN_NOEXCEPT
    {
        if(handle.m_IsFlushOnCloseRequired)
        {
            (void)nn::fs::FlushFile(handle.m_Handle);
        }
        nn::fs::CloseFile(handle.m_Handle);
    }

    nn::Result AlbumFileManipulator::DeleteAlbumFile(
        AlbumCacheDelta* pOutCacheDelta,
        const AlbumFilePath& filepath
    ) NN_NOEXCEPT
    {
        NN_RESULT_DO(ConvertFileSystemResult(
            nn::fs::DeleteFile(filepath.m_Value),
            filepath.m_Direction
        ));

        // サブディレクトリが空になっていたら削除する
        {
            auto path = filepath; // 書き換えが発生するので別変数にコピー
            (void)DeleteSubDirectoriesIfEmpty(path.m_Value, filepath.m_Direction);
        }

        *pOutCacheDelta = AlbumCacheDelta(filepath.m_Storage, filepath.m_Contents, -1);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::GetFileSize(int64_t* pOutValue, const AlbumFileHandle& handle) NN_NOEXCEPT
    {
        NN_RESULT_DO(ConvertFileSystemResult(
            nn::fs::GetFileSize(pOutValue, handle.m_Handle),
            handle.m_Direction
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::ReadFile(size_t* pOutReadSize, const AlbumFileHandle& handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(ConvertFileSystemResult(
            nn::fs::ReadFile(pOutReadSize, handle.m_Handle, offset, buffer,size),
            handle.m_Direction
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::WriteFile(const AlbumFileHandle& handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(ConvertFileSystemResult(
            nn::fs::WriteFile(
                handle.m_Handle,
                offset,
                buffer,
                size,
                handle.m_IsFlushOnCloseRequired ? nn::fs::WriteOption::MakeValue(0) : nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
            ),
            handle.m_Direction
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::ResizeFile(const AlbumFileHandle& handle, int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(size >= 0, ResultAlbumOutOfRange());
        NN_RESULT_DO(ConvertFileSystemResult(nn::fs::SetFileSize(handle.m_Handle, size), handle.m_Direction));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::CheckFileExist(
        bool* pOutValue,
        const AlbumFileId& fileId,
        AlbumStorageDirectionType direction,
        const EnvironmentInfo& env
    ) NN_NOEXCEPT
    {
        AlbumFilePath path = {};
        NN_RESULT_DO(GetFilePath(&path, fileId, direction, env));

        nn::fs::DirectoryEntryType entryType = {};
        nn::Result result = nn::fs::GetEntryType(&entryType, path.m_Value);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                *pOutValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(ConvertFileSystemResult(result, direction));
            }
        NN_RESULT_END_TRY;

        *pOutValue = (entryType == nn::fs::DirectoryEntryType_File);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumFileManipulator::CheckSubDirectoryExist(
        bool* pOutValue,
        const AlbumFileId& directoryId,
        int depth,
        bool isExtra,
        AlbumStorageDirectionType direction
    ) NN_NOEXCEPT
    {
        AlbumFilePath path = {};
        NN_RESULT_DO(GetSubDirectoryPath(&path, directoryId, depth, isExtra, direction));

        nn::fs::DirectoryEntryType entryType = {};
        nn::Result result = nn::fs::GetEntryType(&entryType, path.m_Value);
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                *pOutValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(ConvertFileSystemResult(result, direction));
            }
        NN_RESULT_END_TRY;

        *pOutValue = (entryType == nn::fs::DirectoryEntryType_Directory);
        NN_RESULT_SUCCESS;
    }

}}}}
