﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumFileLockTable.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    const int AlbumFileLockTable::TableSize;

    //------------------------------------------

    AlbumFileLockTable::AlbumFileLockTable() NN_NOEXCEPT
    {
    }

    void AlbumFileLockTable::Initialize() NN_NOEXCEPT
    {
        for(int i = 0; i < TableSize; i++)
        {
            m_LockedFileList[i] = nn::util::nullopt;
        }
    }

    void AlbumFileLockTable::Finalize() NN_NOEXCEPT
    {
        for(int i = 0; i < TableSize; i++)
        {
            m_LockedFileList[i] = nn::util::nullopt;
        }
    }

    //------------------------------------------

    nn::Result AlbumFileLockTable::LockFile(const AlbumFileId& fileId) NN_NOEXCEPT
    {
        // 例え既にロック中のファイルがあっても新しくエントリーを使用する。
        for(int i = 0; i < TableSize; i++)
        {
            if(!m_LockedFileList[i])
            {
                m_LockedFileList[i] = fileId;
                NN_RESULT_SUCCESS;
            }
        }

        // テーブルサイズは開けるファイルの最大数に対して十分な数用意しているので通常、失敗しない。
        NN_SDK_ASSERT(false, "AlbumFileLockTable: Table is full");
        NN_RESULT_THROW(ResultInternalError());
    }

    nn::Result AlbumFileLockTable::UnlockFile(const AlbumFileId& fileId) NN_NOEXCEPT
    {
        // 一致するものを 1 つ探して削除する
        for(int i = 0; i < TableSize; i++)
        {
            if(m_LockedFileList[i])
            {
                AlbumFileId e = *m_LockedFileList[i];
                if(0 == std::memcmp(&e, &fileId, sizeof(AlbumFileId)))
                {
                    m_LockedFileList[i] = nn::util::nullopt;
                    NN_RESULT_SUCCESS;
                }
            }
        }

        // 過剰に Unlock してはいけない
        NN_SDK_ASSERT(false, "AlbumFileLockTable: Unlocking file is not found");
        NN_RESULT_THROW(ResultInternalError());
    }

    bool AlbumFileLockTable::IsFileLocked(const AlbumFileId& fileId) const NN_NOEXCEPT
    {
        for(int i = 0; i < TableSize; i++)
        {
            if(m_LockedFileList[i])
            {
                AlbumFileId e = *m_LockedFileList[i];
                if(0 == std::memcmp(&e, &fileId, sizeof(AlbumFileId)))
                {
                    return true;
                }
            }
        }
        return false;
    }

}}}}
