﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/movie/capsrv_MovieMetaDataBuilder.h>

#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>

NN_STATIC_ASSERT(sizeof(nn::capsrv::movie::MovieMetaInfoData) == nn::capsrv::movie::MovieMetaInfoDataSize);

namespace nn{ namespace capsrv{ namespace movie{

    MovieMetaDataBuilder::MovieMetaDataBuilder() NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
    }

    bool MovieMetaDataBuilder::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialzed;
    }

    void MovieMetaDataBuilder::Initialize(MovieMetaDataVersion version, void* buffer, size_t size, const AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(buffer);
        NN_SDK_REQUIRES_ALIGNED(buffer, NN_ALIGNOF(MovieMetaData));
        NN_SDK_REQUIRES_GREATER_EQUAL(size, MovieMetaDataSize);
        NN_SDK_REQUIRES_EQUAL(version, MovieMetaDataVersion_1);

        std::memset(buffer, 0, size);

        m_MovieMetaDataVersion = version;
        m_IsInitialzed = true;
        m_FileId = fileId;
        m_pMetaData = reinterpret_cast<MovieMetaData*>(buffer);

        // infoData 部分(12KB)を大き目のデータ置き場として使う
        NN_STATIC_ASSERT(sizeof(m_pMetaData->infoData) >= sizeof(AppletData));
        m_pAppletData = reinterpret_cast<AppletData*>(&m_pMetaData->infoData);
        // infoData 部分(12KB)の先頭に AppletData 置き場があり、
        // その次の部分を ApplicationData 置き場として使う
        m_pApplicationData = reinterpret_cast<ApplicationData*>(m_pAppletData + 1);
    }

    void MovieMetaDataBuilder::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        std::memset(this, 0, sizeof(*this));
        NN_SDK_ASSERT(!IsInitialized());
    }

    void MovieMetaDataBuilder::SetMovieDataSize(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsMovieDataSizeReady);
        NN_SDK_REQUIRES(!m_IsBuilt);
        m_MovieDataSize = size;
        m_IsMovieDataSizeReady = true;
    }

    void MovieMetaDataBuilder::SetImageDataRgba(
        const void* rawImage,
        size_t rawImageSize,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsImageDataReady);
        NN_SDK_REQUIRES(!m_IsBuilt);
        m_pRawImageRgba = rawImage;
        m_RawImageRgbaSize = rawImageSize;
        m_RawImageWidth = width;
        m_RawImageHeight = height;
        m_IsImageDataReady = true;
    }

    void MovieMetaDataBuilder::SetImageDataNv12(
        const void* rawImageNv12Y,
        size_t rawImageNv12YSize,
        const void* rawImageNv12Uv,
        size_t rawImageNv12UvSize,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsImageDataReady);
        NN_SDK_REQUIRES(!m_IsBuilt);
        m_pRawImageNv12Y = rawImageNv12Y;
        m_RawImageNv12YSize = rawImageNv12YSize;
        m_pRawImageNv12Uv = rawImageNv12Uv;
        m_RawImageNv12UvSize = rawImageNv12UvSize;
        m_RawImageWidth = width;
        m_RawImageHeight = height;
        m_IsImageDataReady = true;
    }

    void MovieMetaDataBuilder::SetDescription(AlbumFileDescriptionType description) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsDescriptionReady);
        NN_SDK_REQUIRES(!m_IsBuilt);
        m_Description = description;
        m_IsDescriptionReady = true;
    }

    void MovieMetaDataBuilder::SetAttribute(
        int frameCount,
        int frameRateNumerator,
        int frameRateDenominator,
        int dataDurationMilliseconds,
        int keyFrameIntervalFrames,
        bool isCopyrightImageComposited,
        ScreenShotSizeType movieSize,
        ScreenShotOrientationType orientation
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsAttributeReady);
        NN_SDK_REQUIRES(!m_IsBuilt);
        m_Attribute = {};
        m_Attribute.frameCount = frameCount;
        m_Attribute.frameRateNumerator = frameRateNumerator;
        m_Attribute.frameRateDenominator = frameRateDenominator;
        m_Attribute.dataDurationMilliseconds = dataDurationMilliseconds;
        m_Attribute.keyFrameInterval = keyFrameIntervalFrames;
        m_Attribute.flags |= (isCopyrightImageComposited ? nn::capsrv::detail::ScreenShotAttributeFlag_IsCopyrightImageComposited : 0);
        m_Attribute.movieSize = movieSize;
        m_Attribute.orientation = orientation;
        m_IsAttributeReady = true;
    }

    void MovieMetaDataBuilder::SetAppletData(
        const AppletData& appletData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsBuilt);
        *m_pAppletData = appletData;
    }

    void MovieMetaDataBuilder::SetApplicationData(
        const ApplicationData& applicationData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!m_IsBuilt);
        *m_pApplicationData = applicationData;
    }

    nn::Result MovieMetaDataBuilder::Build() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(m_IsMovieDataSizeReady);
        NN_SDK_REQUIRES(m_IsImageDataReady);
        NN_SDK_REQUIRES(m_IsDescriptionReady);
        NN_SDK_REQUIRES(m_IsAttributeReady);
        NN_SDK_REQUIRES(!m_IsBuilt);

        std::memset(m_pMetaData->imageData, 0, sizeof(m_pMetaData->imageData));
        m_Attribute.size        = ScreenShotSize_1280x720;
        m_Attribute.description = m_Description;

        size_t size = 0;
        int64_t makerNoteOffset = 0;
        int64_t makerNoteSize = 0;
        if(m_pRawImageNv12Y)
        {
            NN_RESULT_DO(nn::capsrv::CreateProtoMovieMetaDataNv12(
                &size,
                &makerNoteOffset,
                &makerNoteSize,
                m_pMetaData->imageData,
                sizeof(m_pMetaData->imageData),
                const_cast<void*>(m_pRawImageNv12Y),
                m_RawImageNv12YSize,
                const_cast<void*>(m_pRawImageNv12Uv),
                m_RawImageNv12UvSize,
                m_RawImageWidth,
                m_RawImageHeight,
                m_FileId,
                m_Attribute,
                *m_pAppletData,
                *m_pApplicationData
            ));
        }
        else if(m_pRawImageRgba)
        {
            NN_RESULT_DO(nn::capsrv::CreateProtoMovieMetaDataRgba(
                &size,
                &makerNoteOffset,
                &makerNoteSize,
                m_pMetaData->imageData,
                sizeof(m_pMetaData->imageData),
                m_pRawImageRgba,
                m_RawImageRgbaSize,
                m_RawImageWidth,
                m_RawImageHeight,
                m_FileId,
                m_Attribute,
                *m_pAppletData,
                *m_pApplicationData
            ));
        }
        else
        {
            NN_ABORT("Can't detect raw image format (supported NV12/RGBA)");
        }

        // infoData 部分は Meta 生成用のバッファに使っていたのでクリアする
        std::memset(&m_pMetaData->infoData, 0, sizeof(m_pMetaData->infoData));

        m_pMetaData->infoData.movieDataSize = m_MovieDataSize;
        m_pMetaData->infoData.imageDataSize = size;
        m_pMetaData->infoData.version = m_MovieMetaDataVersion;
        m_MakerNoteOffset = makerNoteOffset;
        m_MakerNoteSize = makerNoteSize;
        m_IsBuilt = true;
        NN_RESULT_SUCCESS;
    }


    nn::Result MovieMetaDataBuilder::WriteToStream(AlbumMovieWriteStreamHandle streamHandle) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(m_IsBuilt);

        uint32_t makerNoteVersion = 0;
        std::memcpy(&makerNoteVersion, reinterpret_cast<const char*>(m_pMetaData) + m_MakerNoteOffset, sizeof(makerNoteVersion));

        NN_RESULT_DO(nn::capsrv::WriteMetaToAlbumMovieWriteStream(
            streamHandle,
            m_pMetaData,
            sizeof(movie::MovieMetaData),
            makerNoteVersion,
            m_MakerNoteOffset,
            m_MakerNoteSize
        ));
        NN_RESULT_SUCCESS;
    }

    uint64_t MovieMetaDataBuilder::GetMakerNoteVersion() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        uint32_t makerNoteVersion = 0;
        std::memcpy(&makerNoteVersion, reinterpret_cast<const char*>(m_pMetaData) + m_MakerNoteOffset, sizeof(makerNoteVersion));
        return makerNoteVersion;
    }

    int64_t MovieMetaDataBuilder::GetMakerNoteOffset() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_MakerNoteOffset;
    }

    int64_t MovieMetaDataBuilder::GetMakerNoteSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_MakerNoteSize;
    }

}}}
