﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/movie/capsrv_CachedMovieStream.h>

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>

namespace nn{ namespace capsrv { namespace movie{

    const int64_t CachedMovieStream::InvalidChunkIndex;

    //----------------------------------------------------------------------------
    CachedMovieStream::CachedMovieStream() NN_NOEXCEPT
        : m_Accessor()
        , m_Strategy()
        , m_pWorkMemory(nullptr)
        , m_WorkMemorySize()
        , m_CacheChunkSize(0)
        , m_CacheChunkCountMax(0)
        , m_pCacheChunkList(nullptr)
        , m_CacheChunkCount(0)
        , m_FileChunkCount(0)
        , m_ExplicitChunkCount(0)
    {
    }

    size_t CachedMovieStream::GetRequiredWorkMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(cacheChunkCount, 1);
        size_t headSize = static_cast<size_t>(sizeof(CacheChunkHead) * cacheChunkCount);
        size_t dataSize = static_cast<size_t>(cacheChunkSize * cacheChunkCount);

        return headSize + dataSize;
    }

    size_t CachedMovieStream::GetRequiredWorkMemoryAlignment() NN_NOEXCEPT
    {
        return NN_ALIGNOF(CacheChunkHead);
    }

    void CachedMovieStream::Initialize(
        int64_t cacheChunkSize,
        int64_t cacheChunkCount,
        const MovieStreamAccessor& accessor,
        const MovieStreamCacheStrategy& strategy,
        void* workMemory,
        size_t workMemorySize,
        int64_t initialChunkCount
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(cacheChunkCount, 1);
        NN_SDK_REQUIRES_NOT_NULL(accessor.read);
        NN_SDK_REQUIRES_NOT_NULL(strategy.notifyFetched);
        NN_SDK_REQUIRES_NOT_NULL(strategy.notifyInvalidated);
        NN_SDK_REQUIRES_NOT_NULL(strategy.queryChunkIndexToInvalidate);
        NN_SDK_REQUIRES_NOT_NULL(workMemory);
        NN_SDK_REQUIRES_ALIGNED(workMemory, GetRequiredWorkMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, GetRequiredWorkMemorySize(cacheChunkSize, cacheChunkCount));
        NN_SDK_REQUIRES_GREATER_EQUAL(initialChunkCount, 0);

        m_Accessor = accessor;
        m_Strategy = strategy;
        m_pWorkMemory = workMemory;
        m_WorkMemorySize = workMemorySize;
        m_CacheChunkSize = cacheChunkSize;
        m_CacheChunkCountMax = cacheChunkCount;
        m_CacheChunkCount = 0;
        m_FileChunkCount = initialChunkCount;
        m_ExplicitChunkCount = 0;

        // workmemory を割り当て
        {
            auto pMemory = reinterpret_cast<char*>(workMemory);
            m_pCacheChunkList = reinterpret_cast<CacheChunkHead*>(pMemory + static_cast<ptrdiff_t>(cacheChunkSize * cacheChunkCount));
            std::memset(m_pCacheChunkList, 0, static_cast<size_t>(sizeof(CacheChunkHead) * cacheChunkCount));
            for(int64_t i = 0; i < cacheChunkCount; i++)
            {
                auto& e = m_pCacheChunkList[i];
                e.chunkIndex = InvalidChunkIndex;
                e.flag = CacheChunkFlag_None;
                e.data = pMemory + static_cast<ptrdiff_t>(cacheChunkSize * i);
            }
        }
    }

    void CachedMovieStream::Finalize() NN_NOEXCEPT
    {
        *this = CachedMovieStream();
    }

    //----------------------------------------------------------------------------

    // NOTE: 必要なら Mutex はこのレイヤで。

    nn::Result CachedMovieStream::Read(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT
    {
        return ReadImpl(pOutReadSize, buffer, size, offset);
    }

    nn::Result CachedMovieStream::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        return WriteImpl(offset, buffer, size);
    }

    nn::Result CachedMovieStream::Resize(int64_t size) NN_NOEXCEPT
    {
        return ResizeImpl(size);
    }

    nn::Result CachedMovieStream::Flush() NN_NOEXCEPT
    {
        return FlushImpl();
    }

    //----------------------------------------------------------------------------

    int64_t CachedMovieStream::CalculateAvailableChunkCount() const NN_NOEXCEPT
    {
        // ファイルとキャッシュの全体の有効チャンク数
        int64_t availableChunkCount = m_FileChunkCount;
        for(int64_t i = 0; i < m_CacheChunkCountMax; i++)
        {
            auto& e = m_pCacheChunkList[i];
            if(e.chunkIndex + 1 > availableChunkCount && (e.flag & CacheChunkFlag_Dirty) != 0)
            {
                availableChunkCount = e.chunkIndex + 1;
            }
        }
        return availableChunkCount;
    }

    CachedMovieStream::CacheChunkHead* CachedMovieStream::FindCacheChunk(int64_t chunkIndex) NN_NOEXCEPT
    {
        for(int64_t i = 0; i < m_CacheChunkCountMax; i++)
        {
            if(m_pCacheChunkList[i].chunkIndex == chunkIndex)
            {
                return &m_pCacheChunkList[i];
            }
        }
        return nullptr;
    }

    CachedMovieStream::CacheChunkHead* CachedMovieStream::FindEmptyCacheChunk() NN_NOEXCEPT
    {
        if(m_CacheChunkCount == m_CacheChunkCountMax)
        {
            return nullptr;
        }

        for(int64_t i = 0; i < m_CacheChunkCountMax; i++)
        {
            if(m_pCacheChunkList[i].chunkIndex == InvalidChunkIndex)
            {
                return &m_pCacheChunkList[i];
            }
        }
        return nullptr;
    }


    nn::Result CachedMovieStream::FlushCacheChunk(CacheChunkHead* pChunk) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(pChunk->chunkIndex, InvalidChunkIndex);
        NN_SDK_ASSERT_GREATER(m_CacheChunkCount, 0);

        auto chunkIndex = pChunk->chunkIndex;
        if((pChunk->flag & CacheChunkFlag_Dirty) != 0)
        {
            int64_t offset = m_CacheChunkSize * chunkIndex;
            NN_RESULT_DO(m_Accessor.write(offset, pChunk->data, static_cast<size_t>(m_CacheChunkSize), m_Accessor.userData));
            pChunk->chunkIndex = InvalidChunkIndex;
            pChunk->flag = CacheChunkFlag_None;

            // 間が飛んだ分をゼロ埋め
            NN_RESULT_DO(WriteOutZeroChunkRange(m_FileChunkCount, chunkIndex, pChunk));
            NN_SDK_ASSERT_LESS_EQUAL(chunkIndex, m_FileChunkCount);

            m_FileChunkCount = std::max(m_FileChunkCount, chunkIndex + 1);
        }
        else
        {
            pChunk->chunkIndex = InvalidChunkIndex;
            pChunk->flag = CacheChunkFlag_None;
        }

        m_CacheChunkCount--;
        m_Strategy.notifyInvalidated(chunkIndex, m_Strategy.userData);
        NN_RESULT_SUCCESS;
    }

    void CachedMovieStream::InvalidateCacheChunk(CacheChunkHead* pChunk) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(pChunk->chunkIndex, InvalidChunkIndex);
        NN_SDK_ASSERT_GREATER(m_CacheChunkCount, 0);

        auto chunkIndex = pChunk->chunkIndex;
        pChunk->chunkIndex = InvalidChunkIndex;
        pChunk->flag = CacheChunkFlag_None;
        m_CacheChunkCount--;
        m_Strategy.notifyInvalidated(chunkIndex, m_Strategy.userData);
    }

    nn::Result CachedMovieStream::FetchCacheChunk(CacheChunkHead* pChunk, int64_t chunkIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(pChunk->chunkIndex, InvalidChunkIndex);
        NN_SDK_REQUIRES_GREATER_EQUAL(chunkIndex, 0);
        NN_SDK_ASSERT_LESS(m_CacheChunkCount, m_CacheChunkCountMax);

        if(chunkIndex >= m_FileChunkCount)
        {
            std::memset(pChunk->data, 0, static_cast<size_t>(m_CacheChunkSize));
        }
        else
        {
            int64_t offset = m_CacheChunkSize * chunkIndex;
            size_t readSize = 0;
            NN_RESULT_DO(m_Accessor.read(&readSize, pChunk->data, static_cast<size_t>(m_CacheChunkSize), offset, m_Accessor.userData));
            if(readSize < static_cast<size_t>(m_CacheChunkSize))
            {
                std::memset(pChunk->data + readSize, 0, static_cast<size_t>(m_CacheChunkSize) - readSize);
            }
        }
        pChunk->chunkIndex = chunkIndex;
        pChunk->flag = CacheChunkFlag_None;
        m_CacheChunkCount++;
        m_Strategy.notifyFetched(chunkIndex, m_Strategy.userData);
        NN_RESULT_SUCCESS;
    }

    nn::Result CachedMovieStream::WriteOutZeroChunkRange(int64_t chunkIndexBeg, int64_t chunkIndexEnd, CacheChunkHead* pWorkChunk) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(chunkIndexBeg, 0);
        NN_SDK_REQUIRES_EQUAL(pWorkChunk->chunkIndex, InvalidChunkIndex);

        if(chunkIndexBeg >= chunkIndexEnd)
        {
            NN_RESULT_SUCCESS;
        }

        std::memset(pWorkChunk->data, 0, static_cast<size_t>(m_CacheChunkSize));
        for(int64_t i = chunkIndexBeg; i < chunkIndexEnd; i++)
        {
            NN_RESULT_DO(m_Accessor.write(m_CacheChunkSize * i, pWorkChunk->data, static_cast<size_t>(m_CacheChunkSize), m_Accessor.userData));
            m_FileChunkCount = std::max(m_FileChunkCount, i + 1);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedMovieStream::RequestCacheChunk(CacheChunkHead** pOutChunk, int64_t chunkIndex) NN_NOEXCEPT
    {
        // キャッシュ中のものを探す
        CacheChunkHead* pChunk = FindCacheChunk(chunkIndex);
        if(pChunk)
        {
            *pOutChunk = pChunk;
            NN_RESULT_SUCCESS;
        }

        // キャッシュしていなければ、新しくキャッシュを作る
        pChunk = FindEmptyCacheChunk();
        if(!pChunk)
        {
            // キャッシュを解放するチャンクを選ぶ
            auto chunkIndexToInvalidate = m_Strategy.queryChunkIndexToInvalidate(m_Strategy.userData);
            auto pChunkToInvalidate = FindCacheChunk(chunkIndexToInvalidate);
            NN_SDK_ASSERT_NOT_NULL(pChunkToInvalidate);
            // チャンクを Flush (&Invalidate)
            NN_RESULT_DO(FlushCacheChunk(pChunkToInvalidate));
            pChunk = pChunkToInvalidate;
        }
        NN_SDK_ASSERT_NOT_NULL(pChunk);

        NN_RESULT_DO(FetchCacheChunk(pChunk, chunkIndex));
        *pOutChunk = pChunk;
        NN_RESULT_SUCCESS;
    }

    void CachedMovieStream::InvalidateCacheChunkGreaterEqualIndex(int64_t chunkIndexBegin) NN_NOEXCEPT
    {
        for(int64_t i = 0; i < m_CacheChunkCountMax; i++)
        {
            if(m_pCacheChunkList[i].chunkIndex >= chunkIndexBegin)
            {
                InvalidateCacheChunk(&m_pCacheChunkList[i]);
            }
        }
    }

    nn::Result CachedMovieStream::ReadImpl(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
        NN_SDK_ASSERT_GREATER_EQUAL(offset, 0);

        if(size == 0)
        {
            *pOutReadSize = 0;
            NN_RESULT_SUCCESS;
        }
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        char* pDst = reinterpret_cast<char*>(buffer);
        int64_t srcOffset = offset;
        size_t remain = size;
        while(remain > 0)
        {
            int64_t chunkIndex = srcOffset / m_CacheChunkSize;
            CacheChunkHead* pChunk = nullptr;
            NN_RESULT_DO(RequestCacheChunk(&pChunk, chunkIndex));
            NN_SDK_ASSERT_NOT_NULL(pChunk);

            int64_t localOffset = srcOffset % m_CacheChunkSize;
            int64_t localSize = std::min(static_cast<int64_t>(remain), m_CacheChunkSize - localOffset);
            std::memcpy(pDst, pChunk->data + static_cast<ptrdiff_t>(localOffset), static_cast<size_t>(localSize));
            srcOffset += localSize;
            pDst      += static_cast<ptrdiff_t>(localSize);
            remain    -= static_cast<size_t>(localSize);
            m_Strategy.notifyRead(chunkIndex, localOffset, localSize, m_Strategy.userData);
        }

        *pOutReadSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result CachedMovieStream::WriteImpl(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);

        if(size == 0)
        {
            // size == 0 の場合は何もしない（nn::fs と同じ仕様）
            NN_RESULT_SUCCESS;
        }
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        const char* pSrc = reinterpret_cast<const char*>(buffer);
        int64_t dstOffset = offset;
        size_t remain = size;
        while(remain > 0)
        {
            int64_t chunkIndex = dstOffset / m_CacheChunkSize;
            CacheChunkHead* pChunk = nullptr;
            NN_RESULT_DO(RequestCacheChunk(&pChunk, chunkIndex));
            NN_SDK_ASSERT_NOT_NULL(pChunk);

            int64_t localOffset = dstOffset % m_CacheChunkSize;
            int64_t localSize = std::min(static_cast<int64_t>(remain), m_CacheChunkSize - localOffset);
            std::memcpy(pChunk->data + static_cast<ptrdiff_t>(localOffset), pSrc, static_cast<size_t>(localSize));
            pChunk->flag |= CacheChunkFlag_Dirty;
            pSrc      += static_cast<ptrdiff_t>(localSize);
            dstOffset += localSize;
            remain    -= static_cast<size_t>(localSize);
            m_Strategy.notifyWritten(chunkIndex, static_cast<int64_t>(localOffset), static_cast<int64_t>(localSize), m_Strategy.userData);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result CachedMovieStream::ResizeImpl(int64_t size) NN_NOEXCEPT
    {
        int64_t newChunkCount = ((size + m_CacheChunkSize - 1) / m_CacheChunkSize);
        int64_t availableChunkCount = CalculateAvailableChunkCount();
        //NN_SDK_LOG("resize %lld/ file=%lld, available=%lld, new=%lld\n", size, m_FileChunkCount, availableChunkCount, newChunkCount);

        // 拡大
        if(newChunkCount > availableChunkCount)
        {
            // チャンク数を予約
            m_ExplicitChunkCount = newChunkCount;
            NN_RESULT_SUCCESS;
        }

        // 縮小
        if(newChunkCount < availableChunkCount)
        {
            // 不要になったキャッシュを破棄
            InvalidateCacheChunkGreaterEqualIndex(newChunkCount);
            availableChunkCount = newChunkCount;
        }

        if(newChunkCount < m_FileChunkCount)
        {
            // 不要になったファイルデータを破棄
            NN_RESULT_DO(m_Accessor.shrink(newChunkCount * m_CacheChunkSize, m_Accessor.userData));
            m_FileChunkCount = newChunkCount;
        }
        NN_SDK_ASSERT_EQUAL(availableChunkCount, newChunkCount);
        NN_SDK_ASSERT_LESS_EQUAL(m_FileChunkCount, newChunkCount);

        // 半端分をゼロ埋めする
        int64_t tailChunkIndex = size / m_CacheChunkSize;
        int64_t tailOffset     = size % m_CacheChunkSize;
        NN_SDK_ASSERT(
            (tailChunkIndex == availableChunkCount && tailOffset == 0) ||
            (tailChunkIndex == availableChunkCount - 1 && tailOffset > 0)
        );
        if(tailChunkIndex < availableChunkCount)
        {
            int64_t tailSize   = m_CacheChunkSize - tailOffset;
            CacheChunkHead* pTailChunk = nullptr;
            NN_RESULT_DO(RequestCacheChunk(&pTailChunk, tailChunkIndex));
            std::memset(pTailChunk->data + static_cast<ptrdiff_t>(tailOffset), 0, static_cast<size_t>(tailSize));
            pTailChunk->flag |= CacheChunkFlag_Dirty;
            //m_Strategy.notifyWritten(tailChunkIndex, tailOffset, tailSize, m_Strategy.userData);
        }

        m_ExplicitChunkCount = newChunkCount;
        NN_RESULT_SUCCESS;
    }

    nn::Result CachedMovieStream::FlushImpl() NN_NOEXCEPT
    {
        for(int64_t i = 0; i < m_CacheChunkCountMax; i++)
        {
            auto& e = m_pCacheChunkList[i];
            if(e.chunkIndex != InvalidChunkIndex)
            {
                NN_RESULT_DO(FlushCacheChunk(&e));
            }
        }

        WriteOutZeroChunkRange(m_FileChunkCount, m_ExplicitChunkCount, &m_pCacheChunkList[0]);
        NN_SDK_ASSERT_GREATER_EQUAL(m_FileChunkCount, m_ExplicitChunkCount);
        NN_SDK_ASSERT_EQUAL(m_CacheChunkCount, 0);
        NN_RESULT_SUCCESS;
    }

}}}
